<?php

declare(strict_types=1);

namespace Drupal\file_visibility\Hook;

use Drupal\Component\Utility\DeprecationHelper;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\file\FileInterface;
use Drupal\file_visibility\FileOriginalUriFieldItemList;
use Drupal\file_visibility\FileVisibilityInterface;

/**
 * Entity operations hooks.
 */
class EntityOperationHooks {

  public function __construct(
    protected readonly FileVisibilityInterface $fileVisibility,
    protected readonly Connection $db,
  ) {}

  use StringTranslationTrait;

  /**
   * Updates field visibility on entity CRUD.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity instance.
   */
  #[Hook('entity_insert')]
  #[Hook('entity_update')]
  #[Hook('entity_delete')]
  #[Hook('entity_translation_delete')]
  #[Hook('entity_revision_delete')]
  public function updateFileVisibility(EntityInterface $entity): void {
    $this->fileVisibility->updateFilesVisibility($entity);
  }

  /**
   * Saves the file entity original URI.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   */
  #[Hook('file_insert')]
  #[Hook('file_update')]
  public function storeOriginalFileUri(FileInterface $file): void {
    if ($file->isTemporary()) {
      return;
    }

    $original = DeprecationHelper::backwardsCompatibleCall(
      currentVersion: \Drupal::VERSION,
      deprecatedVersion: '11.2',
      currentCallable: fn() => $file->getOriginal(),
      deprecatedCallable: fn() => $file->original,
    );

    // File created as permanent or set from temporary to permanent.
    if ($original === NULL || $original->isTemporary()) {
      $this->db->merge(FileVisibilityInterface::TABLE)
        ->keys(['fid' => $file->id()])
        ->fields(['uri' => $file->getFileUri()])
        ->execute();
    }
  }

  /**
   * Cleans up the file entity original URI entry.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   */
  #[Hook('file_delete')]
  public function deleteOriginalFileUri(FileInterface $file): void {
    $this->db->delete(FileVisibilityInterface::TABLE)
      ->condition('fid', $file->id())
      ->execute();
  }

  /**
   * Defines the file original URI computed field.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entityType
   *   The entity type.
   */
  #[Hook('entity_base_field_info')]
  public function defineFileOriginalUriComputedField(EntityTypeInterface $entityType): array {
    $fields = [];

    if ($entityType->id() === 'file') {
      $fields['originalUri'] = BaseFieldDefinition::create('uri')
        ->setName('originalUri')
        ->setTargetEntityTypeId('file')
        ->setLabel($this->t('Original URI'))
        ->setDescription(t("The file's original URI."))
        ->setComputed(TRUE)
        ->setClass(FileOriginalUriFieldItemList::class);
    }

    return $fields;
  }

}
