<?php

declare(strict_types=1);

namespace Drupal\file_visibility_track_usage\DependencyInjection;

use Drupal\Component\Utility\DeprecationHelper;
use Drupal\file_visibility\FileVisibility;
use Drupal\track_usage\Recorder;
use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Compiler pass to change event subscriber priorities.
 *
 * The logic of the TrackUsage file visibility plugin requires that the
 * \Drupal\file_visibility\FileVisibility::doUpdateFileVisibility() event
 * subscriber to act after \Drupal\track_usage\Recorder::registerUsageRecords().
 */
class EventSubscriberOrder implements CompilerPassInterface {

  /**
   * {@inheritdoc}
   */
  public function process(ContainerBuilder $container): void {
    DeprecationHelper::backwardsCompatibleCall(
      currentVersion: \Drupal::VERSION,
      deprecatedVersion: '11.0',
      currentCallable: $this->processDrupal11($container),
      deprecatedCallable: $this->processDrupal10($container),
    );
  }

  /**
   * Process: Drupal >=11.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerBuilder $container
   *   The Symfony container.
   *
   * @return callable
   *   A process callable.
   */
  private function processDrupal11(ContainerBuilder $container): callable {
    return function () use ($container): void {
      $recorderPriority = $fileVisibilityDelta = NULL;

      $definition = $container->getDefinition('event_dispatcher');
      $calls = $definition->getMethodCalls();

      foreach ($calls as $delta => $call) {
        $serviceId = $call[1][1][0]->getValues()[0]->__toString();
        $method = $call[1][1][1];
        if ($serviceId === Recorder::class && $method === 'registerUsageRecords') {
          // Record the Recorder service priority.
          $recorderPriority = $call[1][2];
        }
        if ($serviceId === FileVisibility::class && $method === 'doUpdateFileVisibility') {
          // Record the FileVisibility delta.
          $fileVisibilityDelta = $delta;
        }
      }

      if ($fileVisibilityDelta === NULL || $recorderPriority === NULL) {
        return;
      }

      $calls[$fileVisibilityDelta][1][2] = $recorderPriority - 10;
      $definition->setMethodCalls($calls);
    };
  }

  /**
   * Process: Drupal 10.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerBuilder $container
   *   The Symfony container.
   *
   * @return callable
   *   A process callable.
   */
  private function processDrupal10(ContainerBuilder $container): callable {
    return function () use ($container): void {
      $definition = $container->getDefinition('event_dispatcher');
      $argument = $definition->getArgument(1);
      $subscribersByPriority =& $argument[KernelEvents::RESPONSE];
      $recorderPriority = NULL;

      foreach ($subscribersByPriority as $priority => &$subscribers) {
        foreach ($subscribers as $delta => $subscriber) {
          if ($subscriber['service'] === [Recorder::class, 'registerUsageRecords']) {
            // Record the Recorder service priority.
            $recorderPriority = $priority;
          }

          // Unset the FileVisibility service from the current position.
          if ($subscriber['service'] === [FileVisibility::class, 'doUpdateFileVisibility']) {
            unset($subscribersByPriority[$priority][$delta]);
            $subscribersByPriority[$priority] = array_values($subscribersByPriority[$priority]);
          }
        }
      }

      if ($recorderPriority === NULL) {
        return;
      }

      // Add back FileVisibility service with a lower priority.
      $fileVisibilityPriority = $recorderPriority - 10;
      $subscribersByPriority[$fileVisibilityPriority] ??= [];
      $subscribersByPriority[$fileVisibilityPriority][] = [
        'service' => [FileVisibility::class, 'doUpdateFileVisibility'],
      ];

      $definition->setArgument(1, $argument);
    };
  }

}
