<?php

declare(strict_types=1);

namespace Drupal\file_visibility_track_usage\Form;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\track_usage\Entity\TrackConfigInterface;
use Drupal\file_visibility\FileVisibilityPluginManager;

/**
 * Module's settings form.
 */
class FileVisibilityTrackUsageSettingsForm extends ConfigFormBase {

  use AutowireTrait;

  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FileVisibilityPluginManager $manager,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    if (!$options = $this->getOptions()) {
      return $this->noOptions();
    }

    $form['track_usage_config'] = [
      '#type' => 'select',
      '#title' => $this->t('Track usage configuration'),
      '#description' => $this->t("If no configuration is selected, the 'Track Usages' plugin is disabled."),
      '#options' => $options,
      '#config_target' => new ConfigTarget(
        configName: 'file_visibility_track_usage.settings',
        propertyPath: 'track_usage_config',
        toConfig: fn(string $value) => $value ?: NULL,
      ),
      '#empty_option' => $this->t('- No configuration -'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Provides an access check for file_visibility_track_usage.settings route.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(): AccessResultInterface {
    $config = $this->config('file_visibility.settings');
    $enabledPlugins = $config->get('enabled_plugins');
    $allowed = empty($enabledPlugins) || in_array('track_usage', $enabledPlugins, TRUE);
    return AccessResult::allowedIf($allowed)->addCacheableDependency($config);
  }

  /**
   * Returns the available configurations.
   *
   * @return array<array-key, string>
   *   Form API select element options.
   */
  protected function getOptions(): array {
    $storage = $this->entityTypeManager->getStorage('track_usage_config');
    $ids = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('status', TRUE)
      ->condition('realTimeRecording', TRUE)
      ->execute();

    if (empty($ids)) {
      return [];
    }

    $options = [];
    foreach ($storage->loadMultiple($ids) as $config) {
      assert($config instanceof TrackConfigInterface);
      if (!array_key_exists('file', $config->get('target'))) {
        continue;
      }
      if (!$config->useRealTimeRecording()) {
        continue;
      }
      $options[$config->id()] = $config->label();
    }

    asort($options);

    return $options;
  }

  /**
   * Returns error message as a render array.
   *
   * @return array
   *   Render array.
   */
  protected function noOptions(): array {
    return [
      [
        '#markup' => $this->t("There are no track usage configurations that are qualifying to be used by File Visibility Track Usage plugin. They should met the following conditions:"),
      ],
      [
        '#theme' => 'item_list',
        '#items' => [
          $this->t("The 'file' entity type should be in the target list."),
          $this->t("The the config 'Record entity changes in real time' option should be checked."),
          $this->t('The the configuration should be enabled.'),
        ],
      ],
      [
        '#markup' => $this->t('You can manage configurations at <a href=":url">Track configurations</a>.', [
          ':url' => Url::fromRoute('entity.track_usage_config.collection')->toString(),
        ]),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['file_visibility_track_usage.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'file_visibility_track_usage_settings';
  }

}
