<?php

declare(strict_types=1);

namespace Drupal\Tests\file_visibility_track_usage\Functional;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\StringTranslation\PluralTranslatableMarkup;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\track_usage\Traits\TestingDataTrait;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use Drupal\media\Entity\Media;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\track_usage\Entity\TrackConfig;
use Drupal\user\UserInterface;

/**
 * @covers \Drupal\file_visibility_track_usage\Plugin\FileVisibility\TrackUsage
 * @group file_visibility
 */
class FileVisibilityTaskUsageTest extends BrowserTestBase {

  use TestingDataTrait {
    createDataStructure as protected traitCreateDataStructure;
    createData as protected traitCreateData;
  }

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'image',
    'language',
    'media',
    'node',
    'paragraphs',
    'taxonomy',
    'file_visibility_track_usage',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with elevated permissions.
   */
  protected UserInterface $privileged;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->createConfig();
    $this->createDataStructure();
    $this->createFiles();
    $this->createData();

    $this->privileged = $this->createUser([
      'bypass node access',
      'view unpublished paragraphs',
      'administer media',
    ]);
  }

  /**
   * Tests the 'track_usage' plugin.
   */
  public function testFileVisibilityTrackUsage(): void {
    // Only files used by node 2 are visible as node 1 is unpublished.
    $this->assertPubliclyVisible(1, 3, 4, 5, 6);
    $this->assertPrivilegedVisible(1, 3, 4, 5, 6);

    // Publish node 1.
    $node1 = Node::load(1);
    $node1->setPublished()->save();
    // File 2 was added as is used by node 1.
    $this->assertPubliclyVisible(1, 2, 3, 4, 5, 6);
    $this->assertPrivilegedVisible(1, 2, 3, 4, 5, 6);

    // Unpublish node 1.
    $node1->setUnPublished()->save();
    // File 2 in only used by node 1, which is unpublished.
    $this->assertPubliclyVisible(1, 3, 4, 5, 6);
    $this->assertPrivilegedVisible(1, 3, 4, 5, 6);

    // Add a new file 8 to node 1.
    $node1->get('file')->appendItem($this->createFile(9)->id());
    $node1->save();
    // File 8 is used only by node 1, but the node is unpublished.
    $this->assertPubliclyVisible(1, 3, 4, 5, 6);
    $this->assertPrivilegedVisible(1, 3, 4, 5, 6);

    // Add file 8 to paragraph 5 (under node 2).
    Paragraph::load(5)->set('attachment', 9)->save();
    // File 8 is now visible via node 2, even node 1 is unpublished.
    $this->assertPubliclyVisible(1, 3, 4, 5, 6, 9);
    $this->assertPrivilegedVisible(1, 3, 4, 5, 6, 9);

    // Unpublish paragraph 5 (under node 2).
    Paragraph::load(5)->setUnpublished()->save();
    // File 4 is used by node 1, but that is unpublished. It is also used by
    // node 2 via paragraph 5. As paragraph 5 is unpublished, file 4 is not
    // visible. Same with file 8.
    $this->assertPubliclyVisible(1, 3, 5, 6);
    $this->assertPrivilegedVisible(1, 3, 5, 6);
  }

  /**
   * Asserts that a list of files is visible and the rest are not.
   *
   * @param int|string ...$fids
   *   File entity IDs.
   */
  protected function assertPubliclyVisible(...$fids): void {
    $result = ['access' => [], 'download' => []];

    foreach (File::loadMultiple($fids) as $fid => $file) {
      $accessible = in_array($fid, $fids);
      if ($file->access('download') !== $accessible) {
        $result['access'][$accessible][] = $fid;
      }

      if ($this->getStatusCode($file) !== ($accessible ? 200 : 403)) {
        $result['download'][$accessible][] = $fid;
      }
    }

    $message = [];
    foreach ($result['access'] as $accessible => $fids) {
      $message[] = new PluralTranslatableMarkup(
        count($fids),
        'File entity @fids is @actual but it @expected',
        'Files entities @fids are @actual but they @expected',
        [
          '@fids' => implode(', ', $fids),
          '@actual' => $accessible ? t('not accessible') : t('accessible'),
          '@expected' => $accessible ? t('should be') : t("shouldn't be"),
        ],
      );
    }

    foreach ($result['download'] as $accessible => $fids) {
      $message[] = new PluralTranslatableMarkup(
        count($fids),
        'File @fids is @actual but it @expected',
        'Files @fids are @actual but they @expected',
        [
          '@fids' => implode(', ', $fids),
          '@actual' => $accessible ? t('not downloadable') : t('downloadable'),
          '@expected' => $accessible ? t('should be') : t("shouldn't be"),
        ],
      );
    }

    if ($message) {
      $this->fail(implode("\n", $message));
    }
  }

  /**
   * Asserts that all files are visible to a privileged user.
   *
   * @param int|string ...$fids
   *   File entity IDs.
   */
  protected function assertPrivilegedVisible(...$fids): void {
    \Drupal::entityTypeManager()->getAccessControlHandler('file')->resetCache();
    $this->drupalLogin($this->privileged);
    foreach (File::loadMultiple($fids) as $fid => $file) {
      $this->assertTrue(!$file->access('download', $this->privileged, TRUE)->isForbidden(), "Cannot access file entity $fid");
      $this->assertSame(200, $this->getStatusCode($file), "Cannot download file $fid");

    }
    $this->drupalLogout();
  }

  /**
   * Creates the testing Track Usage configuration.
   */
  protected function createConfig(): void {
    TrackConfig::create([
      'id' => 'file_visibility',
      'label' => 'Unpublished file',
      'activeRevision' => TRUE,
      'realTimeRecording' => TRUE,
      'source' => ['node' => [], 'taxonomy_term' => [], 'user' => []],
      'traversable' => ['media' => [], 'paragraph' => []],
      'target' => ['file' => []],
      'status' => TRUE,
    ])->save();
    $this->config('file_visibility_track_usage.settings')->set('track_usage_config', 'file_visibility')->save();
  }

  /**
   * {@inheritdoc}
   */
  protected function createDataStructure(): void {
    $this->traitCreateDataStructure();

    NodeType::create(['type' => 'article'])->save();
    FieldConfig::create([
      'entity_type' => 'node',
      'bundle' => 'article',
      'field_name' => 'body',
      'label' => 'Article body',
    ])->save();
    FieldConfig::create([
      'entity_type' => 'node',
      'bundle' => 'article',
      'field_name' => 'file',
      'label' => 'Article file',
      'settings' => ['file_extensions' => 'txt html png'],
    ])->save();

    $this->createEntityReferenceField('node', 'article', 'gallery', 'Article media file', 'media');

    $this->addFieldtoParagraphType('top_paragraph', 'body', 'text_with_summary');
    $this->addFieldtoParagraphType('top_paragraph', 'attachment', 'file');
    FieldStorageConfig::create([
      'field_name' => 'paragraph',
      'entity_type' => 'node',
      'type' => 'entity_reference_revisions',
      'cardinality' => '-1',
      'settings' => ['target_type' => 'paragraph'],
    ])->save();
    FieldConfig::create([
      'entity_type' => 'node',
      'bundle' => 'article',
      'field_name' => 'paragraph',
      'settings' => [
        'handler' => 'default:paragraph',
        'handler_settings' => [
          'target_bundles' => [
            'top_paragraph' => 'top_paragraph',
          ],
        ],
      ],
    ])->save();

    Vocabulary::create(['vid' => 'color', 'name' => 'Color'])->save();
    $this->createEntityReferenceField('taxonomy_term', 'color', 'sample', 'Sample', 'media');

    $this->createEntityReferenceField('user', 'user', 'avatar', 'Avatar', 'file');
  }

  /**
   * {@inheritdoc}
   */
  protected function createData(): void {
    $this->traitCreateData();

    Node::create([
      'nid' => 2,
      'type' => 'article',
      'title' => $this->randomString(),
      'body' => $this->buildBody(files: [3, 5]),
      'file' => $this->file[6],
      'gallery' => Media::create([
        'mid' => 3,
        'bundle' => 'file',
        'field_media_file' => $this->file[1],
      ]),
      'paragraph' => Paragraph::create([
        'id' => 5,
        'type' => 'top_paragraph',
        'body' => $this->buildBody(images: [4, 6]),
      ]),
    ])->save();
  }

  /**
   * Returns the response of a GET HTTP request to a file.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   *
   * @return int
   *   The HTTP response status code.
   */
  protected function getStatusCode(FileInterface $file): int {
    $url = $file->createFileUrl(FALSE);
    $mimeType = \Drupal::service('file.mime_type.guesser')->guessMimeType($file->getFileUri()) ?? 'text/plain';
    return $this->getHttpClient()->get($url, [
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
      'allow_redirects' => FALSE,
      'verify' => FALSE,
      'headers' => ['Accept' => $mimeType],
    ])->getStatusCode();
  }

  /**
   * Creates a testing file entity.
   *
   * @param int $fid
   *   File ID.
   *
   * @return \Drupal\file\FileInterface
   *   The testing file entity.
   */
  protected function createFile(int $fid): FileInterface {
    $fileSystem = \Drupal::service('file_system');

    // Create a file in the public directory.
    $dir = 'public://' . strtolower($this->randomMachineName()) . '/' . strtolower($this->randomMachineName());
    $fileSystem->prepareDirectory($dir, FileSystemInterface::CREATE_DIRECTORY);
    $fileName = strtolower($this->randomMachineName()) . '.txt';
    $uri = "$dir/$fileName";
    $fileSystem->saveData($this->randomString(), $uri);

    // Create the file entity.
    $file = File::create([
      'fid' => $fid,
      'uri' => $uri,
      'filename' => $fileName,
    ]);
    $file->save();

    return $file;
  }

}
