<?php

declare(strict_types=1);

namespace Drupal\file_visibility;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\file\FileInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Interface for file visibility service.
 */
interface FileVisibilityInterface extends EventSubscriberInterface {

  /**
   * Location of files in the private filesystem.
   */
  public const PRIVATE_LOCATION = 'private://file-visibility/';

  /**
   * Table to store original file URIs.
   */
  public const TABLE = 'file_visibility';

  /**
   * Moves the files from 'public' to 'private' file system or vice versa.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The source entity.
   */
  public function updateFilesVisibility(EntityInterface $entity): void;

  /**
   * Returns a list of files used by an entity.
   *
   * The files are collected by iterating over all plugins.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The source or traversable entity.
   *
   * @return list<int|string>
   *   A list of file IDs.
   */
  public function getEntityUsedFiles(FieldableEntityInterface $entity): array;

  /**
   * Checks if a file is accessible by a user.
   *
   * More than one entity might use the same file following various paths. E.g.,
   * a file might be attached to a media which is attached to a paragraph, and,
   * finally, the paragraph is attached to a node. But the same node might use
   * the same file in different ways, like linking it directly by using a file
   * field.
   *
   * This method checks whether at least one path to the file consists of
   * user-visible entities. All entities along the path are checked, and all
   * should be visible to the user.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account being checked.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function getFileAccess(FileInterface $file, AccountInterface $account): AccessResultInterface;

}
