<?php

declare(strict_types=1);

namespace Drupal\file_visibility;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\file\FileInterface;

/**
 * Interface for FileVisibility plugins.
 *
 * Such plugins are able to return the source entities of a file entity using a
 * specific method.
 *
 * It's the plugin business to determine the way to reach the source entity of a
 * given file or get all the file entities of a given source entity. For
 * instance, a certain plugin might use the Usage Track data which computes and
 * stores relations between entities.
 */
interface FileVisibilityPluginInterface extends PluginInspectionInterface {

  /**
   * Checks whether this plugin met all conditions to be used.
   *
   * @return bool
   *   Whether this plugin is ready to be used.
   */
  public function applies(): bool;

  /**
   * Returns a list of file entities referred by an entity.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity. Could be a source entity (e.g., a node) or a traversable
   *   entity (e.g., a paragraph or media).
   *
   * @return iterable<int|string>
   *   A list of file IDs.
   */
  public function getEntityFiles(FieldableEntityInterface $entity): iterable;

  /**
   * Returns a list of paths to the file.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   *
   * @return iterable<list<\Drupal\Core\Entity\FieldableEntityInterface>>
   *   A list of file paths.
   */
  public function getPathsToFile(FileInterface $file): iterable;

}
