<?php

declare(strict_types=1);

namespace Drupal\file_visibility\Hook;

use Drupal\Component\Utility\DeprecationHelper;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountInterface;
use Drupal\file\FileInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\file_visibility\FileVisibilityInterface;

/**
 * File access hooks.
 */
class FileAccessHooks {

  public function __construct(
    protected readonly FileVisibilityInterface $fileVisibility,
    protected readonly FileRepositoryInterface $fileRepository,
    protected readonly AccountInterface $currentUser,
  ) {}

  /**
   * Implements hook_ENTITY_TYPE_access() for 'file' entity type.
   */
  #[Hook('file_access')]
  public function fileAccess(FileInterface $file, string $operation, AccountInterface $account): AccessResultInterface {
    if ($operation === 'download' && str_starts_with($file->getFileUri(), FileVisibilityInterface::PRIVATE_LOCATION)) {
      return $this->fileVisibility->getFileAccess($file, $account);
    }
    // No opinion.
    return AccessResult::neutral();
  }

  /**
   * Implements hook_file_download().
   */
  #[Hook('file_download')]
  public function fileDownload(string $uri): array|int|null {
    if (!str_starts_with($uri, FileVisibilityInterface::PRIVATE_LOCATION)) {
      // No opinion.
      return NULL;
    }

    if (!$file = $this->fileRepository->loadByUri($uri)) {
      // No opinion.
      return NULL;
    }

    $access = $this->fileVisibility->getFileAccess($file, $this->currentUser);
    if ($access->isNeutral()) {
      return NULL;
    }

    if ($access->isForbidden()) {
      // No entity referring to this file is accessible for the current user.
      return -1;
    }

    // @see https://www.drupal.org/node/3494172
    // @todo Remove BC layer in Drupal 12.
    return DeprecationHelper::backwardsCompatibleCall(
      currentVersion: \Drupal::VERSION,
      deprecatedVersion: '11.2.0',
      currentCallable: fn() => $file->getDownloadHeaders(),
      deprecatedCallable: fn() => file_get_content_headers($file),
    );
  }

}
