<?php

namespace Drupal\filehash\Drush\Commands;

use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Drupal\Core\Database\Connection;
use Drupal\filehash\Batch\CleanBatch;
use Drupal\filehash\Batch\GenerateBatch;
use Drupal\filehash\FileHashInterface;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for File Hash module.
 */
class FileHashCommands extends DrushCommands {

  use AutowireTrait;

  public function __construct(
    protected Connection $database,
    protected FileHashInterface $fileHash,
  ) {
  }

  /**
   * Generate hashes for existing files.
   */
  #[CLI\Command(name: 'filehash:generate', aliases: [
    'fgen', 'filehash-generate',
  ])]
  #[CLI\Usage(name: 'drush filehash:generate', description: 'Generate hashes for existing files.')]
  public function generate(): void {
    batch_set(GenerateBatch::createBatch());
    $batch =& batch_get();
    $batch['progressive'] = FALSE;
    drush_backend_batch_process();
  }

  /**
   * Remove database columns for disabled hash algorithms.
   */
  #[CLI\Command(name: 'filehash:clean', aliases: ['filehash-clean'])]
  #[CLI\Usage(name: 'drush filehash:clean')]
  public function clean(): void {
    batch_set(CleanBatch::createBatch());
    $batch =& batch_get();
    $batch['progressive'] = FALSE;
    drush_backend_batch_process();
  }

  /**
   * Print a list of duplicate files, if any.
   *
   * @param array{limit: string|int, algorithm: ?string, format: string} $options
   *   The command options.
   */
  #[CLI\Command(name: 'filehash:report')]
  #[CLI\Usage(name: 'drush filehash:report', description: 'Print a list of up to 1000 duplicate files.')]
  #[CLI\Option(name: 'limit', description: 'Maximum number of rows to return.')]
  #[CLI\Option(name: 'algorithm', description: 'Hash algorithm to use, e.g. "sha512_256". If not specified, it will be chosen automatically.')]
  public function report(array $options = ['limit' => 1000, 'algorithm' => NULL, 'format' => 'table']): RowsOfFields {
    if (!$algorithms = $this->fileHash->getEnabledAlgorithms()) {
      throw new \RuntimeException(dt('No hash algorithms are enabled.'));
    }
    if (NULL === $options['algorithm']) {
      $options['algorithm'] = end($algorithms);
    }
    elseif (!isset($algorithms[$options['algorithm']])) {
      throw new \InvalidArgumentException('Invalid algorithm.');
    }
    if (!is_numeric($options['limit']) || $options['limit'] < 0) {
      throw new \InvalidArgumentException('Invalid limit.');
    }
    $subquery = $this->database->select('file_managed', 'fm1')
      ->where("fm1.{$options['algorithm']} = fm2.{$options['algorithm']}")
      ->range(1, 1);
    $subquery->addExpression('1');
    $result = $this->database->select('file_managed', 'fm2')
      ->fields('fm2', [$options['algorithm'], 'fid', 'uri'])
      ->exists($subquery)
      ->orderBy($options['algorithm'])
      ->orderBy('fid')
      ->range(0, (int) $options['limit'])
      ->execute();
    if (NULL === $result) {
      throw new \UnexpectedValueException('Query failed.');
    }
    $rows = $result->fetchAll(\PDO::FETCH_ASSOC);
    $logger = $this->logger();
    if (!$rows && $logger) {
      $logger->success(dt('No duplicate files found.'));
    }
    return new RowsOfFields($rows);
  }

}
