<?php

namespace Drupal\filemime\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\MimeType\MimeTypeMapInterface;
use Drupal\Core\File\Event\MimeTypeMapLoadedEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Modifies the MIME type map by calling hook_file_mimetype_mapping_alter().
 */
class MimeTypeMapLoadedSubscriber implements EventSubscriberInterface {

  /**
   * Constructs the event subscriber.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * Handle the event by altering the MIME type map.
   */
  public function onMimeTypeMapLoaded(MimeTypeMapLoadedEvent $event): void {
    $config = $this->configFactory->get('filemime.settings');
    $file = $config->get('file');
    $types = $config->get('types');
    if (\is_string($file) && is_readable($file)) {
      $this->processMimeTypes(file($file) ?: [], $event->map);
    }
    if (\is_string($types)) {
      $this->processMimeTypes(explode("\n", $types), $event->map);
    }
  }

  /**
   * Adds an array of MIME type mapping strings to the map.
   *
   * @param string[] $mimetypes
   *   The array of MIME type mapping strings.
   * @param \Drupal\Core\File\MimeType\MimeTypeMapInterface $map
   *   The MIME map object.
   */
  public function processMimeTypes(array $mimetypes, MimeTypeMapInterface $map): void {
    foreach ($mimetypes as $mimetype) {
      $tokens = preg_split('/[\s]+/', $mimetype, -1, PREG_SPLIT_NO_EMPTY);
      if (!$tokens) {
        continue;
      }
      foreach ($tokens as $index => $token) {
        // If this token starts with #, consider it a comment and break to the
        // next MIME type.
        if (substr($token, 0, 1) === '#') {
          break;
        }
        // If this is not the first token, it must be an extension. Add it to
        // the extensions array.
        if ($index) {
          $map->addMapping($tokens[0], $token);
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = [];
    $events[MimeTypeMapLoadedEvent::class][] = ['onMimeTypeMapLoaded'];
    return $events;
  }

}
