# FilePond Module

A FilePond-based file upload element for Drupal forms with chunked uploads,
drag-and-drop, image previews, and reordering support.

This module is similar in functionality to
[DropzoneJS](https://www.drupal.org/project/dropzonejs), but uses the
[FilePond](https://pqina.nl/filepond/) JavaScript library instead. If you're
familiar with DropzoneJS, you'll find a similar architecture here: a form
element, Entity Browser widget, and Media Library integration.

## Table of Contents

- [Installation](#installation)
- [Configuration](#configuration)
  - [Global Settings](#global-settings)
  - [Element Properties](#element-properties)
  - [Reserved Options](#reserved-options-cannot-override-in-config)
  - [Image Preview Modes](#image-preview-modes)
- [Usage](#usage)
  - [Field Widget](#field-widget)
  - [Entity Browser Widget](#entity-browser-widget)
  - [Media Library Integration](#media-library-integration)
  - [Form Element](#form-element)
- [Event Subscriber](#event-subscriber)
- [Permissions](#permissions)
- [Altering Elements](#altering-elements)
- [Submit Button Control](#submit-button-control)
- [Limitations](#limitations)
- [Resources](#resources)

## Installation

Install FilePond libraries via Composer using Asset Packagist:

```bash
composer require npm-asset/filepond \
  npm-asset/filepond-plugin-file-validate-type \
  npm-asset/filepond-plugin-file-validate-size \
  npm-asset/filepond-plugin-image-preview \
  npm-asset/filepond-plugin-file-poster \
  npm-asset/filepond-plugin-image-crop
```

**Requirements:**
- Asset Packagist repository in composer.json
- `oomphinc/composer-installers-extender` package

## Configuration

### Global Settings

Configure defaults at **Administration > Configuration > Media > FilePond**
(`/admin/config/media/filepond`).

### Element Properties

#### Server-Side Properties (Root Level)

These control validation and file storage:

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `#extensions` | string | `'jpg jpeg gif png'` | Space-separated allowed extensions |
| `#max_filesize` | string | PHP upload_max_filesize | Max file size (e.g., `'20M'`) |
| `#max_files` | int | From module config | Maximum files. `0` = unlimited |
| `#upload_location` | string | `'public://filepond-uploads'` | Destination URI (supports tokens) |
| `#default_value` | array | `[]` | Existing file entity IDs |

#### UI Options (#config)

Pass FilePond options via `#config`:

```php
$form['images'] = [
  '#type' => 'filepond',
  '#config' => [
    'labelIdle' => 'Drop files here or <span class="filepond--label-action">Browse</span>',
    'allowReorder' => TRUE,
    'maxParallelUploads' => 4,
  ],
];
```

**Common options:**

| Option | Type | Default | Description |
|--------|------|---------|-------------|
| `labelIdle` | string | `'Drag & drop...'` | Drop zone prompt text |
| `allowReorder` | bool | `FALSE` | Enable drag-to-reorder |
| `maxParallelUploads` | int | `3` | Simultaneous uploads |
| `styleItemPanelAspectRatio` | float | `0.5625` | Panel aspect ratio (height/width) |
| `previewFitMode` | string | `'contain'` | `'contain'` or `'cover'` |
| `previewImageStyle` | string | `NULL` | Drupal image style for thumbnails |
| `labelProcessing` | string | `'Processing...'` | Text during server processing |
| `imagePreviewMaxHeight` | int | `256` | Max preview height in pixels |

See [FilePond docs](https://pqina.nl/filepond/docs/api/instance/properties/) for
all available options.

### Reserved Options (Cannot Override in #config)

The following options are controlled by root-level properties and **cannot** be
set in `#config`. If you try to set them, they will be silently removed:

| #config key | Use this instead | Why |
|-------------|------------------|-----|
| `extensions` | `#extensions` | Server-side validation source of truth |
| `maxFilesize` | `#max_filesize` | Server-side validation |
| `maxFiles` | `#max_files` | Server-side limit enforcement |
| `uploadLocation` | `#upload_location` | Server-side file destination |
| `acceptedFileTypes` | `#extensions` | Auto-derived from extensions |

**Why this restriction?**

These options control server-side validation. Allowing them in `#config` would
risk mismatches between what the server validates and what the UI shows. For
example, if `acceptedFileTypes` could be set separately, users might select
files the UI says are allowed but the server rejects.

```php
// WRONG - these #config values will be ignored:
$form['images']['#extensions'] = 'jpg png';
$form['images']['#config']['extensions'] = 'jpg png gif exe';  // Ignored!
$form['images']['#config']['acceptedFileTypes'] = ['image/gif'];  // Ignored!

// RIGHT - use root properties:
$form['images']['#extensions'] = 'jpg png gif';
```

### Image Preview Modes

The `previewFitMode` option controls how image previews display:

- **`contain`** (default): Contains the image within the panels bounds.
- **`cover`**: Crops image to fill panel (uses Image Crop plugin)

```php
'#config' => [
  'styleItemPanelAspectRatio' => 0.5625,  // 16:9 panels
  'previewFitMode' => 'cover',             // Crop to fill
  'imagePreviewMaxHeight' => 300,          // Constrain height
],
```

When using `cover` mode, `imagePreviewMaxHeight` helps the Image Crop plugin
work correctly by constraining the preview dimensions.

### Aspect Ratio Reference

Common values for `styleItemPanelAspectRatio` (height/width):

| Ratio | Value | Description |
|-------|-------|-------------|
| 16:9 | 0.5625 | Landscape (default) |
| 4:3 | 0.75 | Landscape |
| 1:1 | 1.0 | Square |
| 3:4 | 1.333 | Portrait |
| 9:16 | 1.778 | Portrait |

## Field Widget

The module includes a FilePond widget for image fields.

### Widget Settings

Configure in **Manage form display** for any image field:

- **Preview image style**: Drupal image style for existing image thumbnails
- **Allow reorder**: Enable drag-to-reorder
- **Panel aspect ratio**: Ratio string like `16:9`, `4:3`, or `1:1`
- **Preview fit mode**: `contain` (letterbox) or `cover` (crop)

### Field-Based Routes

The widget uses field-based upload routes that derive settings from the field
configuration:

- Upload location from field's file directory setting
- Extensions from field's allowed file types
- Max filesize from field settings

No additional configuration needed - it reads directly from your field setup.

## Entity Browser Widget

The filepond_eb_widget submodule provides a FilePond widget for Entity Browser:

```bash
drush en filepond_eb_widget
```

Configure in your Entity Browser under Widgets > Add widget > "FilePond Media Upload".

## Media Library Integration

Replace Drupal's default Media Library upload widget:

1. Go to `/admin/config/media/filepond`
2. Enable "FilePond widget in Media Library"
3. Save

The image upload field in the image library will the use a filepond widget.

## Field Element

```php
$form['images'] = [
  '#type' => 'filepond',
  '#title' => t('Upload Images'),
  '#extensions' => 'jpg jpeg png gif',
  '#max_filesize' => '20M',
  '#max_files' => 5,
  '#upload_location' => 'public://uploads',
];
```
On submit, the element returns file IDs:

```php
public function submitForm(array &$form, FormStateInterface $form_state) {
  $values = $form_state->getValue('images');
  $fids = $values['fids'];  // Array of file entity IDs

  $files = \Drupal\file\Entity\File::loadMultiple($fids);
  foreach ($files as $file) {
    // Create media, attach to nodes, etc.
  }
}
```

## Event Subscriber

Subscribe to `FilePondUploadCompleteEvent` to perform actions after file upload
(e.g., create media entities, extract EXIF data). The event fires **after** the
file entity is saved.

**Note:** At this point, the file is still **temporary** (`status = 0`). It
becomes permanent when the parent form is submitted and file usage is
recorded. The Media Library, File widget, and Entity Browser widget do this
automatically. If you are using the form element on a custom form you need
to set the files as permanent in your submit handler.

### Basic Example

```yaml
# my_module.services.yml
services:
  my_module.filepond_subscriber:
    class: Drupal\my_module\EventSubscriber\FilePondSubscriber
    tags:
      - { name: event_subscriber }
```

```php
// src/EventSubscriber/FilePondSubscriber.php
namespace Drupal\my_module\EventSubscriber;

use Drupal\filepond\Event\FilePondUploadCompleteEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class FilePondSubscriber implements EventSubscriberInterface {

  public static function getSubscribedEvents() {
    return [
      FilePondUploadCompleteEvent::EVENT_NAME => 'onUploadComplete',
    ];
  }

  public function onUploadComplete(FilePondUploadCompleteEvent $event) {
    $file = $event->getFile();
    $context = $event->getContext();

    // Do something with the uploaded file.
  }

}
```

### Event Methods

| Method | Returns | Description |
|--------|---------|-------------|
| `getFile()` | `FileInterface` | The saved file entity |
| `getOriginalFilename()` | `string` | Original filename from client |
| `getContext()` | `array` | Context identifying upload source |
| `getContextValue($key)` | `mixed` | Get specific context value |
| `setResult($key, $value)` | `$this` | Pass data back to controller |

### Context by Upload Source

**Field Widget:**
```php
['entity_type' => 'node', 'bundle' => 'article', 'field_name' => 'field_image']
```

**Entity Browser:**
```php
['entity_browser' => 'media_browser', 'widget_uuid' => '...', 'media_type' => 'image']
```

**Media Library / Custom Forms:**
```php
['element_name' => 'upload', 'element_parents' => [...], 'field_name' => '...']
```

### Identifying Upload Source

```php
public function onUploadComplete(FilePondUploadCompleteEvent $event) {
  $context = $event->getContext();

  if (!empty($context['entity_browser'])) {
    // From Entity Browser - it handles media creation.
    return;
  }

  if (!empty($context['field_name']) && $context['field_name'] === 'field_portfolio') {
    // From specific field widget.
    $this->processPortfolioImage($event->getFile());
  }
}
```

## Permissions

Users need the **FilePond: Upload files** permission to use the upload element.

## Altering Elements

### Per-Form

```php
function my_module_form_alter(&$form, $form_state, $form_id) {
  if (isset($form['images']['#type']) && $form['images']['#type'] === 'filepond') {
    $form['images']['#max_files'] = 10;
    $form['images']['#config']['allowReorder'] = TRUE;
  }
}
```

### Site-Wide

```php
function my_module_element_info_alter(array &$info) {
  if (isset($info['filepond'])) {
    $info['filepond']['#process'][] = '_my_module_process_filepond';
  }
}

function _my_module_process_filepond($element, $form_state, $form) {
  $element['#config']['labelIdle'] = 'Custom site-wide prompt';
  return $element;
}
```

**Note:** Use a `#process` callback for `#config` alterations. Setting `#config`
directly in `hook_element_info_alter()` may be overwritten by forms.

## Submit Button Control

FilePond automatically disables form submit buttons while files are uploading
or processing. This prevents form submission before file IDs are available.

### Custom Submit Validators

Modules can register additional validators to control when submit is disabled:

```javascript
// In your module's JS file (must load after filepond.base.js):
(function (Drupal) {
  // Register a validator - return TRUE to disable submit.
  Drupal.filepond.addSubmitValidator(function (pond, form) {
    // Example: disable if less than 3 files
    var files = pond.getFiles();
    return files.length < 3;
  });
})(Drupal);
```

Validators receive:
- `pond` - The FilePond instance
- `form` - The containing form element

If **any** validator returns `true`, submit buttons are disabled.

### Built-in Validators

| Validator | Source | Condition |
|-----------|--------|-----------|
| Processing | `filepond.base.js` | Any file is uploading/processing |
| No files (EB) | `filepond_eb_widget.js` | No files selected (Entity Browser only) |

### Library Dependencies

Your library must depend on `filepond/filepond.base` to ensure the validator
registry is available:

```yaml
# my_module.libraries.yml
my_module.filepond:
  js:
    js/my_module.filepond.js: { }
  dependencies:
    - filepond/filepond.base
```

## Limitations

FilePond requires JavaScript. When JS is disabled, users see a warning message.
This is a known limitation of FilePond's async upload model.

## Resources

- [FilePond Documentation](https://pqina.nl/filepond/docs/)
- [FilePond Options Reference](https://pqina.nl/filepond/docs/api/instance/properties/)
- [Image Preview Plugin](https://pqina.nl/filepond/docs/api/plugins/image-preview/)
- [Image Crop Plugin](https://pqina.nl/filepond/docs/api/plugins/image-crop/)
