/**
 * @file
 * FilePond Drupal integration helpers.
 *
 * Provides common functionality for FilePond integration with Drupal:
 * - CSRF token handling
 * - Server configuration builder
 * - Common initialization patterns
 */

/* global FilePond, FilePondPluginFileValidateType, FilePondPluginFileValidateSize, FilePondPluginImagePreview, FilePondPluginFilePoster, FilePondPluginImageCrop */

(function (Drupal, drupalSettings) {
  /**
   * FilePond Drupal integration namespace.
   *
   * @namespace
   */
  Drupal.filepond = Drupal.filepond || {};

  /**
   * Cached CSRF token.
   *
   * @type {string|null}
   */
  Drupal.filepond.csrfToken = null;

  /**
   * Fetches the CSRF token from Drupal.
   *
   * @return {Promise<string>}
   *   Promise resolving to the CSRF token.
   */
  Drupal.filepond.fetchCsrfToken = function () {
    // Return cached token if available.
    if (Drupal.filepond.csrfToken) {
      return Promise.resolve(Drupal.filepond.csrfToken);
    }

    return fetch('/session/token')
      .then(function (response) {
        return response.text();
      })
      .then(function (token) {
        Drupal.filepond.csrfToken = token;
        return token;
      });
  };

  /**
   * Registers standard FilePond plugins.
   *
   * Call this before creating FilePond instances.
   * Safely checks if plugins are available before registering.
   */
  Drupal.filepond.registerPlugins = function () {
    if (
      typeof FilePondPluginFileValidateType !== 'undefined' &&
      !FilePond.find('filepond-plugin-file-validate-type')
    ) {
      FilePond.registerPlugin(FilePondPluginFileValidateType);
    }
    if (
      typeof FilePondPluginFileValidateSize !== 'undefined' &&
      !FilePond.find('filepond-plugin-file-validate-size')
    ) {
      FilePond.registerPlugin(FilePondPluginFileValidateSize);
    }
    if (
      typeof FilePondPluginImagePreview !== 'undefined' &&
      !FilePond.find('filepond-plugin-image-preview')
    ) {
      FilePond.registerPlugin(FilePondPluginImagePreview);
    }
    if (
      typeof FilePondPluginFilePoster !== 'undefined' &&
      !FilePond.find('filepond-plugin-file-poster')
    ) {
      FilePond.registerPlugin(FilePondPluginFilePoster);
    }
    if (
      typeof FilePondPluginImageCrop !== 'undefined' &&
      !FilePond.find('filepond-plugin-image-crop')
    ) {
      FilePond.registerPlugin(FilePondPluginImageCrop);
    }
  };

  /**
   * Builds server configuration with CSRF headers.
   *
   * @param {Object} settings
   *   Settings object with endpoint URLs.
   * @param {string} settings.processUrl
   *   The upload process endpoint.
   * @param {string} settings.revertUrl
   *   The revert/cancel endpoint.
   * @param {string} settings.patchUrl
   *   The chunked upload patch endpoint.
   * @param {string} csrfToken
   *   The CSRF token.
   *
   * @return {Object}
   *   FilePond server configuration object.
   */
  Drupal.filepond.buildServerConfig = function (settings, csrfToken) {
    const headers = {};
    if (csrfToken) {
      headers['X-CSRF-Token'] = csrfToken;
    }

    return {
      process: {
        url: settings.processUrl,
        headers,
      },
      revert: {
        url: settings.revertUrl,
        headers,
      },
      patch: {
        url: `${settings.patchUrl}/`,
        headers,
      },
    };
  };

  /**
   * Gets default FilePond labels (English).
   *
   * @return {Object}
   *   Object with label properties.
   */
  Drupal.filepond.getDefaultLabels = function () {
    return {
      labelIdle:
        'Drag & Drop your files or <span class="filepond--label-action">Browse</span>',
      labelFileProcessing: 'Uploading...',
      labelFileProcessingComplete: 'Upload complete',
      labelFileProcessingError: 'Upload error',
      labelTapToCancel: 'tap to cancel',
      labelTapToRetry: 'tap to retry',
    };
  };

  /**
   * Creates a FilePond instance with Drupal integration.
   *
   * @param {HTMLElement} element
   *   The input element to enhance.
   * @param {Object} options
   *   FilePond options. Required properties:
   *   - processUrl: Upload endpoint.
   *   - revertUrl: Revert endpoint.
   *   - patchUrl: Chunked upload endpoint.
   *
   * @return {Promise<Object>}
   *   Promise resolving to the FilePond instance.
   */
  Drupal.filepond.create = function (element, options) {
    return Drupal.filepond.fetchCsrfToken().then(function (csrfToken) {
      // Register plugins if not already done.
      Drupal.filepond.registerPlugins();

      // Build server config with CSRF token.
      const serverConfig = Drupal.filepond.buildServerConfig(
        {
          processUrl: options.processUrl,
          revertUrl: options.revertUrl,
          patchUrl: options.patchUrl,
        },
        csrfToken,
      );

      // Merge default labels with provided options.
      const labels = Drupal.filepond.getDefaultLabels();

      // Build final config.
      const config = {
        // Defaults.
        credits: false,
        allowMultiple: true,
        maxParallelUploads: 3,
        chunkUploads: true,
        chunkForce: false,
        imagePreviewMaxFileSize: '7MB',
        ...labels,
        ...options,
        // Server config always uses our built version.
        server: serverConfig,
      };

      // Create the instance.
      return FilePond.create(element, config);
    });
  };

  /**
   * Gets or creates a message wrapper for the given context.
   *
   * Checks in order:
   * 1. Modal context (jQuery UI dialog)
   * 2. Widget container (form-item wrapper)
   * 3. Returns null to fall back to page-level messages
   *
   * @param {HTMLElement} contextElement
   *   An element within the context (typically pond.element, the FilePond root).
   *
   * @return {HTMLElement|null}
   *   The message wrapper element, or null for page-level messages.
   */
  Drupal.filepond.getMessageWrapper = function (contextElement) {
    if (!contextElement) {
      return null;
    }

    // First check if we're inside a jQuery UI dialog (modal).
    let dialog = contextElement.closest('.ui-dialog');
    if (!dialog) {
      const pondRoot = contextElement.closest('.filepond--root');
      if (pondRoot) {
        dialog = pondRoot.closest('.ui-dialog');
      }
    }

    if (dialog) {
      const dialogContent = dialog.querySelector('.ui-dialog-content');
      if (dialogContent) {
        let dialogWrapper = dialogContent.querySelector(
          '[data-drupal-messages]',
        );
        if (!dialogWrapper) {
          dialogWrapper = document.createElement('div');
          dialogWrapper.setAttribute('data-drupal-messages', '');
          dialogContent.insertBefore(dialogWrapper, dialogContent.firstChild);
        }
        return dialogWrapper;
      }
    }

    // Not in a modal - find the widget container.
    // Look for form-item wrapper that contains this FilePond element.
    const formItem = contextElement.closest('.js-form-item, .form-item');
    if (formItem) {
      let formWrapper = formItem.querySelector('[data-drupal-messages]');
      if (!formWrapper) {
        // Create a message wrapper at the top of the form item.
        formWrapper = document.createElement('div');
        formWrapper.setAttribute('data-drupal-messages', '');
        formWrapper.classList.add('filepond-messages');
        formItem.insertBefore(formWrapper, formItem.firstChild);
      }
      return formWrapper;
    }

    return null;
  };

  /**
   * Gets or creates a message wrapper for modal context.
   *
   * @param {HTMLElement} contextElement
   *   An element within the modal (typically pond.element, the FilePond root).
   *
   * @return {HTMLElement|null}
   *   The message wrapper element, or null if not in modal.
   *
   * @deprecated Use getMessageWrapper() instead.
   */
  Drupal.filepond.getModalMessageWrapper = function (contextElement) {
    return Drupal.filepond.getMessageWrapper(contextElement);
  };

  /**
   * Helper to show a rejection message using Drupal.Message.
   *
   * @param {string} message
   *   The message to display.
   * @param {Object} options
   *   Options object.
   * @param {string} options.type
   *   Message type: 'warning', 'error', 'status'. Default 'warning'.
   * @param {string|null} options.previousMessageId
   *   ID of previous message to remove before adding new one.
   * @param {HTMLElement} options.context
   *   Context element to determine message placement. If inside a modal,
   *   messages will display within the modal instead of the main page.
   *
   * @return {string|null}
   *   The message ID, or null if Drupal.Message not available.
   */
  Drupal.filepond.showMessage = function (message, options) {
    options = options || {};
    const type = options.type || 'warning';

    if (typeof Drupal.Message === 'undefined') {
      return null;
    }

    // Determine message wrapper - use widget/modal wrapper if context provided.
    let wrapper = null;
    if (options.context) {
      wrapper = Drupal.filepond.getMessageWrapper(options.context);
    }

    const messages = wrapper
      ? new Drupal.Message(wrapper)
      : new Drupal.Message();

    // Remove previous message if specified.
    if (options.previousMessageId) {
      try {
        messages.remove(options.previousMessageId);
      } catch (e) {
        // Message may have already been removed or doesn't exist.
      }
    }

    const messageId = messages.add(message, { type });

    // Trigger fade-in animation after a brief delay for DOM render.
    if (messageId && wrapper) {
      setTimeout(function () {
        const messageEl = wrapper.querySelector(
          `[data-drupal-message-id="${messageId}"]`,
        );
        if (messageEl) {
          messageEl.classList.add('fade-in');
        }
      }, 10);
    }

    return messageId;
  };

  /**
   * Utility to check if FilePond is processing files.
   *
   * @param {Object} pond
   *   The FilePond instance.
   *
   * @return {boolean}
   *   TRUE if any files are processing.
   */
  Drupal.filepond.isProcessing = function (pond) {
    const files = pond.getFiles();
    return files.some(function (file) {
      const { status } = file;
      return (
        status === FilePond.FileStatus.PROCESSING ||
        status === FilePond.FileStatus.PROCESSING_QUEUED ||
        status === FilePond.FileStatus.LOADING
      );
    });
  };

  /**
   * Utility to count pending files (not yet completed).
   *
   * @param {Object} pond
   *   The FilePond instance.
   *
   * @return {number}
   *   Count of non-completed files.
   */
  Drupal.filepond.getPendingCount = function (pond) {
    const files = pond.getFiles();
    return files.filter(function (file) {
      return file.status !== FilePond.FileStatus.PROCESSING_COMPLETE;
    }).length;
  };

  /**
   * Registry of submit state validators.
   *
   * Modules can add validators that return TRUE if submit should be disabled.
   * All validators are checked - if any returns TRUE, submit is disabled.
   *
   * @type {Array}
   */
  Drupal.filepond.submitValidators = [];

  /**
   * Currently hovered FilePond instance for context-aware paste.
   *
   * @type {Object|null}
   */
  Drupal.filepond.hoveredPond = null;

  // Global paste listener - adds to hovered FilePond instance only.
  document.addEventListener('paste', function (e) {
    const pond = Drupal.filepond.hoveredPond;
    if (!pond) {
      return;
    }

    const { items } = e.clipboardData || e.originalEvent.clipboardData;
    let hasImage = false;

    for (let i = 0; i < items.length; i++) {
      if (items[i].type.indexOf('image') !== -1) {
        const file = items[i].getAsFile();
        if (file) {
          pond.addFile(file);
          hasImage = true;
        }
      }
    }

    if (hasImage) {
      e.preventDefault();
    }
  });

  /**
   * Registers a submit state validator.
   *
   * @param {Function} validator
   *   Function that receives (pond, form) and returns TRUE to disable submit.
   */
  Drupal.filepond.addSubmitValidator = function (validator) {
    Drupal.filepond.submitValidators.push(validator);
  };

  // Register the default validator: disable while processing.
  Drupal.filepond.addSubmitValidator(function (pond) {
    const files = pond.getFiles();
    return files.some(function (file) {
      const { status } = file;
      return (
        status === FilePond.FileStatus.PROCESSING ||
        status === FilePond.FileStatus.PROCESSING_QUEUED ||
        status === FilePond.FileStatus.LOADING
      );
    });
  });

  /**
   * Updates form submit button state based on registered validators.
   *
   * Runs all registered validators - if any returns TRUE, submit is disabled.
   *
   * @param {Object} pond
   *   The FilePond instance.
   */
  Drupal.filepondUpdateSubmitState = function (pond) {
    // Find the containing form.
    const form = pond.element.closest('form');
    if (!form) {
      return;
    }

    // Run all validators - disable if any returns true.
    const shouldDisable = Drupal.filepond.submitValidators.some(
      function (validator) {
        return validator(pond, form);
      },
    );

    // Find all submit buttons in the form.
    const submitButtons = form.querySelectorAll(
      'input[type="submit"], button[type="submit"], .form-submit',
    );

    submitButtons.forEach(function (button) {
      if (shouldDisable) {
        button.disabled = true;
        button.classList.add('filepond-disabled');
      } else {
        button.disabled = false;
        button.classList.remove('filepond-disabled');
      }
    });
  };
})(Drupal, drupalSettings);
