<?php

declare(strict_types=1);

namespace Drupal\filepond_eb_widget\Controller;

use Drupal\Component\Utility\Bytes;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\filepond\Controller\UploadController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Handles FilePond upload requests for Entity Browser widget context.
 *
 * Extends the main UploadController and adds EB-specific option resolution.
 * Routes encode entity_browser and widget_uuid in the URL path.
 *
 * Routes:
 * - POST /filepond/eb/process/{entity_browser}/{widget_uuid}
 * - PATCH /filepond/eb/patch/{entity_browser}/{widget_uuid}/{transferId}
 * - DELETE /filepond/eb/revert/{entity_browser}/{widget_uuid}
 */
class EbUploadController extends UploadController {

  /**
   * Access check for Entity Browser upload routes.
   *
   * @param string $entity_browser
   *   The entity browser ID.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(string $entity_browser): AccessResultInterface {
    $account = $this->currentUser();

    // Check filepond permission.
    if (!$account->hasPermission('filepond upload files')) {
      return AccessResult::forbidden()->cachePerPermissions();
    }

    // Check entity browser access permission.
    $eb_permission = 'access ' . $entity_browser . ' entity browser pages';
    return AccessResult::allowedIfHasPermission($account, $eb_permission);
  }

  /**
   * Gets upload options from Entity Browser widget config.
   *
   * @param string $entity_browser_id
   *   The entity browser ID.
   * @param string $widget_uuid
   *   The widget UUID.
   *
   * @return array
   *   Handler options array.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   If entity browser or widget not found.
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   If widget is not a FilePond widget or media type not configured.
   */
  protected function getOptionsFromWidget(string $entity_browser_id, string $widget_uuid): array {
    /** @var \Drupal\entity_browser\EntityBrowserInterface|null $entity_browser */
    $entity_browser = $this->entityTypeManager()
      ->getStorage('entity_browser')
      ->load($entity_browser_id);

    if (!$entity_browser) {
      throw new NotFoundHttpException('Entity browser not found');
    }

    $widget = $entity_browser->getWidgets()->get($widget_uuid);
    if (!$widget) {
      throw new NotFoundHttpException('Widget not found');
    }

    if ($widget->getPluginId() !== 'filepond_media') {
      throw new AccessDeniedHttpException('Invalid widget type');
    }

    $config = $widget->getConfiguration();
    $settings = $config['settings'] ?? [];

    $media_type_id = $settings['media_type'] ?? '';
    if (empty($media_type_id)) {
      throw new AccessDeniedHttpException('Media type not configured');
    }

    /** @var \Drupal\media\MediaTypeInterface|null $media_type */
    $media_type = $this->entityTypeManager()
      ->getStorage('media_type')
      ->load($media_type_id);

    if (!$media_type) {
      throw new AccessDeniedHttpException('Media type not found');
    }

    $source = $media_type->getSource();
    $source_field = $source->getSourceFieldDefinition($media_type);
    if (!$source_field) {
      throw new AccessDeniedHttpException('Source field not found');
    }

    // Build options from field settings (if inherit) or widget config.
    if (!empty($settings['inherit_settings'])) {
      $field_settings = $source_field->getSettings();
      $field_storage = $source_field->getFieldStorageDefinition()->getSettings();

      $extensions = $field_settings['file_extensions'] ?? 'jpg jpeg png gif';
      $max_filesize = $field_settings['max_filesize'] ?? '';
      $uri_scheme = $field_storage['uri_scheme'] ?? 'public';
      $file_directory = $this->token->replace($field_settings['file_directory'] ?? '');
      $upload_location = $uri_scheme . '://' . $file_directory;
    }
    else {
      $extensions = $settings['extensions'] ?? 'jpg jpeg png gif';
      $max_filesize = $settings['max_filesize'] ?? '';
      $upload_location = $this->token->replace($settings['upload_location'] ?? 'public://');
    }

    $extensions_array = array_filter(array_map('trim', explode(' ', strtolower($extensions))));

    $options = [
      'allowed_extensions' => $extensions_array,
      'allowed_mime_types' => $this->extensionsToMimeTypes($extensions_array),
      'destination' => $upload_location,
      'context' => [
        'entity_browser' => $entity_browser_id,
        'widget_uuid' => $widget_uuid,
        'media_type' => $media_type_id,
      ],
    ];

    if (!empty($max_filesize)) {
      $options['max_size'] = is_numeric($max_filesize)
        ? (int) $max_filesize
        : Bytes::toNumber($max_filesize);
    }

    return $options;
  }

  /**
   * Handles EB process requests (POST).
   */
  public function ebProcess(Request $request, string $entity_browser, string $widget_uuid): Response {
    $options = $this->getOptionsFromWidget($entity_browser, $widget_uuid);
    return $this->handleProcess($request, $options);
  }

  /**
   * Handles EB PATCH requests for chunked uploads.
   */
  public function ebPatch(Request $request, string $entity_browser, string $widget_uuid, string $transferId): Response {
    $options = $this->getOptionsFromWidget($entity_browser, $widget_uuid);
    return $this->handlePatch($request, $transferId, $options);
  }

  /**
   * Handles EB DELETE requests to revert/cancel uploads.
   */
  public function ebRevert(Request $request, string $entity_browser, string $widget_uuid): Response {
    // Validate widget exists.
    $this->getOptionsFromWidget($entity_browser, $widget_uuid);
    return $this->handleRevert($request);
  }

}
