<?php

declare(strict_types=1);

namespace Drupal\filepond_eb_widget\Plugin\EntityBrowser\Widget;

use Drupal\Component\Utility\Bytes;
use Drupal\Component\Utility\Environment;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\Core\Utility\Token;
use Drupal\entity_browser\WidgetBase;
use Drupal\file\FileInterface;
use Drupal\filepond\Element\FilePond;
use Drupal\filepond\MediaSourceFieldTrait;
use Drupal\media\MediaTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an Entity Browser widget that uploads and creates media entities.
 *
 * Uses the FilePond form element which creates file entities during upload.
 * On submit, this widget creates media entities from those files.
 *
 * @EntityBrowserWidget(
 *   id = "filepond_media",
 *   label = @Translation("FilePond Media Upload"),
 *   description = @Translation("Upload files with FilePond and create media entities."),
 *   auto_select = TRUE
 * )
 */
class FilePondMediaWidget extends WidgetBase {

  use MediaSourceFieldTrait;

  /**
   * The current user.
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The token service.
   */
  protected Token $token;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->currentUser = $container->get('current_user');
    $instance->token = $container->get('token');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return array_merge(parent::defaultConfiguration(), [
      'media_type' => '',
      'inherit_settings' => TRUE,
      'upload_location' => 'public://[date:custom:Y]-[date:custom:m]',
      'max_filesize' => Environment::getUploadMaxSize() / pow(Bytes::KILOBYTE, 2) . 'M',
      'extensions' => 'jpg jpeg gif png',
      'upload_prompt' => $this->t('Drag & drop files or <span class="filepond--label-action">Browse</span>'),
      'max_files' => 0,
      'chunk_size' => 5,
      'max_parallel_uploads' => 3,
      'item_panel_aspect_ratio' => '1:1',
      'preview_fit_mode' => 'contain',
    ]);
  }

  /**
   * Gets the configured media type.
   *
   * @return \Drupal\media\MediaTypeInterface|null
   *   The media type entity or NULL if not configured/found.
   */
  protected function getMediaType(): ?MediaTypeInterface {
    if (empty($this->configuration['media_type'])) {
      return NULL;
    }
    return $this->entityTypeManager
      ->getStorage('media_type')
      ->load($this->configuration['media_type']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $config = $this->configuration;

    // Media type selection.
    $form['media_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Media type'),
      '#required' => TRUE,
      '#default_value' => $config['media_type'],
      '#description' => $this->t('The type of media entity to create from uploaded files.'),
    ];

    $media_types = $this->entityTypeManager->getStorage('media_type')->loadMultiple();
    if (!empty($media_types)) {
      foreach ($media_types as $media_type) {
        $form['media_type']['#options'][$media_type->id()] = $media_type->label();
      }
    }
    else {
      $form['media_type']['#disabled'] = TRUE;
      $form['media_type']['#description'] = $this->t('You must @create_media_type before using this widget.', [
        '@create_media_type' => Link::createFromRoute($this->t('create a media type'), 'entity.media_type.add_form')->toString(),
      ]);
    }

    // Inherit settings from the media type's source field.
    $form['inherit_settings'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Inherit settings from the media type'),
      '#description' => $this->t('When enabled, upload location, file size, and extensions are taken from the media type source field configuration.'),
      '#default_value' => $config['inherit_settings'],
      '#weight' => 10,
      '#attributes' => ['class' => ['filepond-inherit-settings']],
    ];

    // States to show manual config fields only when inherit_settings is off.
    $manual_states = [
      'visible' => [
        '.filepond-inherit-settings' => ['checked' => FALSE],
      ],
    ];

    $form['upload_location'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Upload location'),
      '#default_value' => $config['upload_location'],
      '#description' => $this->t('Destination folder for uploaded files. Supports tokens like [date:custom:Y-m].'),
      '#weight' => 11,
      '#states' => $manual_states,
    ];

    // Extract numeric value from max_filesize (e.g., "10M" -> 10).
    preg_match('%\d+%', $config['max_filesize'], $matches);
    $max_filesize = !empty($matches) ? array_shift($matches) : '10';

    $form['max_filesize'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum file size'),
      '#min' => 1,
      '#field_suffix' => $this->t('MB'),
      '#default_value' => $max_filesize,
      '#weight' => 12,
      '#states' => $manual_states,
    ];

    $form['extensions'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Allowed file extensions'),
      '#description' => $this->t('Space-separated list of allowed extensions.'),
      '#default_value' => $config['extensions'],
      '#weight' => 13,
      '#states' => $manual_states,
    ];

    $form['upload_prompt'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Upload prompt'),
      '#description' => $this->t('Text shown in the upload area prompting users to drag or click.'),
      '#default_value' => $config['upload_prompt'],
      '#weight' => 20,
    ];

    $form['max_files'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum number of files per upload'),
      '#min' => 0,
      '#default_value' => $config['max_files'],
      '#description' => $this->t('0 for unlimited.'),
      '#weight' => 21,
    ];

    $form['chunk_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Chunk size'),
      '#min' => 1,
      '#max' => 50,
      '#field_suffix' => $this->t('MB'),
      '#default_value' => $config['chunk_size'],
      '#description' => $this->t('Size of each upload chunk for large files.'),
      '#weight' => 22,
    ];

    $form['max_parallel_uploads'] = [
      '#type' => 'number',
      '#title' => $this->t('Max parallel uploads'),
      '#min' => 1,
      '#max' => 10,
      '#default_value' => $config['max_parallel_uploads'],
      '#description' => $this->t('Number of files to upload simultaneously.'),
      '#weight' => 23,
    ];

    $form['item_panel_aspect_ratio'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Panel aspect ratio'),
      '#default_value' => $config['item_panel_aspect_ratio'],
      '#description' => $this->t('Aspect ratio for file panels in ratio format (e.g., "16:9", "4:3", "1:1").'),
      '#size' => 10,
      '#weight' => 24,
      '#element_validate' => [[FilePond::class, 'validateAspectRatio']],
    ];

    $form['preview_fit_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Preview fit mode'),
      '#options' => [
        'contain' => $this->t('Contain (letterbox)'),
        'cover' => $this->t('Cover (crop to fill)'),
      ],
      '#default_value' => $config['preview_fit_mode'],
      '#description' => $this->t('How image previews fit within the panel.'),
      '#weight' => 25,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues()['table'][$this->uuid()]['form'];

    if (!empty($values['extensions'])) {
      $extensions = explode(' ', $values['extensions']);
      foreach ($extensions as $extension) {
        if (preg_match('%^\w*$%', $extension) !== 1) {
          $form_state->setErrorByName('extensions', $this->t('Invalid extension list format.'));
          break;
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    // Append 'M' to max_filesize.
    $this->configuration['max_filesize'] = $this->configuration['max_filesize'] . 'M';
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies(): array {
    $dependencies = parent::calculateDependencies();

    $media_type = $this->getMediaType();
    if ($media_type) {
      $dependencies[$media_type->getConfigDependencyKey()][] = $media_type->getConfigDependencyName();
    }
    $dependencies['module'][] = 'media';

    return $dependencies;
  }

  /**
   * {@inheritdoc}
   */
  public function getForm(array &$original_form, FormStateInterface $form_state, array $additional_widget_parameters): array {
    $form = parent::getForm($original_form, $form_state, $additional_widget_parameters);
    $config = $this->getConfiguration();

    // Get cardinality from validators if set.
    $max_files = $config['settings']['max_files'] ?? 0;
    $validators = $form_state->get(['entity_browser', 'validators']);
    if (!empty($validators['cardinality']['cardinality']) && $validators['cardinality']['cardinality'] > 0) {
      $max_files = $validators['cardinality']['cardinality'];
    }

    // Get entity browser ID from the form object for EB-specific routes.
    /** @var \Drupal\entity_browser\Form\EntityBrowserForm $form_object */
    $form_object = $form_state->getFormObject();
    $entity_browser = $form_object->getEntityBrowser();
    $entity_browser_id = $entity_browser->id();
    $widget_uuid = $this->uuid();

    // Build EB-specific upload URLs that bypass tempstore. The controller
    // will load config directly from the EB widget and media type.
    $process_url = Url::fromRoute('filepond_eb_widget.process', [
      'entity_browser' => $entity_browser_id,
      'widget_uuid' => $widget_uuid,
    ])->toString();

    // Build patch URL with placeholder for transfer ID.
    $patch_url = Url::fromRoute('filepond_eb_widget.patch', [
      'entity_browser' => $entity_browser_id,
      'widget_uuid' => $widget_uuid,
      'transferId' => '__TRANSFER_ID__',
    ])->toString();
    // Remove the placeholder - JS will append the actual transfer ID.
    $patch_url = str_replace('/__TRANSFER_ID__', '', $patch_url);

    $revert_url = Url::fromRoute('filepond_eb_widget.revert', [
      'entity_browser' => $entity_browser_id,
      'widget_uuid' => $widget_uuid,
    ])->toString();

    // Get chunk size - config stores MB, element expects bytes.
    $chunk_size_mb = $config['settings']['chunk_size'] ?? 5;
    $chunk_size_bytes = $chunk_size_mb * 1024 * 1024;

    // FilePond upload element with EB-specific routes (bypasses tempstore).
    $form['upload'] = [
      '#type' => 'filepond',
      '#title' => $this->t('Upload files'),
      '#required' => TRUE,
      // Server-side properties (root level).
      // Use ?: NULL so empty values fall back to filepond.settings defaults.
      // Note: #max_files uses 0 for unlimited, so don't use ?: for it.
      '#extensions' => $config['settings']['extensions'] ?: NULL,
      '#max_filesize' => $config['settings']['max_filesize'] ?: NULL,
      '#max_files' => $max_files,
      '#upload_location' => $this->getUploadLocation() ?: NULL,
      // FilePond UI options.
      // Only set values if not empty - empty values fall back to
      // filepond.settings.
      '#config' => array_filter([
        'labelIdle' => $config['settings']['upload_prompt'] ?: NULL,
        'chunkSize' => $chunk_size_bytes ?: NULL,
        'maxParallelUploads' => $config['settings']['max_parallel_uploads'] ?: NULL,
        'styleItemPanelAspectRatio' => $config['settings']['item_panel_aspect_ratio'] ?: NULL,
        'previewFitMode' => $config['settings']['preview_fit_mode'] ?: NULL,
        // Disable reorder - files processed immediately, order irrelevant.
        'allowReorder' => FALSE,
        // Use EB routes instead of tempstore routes.
        'processUrl' => $process_url,
        'patchUrl' => $patch_url,
        'revertUrl' => $revert_url,
      ], fn($v) => $v !== NULL),
    ];

    // Attach widget styles.
    $form['#attached']['library'][] = 'filepond_eb_widget/widget';

    return $form;
  }

  /**
   * Gets the token-replaced upload location.
   *
   * @return string
   *   The upload location URI.
   */
  protected function getUploadLocation(): string {
    return $this->token->replace($this->configuration['upload_location']);
  }

  /**
   * {@inheritdoc}
   */
  public function validate(array &$form, FormStateInterface $form_state): void {
    $trigger = $form_state->getTriggeringElement();

    // Only validate on actual submit.
    if ($trigger['#type'] !== 'submit' || $trigger['#name'] !== 'op') {
      return;
    }

    // Get file IDs from the filepond element.
    $upload_value = $form_state->getValue(['upload']);
    $fids = $upload_value['fids'] ?? [];

    if (empty($fids)) {
      $form_state->setError($form['widget']['upload'], $this->t('Please upload at least one file.'));
      return;
    }

    // Verify media type is configured.
    $media_type = $this->getMediaType();
    if (!$media_type) {
      $form_state->setError($form['widget']['upload'], $this->t('No media type configured for this widget.'));
      return;
    }

    parent::validate($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function prepareEntities(array $form, FormStateInterface $form_state): array {
    $upload_value = $form_state->getValue(['upload']);
    $fids = $upload_value['fids'] ?? [];

    if (empty($fids)) {
      return [];
    }

    $media_type = $this->getMediaType();
    if (!$media_type) {
      return [];
    }

    // Load file entities.
    $file_storage = $this->entityTypeManager->getStorage('file');
    $files = $file_storage->loadMultiple($fids);

    // Get source field from media type.
    $source_field = $media_type->getSource()->getConfiguration()['source_field'];

    // Create media entities (not saved yet).
    $media_entities = [];
    foreach ($files as $file) {
      if (!$file instanceof FileInterface) {
        continue;
      }

      $media_entities[] = $this->entityTypeManager->getStorage('media')->create([
        'bundle' => $media_type->id(),
        $source_field => $this->buildSourceFieldValue($file, $media_type),
      ]);
    }

    return $media_entities;
  }

  /**
   * {@inheritdoc}
   */
  public function submit(array &$element, array &$form, FormStateInterface $form_state): void {
    $media_entities = $this->prepareEntities($form, $form_state);

    if (empty($media_entities)) {
      return;
    }

    $media_type = $this->getMediaType();
    $source_field = $media_type->getSource()->getConfiguration()['source_field'];

    // Save all media entities.
    foreach ($media_entities as $media_entity) {
      // Mark file as permanent before saving media.
      $file = $media_entity->get($source_field)->entity;
      if ($file instanceof FileInterface && !$file->isPermanent()) {
        $file->setPermanent();
        $file->save();
      }
      $media_entity->save();
    }

    $this->selectEntities($media_entities, $form_state);
    $this->clearFormValues($element, $form_state);
  }

  /**
   * Clears form values after submission.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function clearFormValues(array &$element, FormStateInterface $form_state): void {
    $form_state->setValueForElement($element['upload'], ['fids' => []]);
    NestedArray::setValue(
      $form_state->getUserInput(),
      array_merge($element['upload']['#parents'], ['fids']),
      ''
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function handleWidgetContext($widget_context): void {
    // Handle default widget context from parent.
    foreach ($this->defaultConfiguration() as $key => $value) {
      if (isset($widget_context[$key]) && isset($this->configuration[$key])) {
        $this->configuration[$key] = $widget_context[$key];
      }
    }

    // If inherit_settings is enabled, get settings from the media type.
    if ($this->configuration['inherit_settings']) {
      $media_type = $this->getMediaType();
      if ($media_type) {
        $source = $media_type->getSource();
        $field = $source->getSourceFieldDefinition($media_type);
        if ($field) {
          $field_storage = $field->getFieldStorageDefinition();
          $field_settings = $field->getSettings();

          // Build upload location from uri_scheme + file_directory.
          $uri_scheme = $field_storage->getSettings()['uri_scheme'] ?? 'public';
          $file_directory = $field_settings['file_directory'] ?? '';
          // Replace tokens in file_directory.
          $file_directory = $this->token->replace($file_directory);
          $this->configuration['upload_location'] = $uri_scheme . '://' . $file_directory;

          // Get max filesize from field settings.
          if (!empty($field_settings['max_filesize'])) {
            $this->configuration['max_filesize'] = $field_settings['max_filesize'];
          }

          // Get extensions from field settings.
          if (!empty($field_settings['file_extensions'])) {
            $this->configuration['extensions'] = $field_settings['file_extensions'];
          }
        }
      }
    }
  }

}
