<?php

declare(strict_types=1);

namespace Drupal\filepond\Controller;

use Drupal\Component\Serialization\Json;
use Drupal\Component\Utility\Bytes;
use Drupal\Component\Utility\Crypt;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\PrivateKey;
use Drupal\Core\Site\Settings;
use Drupal\Core\Utility\Token;
use Drupal\filepond\Element\FilePond;
use Drupal\filepond\FilePondUploadHandler;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Handles FilePond upload requests.
 *
 * This controller handles the FilePond server protocol with multiple route
 * types:
 *
 * Form-based routes (State API):
 * - POST /filepond/form/{form_id}/{element_name}/process
 * - PATCH /filepond/form/{form_id}/{element_name}/patch/{transferId}
 * - DELETE /filepond/form/{form_id}/{element_name}/revert
 *
 * Field widget routes (from field definition):
 * - POST /filepond/field/{entity_type}/{bundle}/{field_name}/process
 * - etc.
 *
 * Media library routes (from media type):
 * - POST /filepond/media/{media_type}/process
 * - etc.
 *
 * Form-based routes store config in State API keyed by form_id:element_name.
 * This is stored once per form element (not per user), making it efficient
 * for high-traffic forms.
 */
class UploadController extends ControllerBase {

  /**
   * The upload handler service.
   */
  protected FilePondUploadHandler $uploadHandler;

  /**
   * The entity field manager.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The token service.
   */
  protected Token $token;

  /**
   * The private key service.
   */
  protected PrivateKey $privateKey;

  /**
   * Constructs an UploadController.
   *
   * @param \Drupal\filepond\FilePondUploadHandler $upload_handler
   *   The upload handler service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Drupal\Core\PrivateKey $private_key
   *   The private key service.
   */
  public function __construct(
    FilePondUploadHandler $upload_handler,
    EntityFieldManagerInterface $entity_field_manager,
    Token $token,
    PrivateKey $private_key,
  ) {
    $this->uploadHandler = $upload_handler;
    $this->entityFieldManager = $entity_field_manager;
    $this->token = $token;
    $this->privateKey = $private_key;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('filepond.upload_handler'),
      $container->get('entity_field.manager'),
      $container->get('token'),
      $container->get('private_key'),
    );
  }

  /**
   * Handles form-based process requests (POST).
   *
   * For chunked uploads: Creates transfer ID when Upload-Length header present.
   * For regular uploads: Processes the file directly.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $form_id
   *   The form ID.
   * @param string $element_name
   *   The element name.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response with transfer ID or file ID.
   */
  public function formProcess(Request $request, string $form_id, string $element_name): Response {
    $options = $this->getOptionsFromState($form_id, $element_name);
    if ($options === NULL) {
      return new Response('Upload configuration not found for this form element', 404);
    }
    return $this->handleProcess($request, $options);
  }

  /**
   * Handles a process request (regular upload or chunked init).
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param array $options
   *   Upload options.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response with transfer ID or file ID.
   */
  protected function handleProcess(Request $request, array $options): Response {
    // Check for file in request FIRST.
    // If a file is present, this is a regular (non-chunked) upload.
    $file = $request->files->get('filepond');
    if ($file) {
      return $this->handleRegularUpload($request, $file, $options);
    }

    // No file in request - check if this is a chunked upload initialization.
    // FilePond sends Upload-Length header (without file) for chunked uploads.
    $uploadLength = $request->headers->get('Upload-Length', '');
    if (!empty($uploadLength)) {
      return $this->initializeChunkedUpload($request, $options);
    }

    // No file and no Upload-Length - invalid request.
    return new Response('No file uploaded and no Upload-Length header', 400);
  }

  /**
   * Initializes a chunked upload.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param array $options
   *   Upload options.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response with transfer ID.
   */
  protected function initializeChunkedUpload(Request $request, array $options): Response {
    $uploadLength = (int) $request->headers->get('Upload-Length', '0');
    // FilePond URL-encodes the filename. Default to .jpg if header missing.
    $rawUploadName = $request->headers->get('Upload-Name');
    $uploadName = urldecode($rawUploadName ?? 'unknown.jpg');

    if ($uploadLength <= 0) {
      return new Response('Invalid upload length', 400);
    }

    $result = $this->uploadHandler->initializeChunkedUpload(
      $uploadLength,
      $uploadName,
      $options
    );

    if (!$result['success']) {
      return new Response($result['error'] ?? 'Upload failed', $result['code'] ?? 500);
    }

    return new Response($result['transfer_id'], 200, [
      'Content-Type' => 'text/plain',
    ]);
  }

  /**
   * Handles a regular (non-chunked) file upload.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param \Symfony\Component\HttpFoundation\File\UploadedFile $file
   *   The uploaded file.
   * @param array $options
   *   Upload options.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response with file ID.
   */
  protected function handleRegularUpload(Request $request, $file, array $options): Response {
    $result = $this->uploadHandler->handleUpload($file, $options);

    if (!$result['success']) {
      return new Response($result['error'] ?? 'Upload failed', $result['code'] ?? 500);
    }

    // Return signed file ID - prevents attackers from guessing IDs.
    return new Response($this->signFileId($result['file']->id()), 200, [
      'Content-Type' => 'text/plain',
    ]);
  }

  /**
   * Handles form-based PATCH requests for chunked uploads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $form_id
   *   The form ID.
   * @param string $element_name
   *   The element name.
   * @param string $transferId
   *   The transfer ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function formPatch(Request $request, string $form_id, string $element_name, string $transferId): Response {
    $options = $this->getOptionsFromState($form_id, $element_name);
    if ($options === NULL) {
      return new Response('Upload configuration not found for this form element', 404);
    }
    return $this->handlePatch($request, $transferId, $options);
  }

  /**
   * Handles form-based DELETE requests to revert/cancel uploads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $form_id
   *   The form ID.
   * @param string $element_name
   *   The element name.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function formRevert(Request $request, string $form_id, string $element_name): Response {
    $options = $this->getOptionsFromState($form_id, $element_name);
    if ($options === NULL) {
      return new Response('Upload configuration not found for this form element', 404);
    }
    return $this->handleRevert($request);
  }

  /**
   * Handles a PATCH request for chunked uploads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request containing chunk data.
   * @param string $transferId
   *   The transfer ID.
   * @param array $options
   *   Upload options.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response with offset or file ID on completion.
   */
  protected function handlePatch(Request $request, string $transferId, array $options): Response {
    $offset = (int) $request->headers->get('Upload-Offset', '0');
    $chunk = $request->getContent();

    // FilePond sends Upload-Name and Upload-Length in PATCH requests.
    $uploadName = $request->headers->get('Upload-Name');
    if ($uploadName) {
      $options['upload_name'] = urldecode($uploadName);
    }
    $uploadLength = $request->headers->get('Upload-Length');
    if ($uploadLength) {
      $options['upload_length'] = (int) $uploadLength;
    }

    $result = $this->uploadHandler->processChunk($transferId, $chunk, $offset, $options);

    if (!$result['success']) {
      return new Response($result['error'] ?? 'Chunk failed', $result['code'] ?? 500);
    }

    // If complete, return JSON with transfer ID and signed file ID.
    if (!empty($result['complete']) && !empty($result['file'])) {
      return new Response(Json::encode([
        'transferId' => $transferId,
        'fileId' => $this->signFileId($result['file']->id()),
      ]), 200, [
        'Content-Type' => 'application/json',
      ]);
    }

    // Return new offset for next chunk.
    return new Response('', 204, [
      'Upload-Offset' => (string) $result['offset'],
    ]);
  }

  /**
   * Handles a DELETE request to revert/cancel uploads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request containing the server ID in the body.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Empty response on success.
   */
  protected function handleRevert(Request $request): Response {
    $serverId = trim($request->getContent());
    if (empty($serverId)) {
      return new Response('No server ID', 400);
    }

    // Determine the type of server ID and validate accordingly.
    // 1. Signed file ID (fid:hmac) - verify HMAC, extract file ID.
    // 2. Transfer ID (32-char hex) - temp file cleanup during chunked upload.
    // 3. Plain numeric ID - reject to prevent guessing attacks.
    if (str_contains($serverId, ':')) {
      // Signed file ID - verify HMAC.
      $fileId = $this->verifySignedFileId($serverId);
      if ($fileId === NULL) {
        return new Response('Invalid file token', 403);
      }
      // Pass verified file ID to handler.
      $serverId = $fileId;
    }
    elseif (is_numeric($serverId)) {
      // Plain numeric ID without signature - reject to prevent guessing.
      // All file IDs from our upload endpoints are signed. Existing files
      // use plain IDs but they're permanent so revert is a no-op anyway.
      // Return success so FilePond removes from UI.
      return new Response('skipped_unsigned', 200);
    }
    // Transfer ID (hex string) passes through to handler for temp cleanup.
    $result = $this->uploadHandler->revert($serverId);

    if (!$result['success']) {
      return new Response($result['error'] ?? 'Revert failed', $result['code'] ?? 500);
    }

    return new Response($result['action'] ?? 'success', 200);
  }

  /**
   * Retrieves element configuration from State API.
   *
   * Converts the stored config to the format expected by FilePondUploadHandler.
   *
   * @param string $form_id
   *   The form ID.
   * @param string $element_name
   *   The element name.
   *
   * @return array|null
   *   Upload options array or NULL if not found.
   */
  protected function getOptionsFromState(string $form_id, string $element_name): ?array {
    $config = FilePond::getConfig($form_id, $element_name);
    if ($config === NULL) {
      return NULL;
    }

    // Convert stored config to upload handler options format.
    $options = [];

    if (!empty($config['extensions'])) {
      // Convert space-separated string to array.
      $extensions = array_filter(
        explode(' ', strtolower($config['extensions']))
      );
      $options['allowed_extensions'] = $extensions;

      // Also build MIME types list from extensions.
      $options['allowed_mime_types'] = $this->extensionsToMimeTypes($extensions);
    }

    if (!empty($config['max_filesize'])) {
      // Handler expects bytes, config may be string like '10M' or bytes.
      if (is_numeric($config['max_filesize'])) {
        $options['max_size'] = (int) $config['max_filesize'];
      }
      else {
        $options['max_size'] = Bytes::toNumber($config['max_filesize']);
      }
    }

    if (!empty($config['upload_location'])) {
      // Replace tokens at upload time (not form render time) so user-specific
      // tokens like [current-user:uid] resolve to the uploading user.
      $destination = $this->token->replace(
        $config['upload_location'],
        ['current-user' => $this->currentUser()],
        ['clear' => TRUE]
      );
      $options['destination'] = $destination;
    }

    // Pass through context for event subscribers.
    if (!empty($config['context'])) {
      $options['context'] = $config['context'];
    }

    return $options;
  }

  /**
   * Converts file extensions to MIME types.
   *
   * @param array $extensions
   *   Array of file extensions.
   *
   * @return array
   *   Array of MIME types.
   *
   * @todo This hardcoded mapping only covers image types. Non-image extensions
   *   (pdf, mp4, etc.) won't have MIME types returned, so server-side MIME
   *   validation will fail for them. Consider using Drupal's core MIME type
   *   guesser service instead (like FilePond::extensionsToMimeTypes does),
   *   with legacy types added as needed. The legacy types (image/pjpeg,
   *   image/jpg) were added during testing to handle edge cases - verify
   *   they're still needed before removing.
   */
  protected function extensionsToMimeTypes(array $extensions): array {
    $mimeTypes = [];
    // Include legacy MIME types like image/pjpeg for compatibility.
    $mapping = [
      'jpg' => ['image/jpeg', 'image/jpg', 'image/pjpeg'],
      'jpeg' => ['image/jpeg', 'image/jpg', 'image/pjpeg'],
      'png' => ['image/png'],
      'gif' => ['image/gif'],
      'webp' => ['image/webp'],
    ];

    foreach ($extensions as $ext) {
      if (isset($mapping[$ext])) {
        $mimeTypes = array_merge($mimeTypes, $mapping[$ext]);
      }
    }

    return array_unique($mimeTypes);
  }

  /**
   * Gets upload options from field configuration.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle name.
   * @param string $field_name
   *   The field name.
   *
   * @return array
   *   Upload options array.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   If field not found.
   */
  protected function getOptionsFromField(string $entity_type, string $bundle, string $field_name): array {
    // Validate entity type exists.
    if (!$this->entityTypeManager()->hasDefinition($entity_type)) {
      throw new NotFoundHttpException('Entity type not found');
    }

    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type, $bundle);

    if (!isset($field_definitions[$field_name])) {
      throw new NotFoundHttpException('Field not found');
    }

    $field_definition = $field_definitions[$field_name];
    $settings = $field_definition->getSettings();
    $storage_settings = $field_definition->getFieldStorageDefinition()->getSettings();

    // Build upload location from field settings.
    $file_directory = $this->token->replace($settings['file_directory'] ?? '');
    $uri_scheme = $storage_settings['uri_scheme'] ?? 'public';
    $upload_location = $uri_scheme . '://' . $file_directory;

    // Get extensions.
    $extensions_string = $settings['file_extensions'] ?? 'png gif jpg jpeg';
    $extensions = array_filter(explode(' ', strtolower($extensions_string)));

    // Build options.
    $options = [
      'allowed_extensions' => $extensions,
      'allowed_mime_types' => $this->extensionsToMimeTypes($extensions),
      'destination' => $upload_location,
      'context' => [
        'entity_type' => $entity_type,
        'bundle' => $bundle,
        'field_name' => $field_name,
      ],
    ];

    // Add max size if set.
    if (!empty($settings['max_filesize'])) {
      if (is_numeric($settings['max_filesize'])) {
        $options['max_size'] = (int) $settings['max_filesize'];
      }
      else {
        $options['max_size'] = Bytes::toNumber($settings['max_filesize']);
      }
    }

    return $options;
  }

  /**
   * Handles field-based process requests (POST).
   */
  public function fieldProcess(Request $request, string $entity_type, string $bundle, string $field_name): Response {
    $options = $this->getOptionsFromField($entity_type, $bundle, $field_name);
    return $this->handleProcess($request, $options);
  }

  /**
   * Handles field-based PATCH requests for chunked uploads.
   */
  public function fieldPatch(Request $request, string $entity_type, string $bundle, string $field_name, string $transferId): Response {
    $options = $this->getOptionsFromField($entity_type, $bundle, $field_name);
    return $this->handlePatch($request, $transferId, $options);
  }

  /**
   * Handles field-based DELETE requests to revert/cancel uploads.
   */
  public function fieldRevert(Request $request, string $entity_type, string $bundle, string $field_name): Response {
    // Validate the field exists.
    $this->getOptionsFromField($entity_type, $bundle, $field_name);
    return $this->handleRevert($request);
  }

  /**
   * Gets upload options from media type's source field.
   *
   * @param string $media_type_id
   *   The media type ID.
   *
   * @return array
   *   Upload options array.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   If media type not found or has no file source field.
   */
  protected function getOptionsFromMediaType(string $media_type_id): array {
    /** @var \Drupal\media\MediaTypeInterface|null $media_type */
    $media_type = $this->entityTypeManager()->getStorage('media_type')->load($media_type_id);
    if (!$media_type) {
      throw new NotFoundHttpException('Media type not found');
    }

    // Get the source field for this media type.
    $source = $media_type->getSource();
    $source_field = $source->getSourceFieldDefinition($media_type);
    if (!$source_field) {
      throw new NotFoundHttpException('Media type has no source field');
    }

    $settings = $source_field->getSettings();
    $storage_settings = $source_field->getFieldStorageDefinition()->getSettings();

    // Build upload location from field settings.
    $file_directory = $this->token->replace($settings['file_directory'] ?? '');
    $uri_scheme = $storage_settings['uri_scheme'] ?? 'public';
    $upload_location = $uri_scheme . '://' . $file_directory;

    // Get extensions.
    $extensions_string = $settings['file_extensions'] ?? 'png gif jpg jpeg';
    $extensions = array_filter(explode(' ', strtolower($extensions_string)));

    // Build options.
    $options = [
      'allowed_extensions' => $extensions,
      'allowed_mime_types' => $this->extensionsToMimeTypes($extensions),
      'destination' => $upload_location,
      'context' => [
        'media_type' => $media_type_id,
      ],
    ];

    // Add max size if set.
    if (!empty($settings['max_filesize'])) {
      if (is_numeric($settings['max_filesize'])) {
        $options['max_size'] = (int) $settings['max_filesize'];
      }
      else {
        $options['max_size'] = Bytes::toNumber($settings['max_filesize']);
      }
    }

    return $options;
  }

  /**
   * Handles media-type-based process requests (POST).
   */
  public function mediaProcess(Request $request, string $media_type): Response {
    $options = $this->getOptionsFromMediaType($media_type);
    return $this->handleProcess($request, $options);
  }

  /**
   * Handles media-type-based PATCH requests for chunked uploads.
   */
  public function mediaPatch(Request $request, string $media_type, string $transferId): Response {
    $options = $this->getOptionsFromMediaType($media_type);
    return $this->handlePatch($request, $transferId, $options);
  }

  /**
   * Handles media-type-based DELETE requests to revert/cancel uploads.
   */
  public function mediaRevert(Request $request, string $media_type): Response {
    // Validate the media type exists.
    $this->getOptionsFromMediaType($media_type);
    return $this->handleRevert($request);
  }

  /**
   * Signs a file ID to create a tamper-proof token.
   *
   * This prevents attackers from guessing file IDs and deleting other users'
   * uploads. The token format is "fid:hmac" where the HMAC is computed using
   * the site's private key.
   *
   * @param int|string $fileId
   *   The file entity ID.
   *
   * @return string
   *   The signed token in format "fid:hmac".
   */
  public function signFileId(int|string $fileId): string {
    $hmac = Crypt::hmacBase64(
      'filepond-file-' . $fileId,
      $this->privateKey->get() . Settings::getHashSalt()
    );
    return $fileId . ':' . $hmac;
  }

  /**
   * Verifies a signed file ID token and extracts the file ID.
   *
   * @param string $signedToken
   *   The signed token in format "fid:hmac".
   *
   * @return string|null
   *   The file ID if valid, or NULL if verification fails.
   */
  protected function verifySignedFileId(string $signedToken): ?string {
    // Check for colon separator.
    if (!str_contains($signedToken, ':')) {
      return NULL;
    }

    [$fileId, $providedHmac] = explode(':', $signedToken, 2);

    // File ID must be numeric.
    if (!is_numeric($fileId)) {
      return NULL;
    }

    // Compute expected HMAC and compare.
    $expectedHmac = Crypt::hmacBase64(
      'filepond-file-' . $fileId,
      $this->privateKey->get() . Settings::getHashSalt()
    );

    if (!hash_equals($expectedHmac, $providedHmac)) {
      return NULL;
    }

    return $fileId;
  }

}
