<?php

declare(strict_types=1);

namespace Drupal\filepond\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\file\FileInterface;
use Drupal\filepond\MediaSourceFieldTrait;
use Drupal\media\MediaTypeInterface;
use Drupal\media_library\Form\FileUploadForm;

/**
 * Creates a form to create media entities from FilePond uploaded files.
 *
 * Unlike core's FileUploadForm which uses managed_file, this form uses the
 * FilePond upload element. File entities are created during upload (not on
 * form submit), so we just need to load them by ID and process them.
 */
class FilePondMediaUploadForm extends FileUploadForm {

  use MediaSourceFieldTrait;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return $this->getBaseFormId() . '_filepond';
  }

  /**
   * {@inheritdoc}
   */
  protected function buildInputElement(array $form, FormStateInterface $form_state) {
    $media_type = $this->getMediaType($form_state);
    $item = $this->createFileItem($media_type);

    /** @var \Drupal\media_library\MediaLibraryState $state */
    $state = $this->getMediaLibraryState($form_state);
    if (!$state->hasSlotsAvailable()) {
      return $form;
    }

    $config = $this->configFactory()->get('filepond.settings');
    $settings = $item->getFieldDefinition()->getSettings();

    // Limit cardinality to the default settings, if they are set.
    $slots = $config->get('defaults.max_files') ?: $state->getAvailableSlots();
    // FilePond uses 0 as unlimited, Drupal uses -1.
    $max_files = $slots < 1 ? 0 : $slots;

    // Get process callbacks from the filepond element.
    $process = (array) $this->elementInfo->getInfoProperty('filepond', '#process', []);

    // Build media-type-based upload URLs (no tempstore needed).
    $media_type_id = $media_type->id();
    $process_url = Url::fromRoute('filepond.media_process', [
      'media_type' => $media_type_id,
    ])->toString();
    $patch_url = Url::fromRoute('filepond.media_patch', [
      'media_type' => $media_type_id,
      'transferId' => '__TRANSFER_ID__',
    ])->toString();
    // Remove the placeholder suffix - JS will append the actual transfer ID.
    $patch_url = str_replace('/__TRANSFER_ID__', '', $patch_url);
    $revert_url = Url::fromRoute('filepond.media_revert', [
      'media_type' => $media_type_id,
    ])->toString();

    // Add a container to group the input elements for styling purposes.
    $form['container'] = [
      '#type' => 'container',
    ];

    // Build the FilePond element.
    $form['container']['upload'] = [
      '#type' => 'filepond',
      '#title' => $this->formatPlural($max_files ?: 10, 'Add file', 'Add files'),
      '#process' => array_merge(['::validateUploadElement'], $process),
      '#cardinality' => $max_files ?: FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
      // Server-side properties (root level).
      '#extensions' => $settings['file_extensions'] ?: NULL,
      '#max_filesize' => $settings['max_filesize'] ?: NULL,
      '#max_files' => $max_files,
      '#upload_location' => $item->getUploadLocation() ?: NULL,
    ];

    // Get defaults from config.
    $defaults = $config->get('defaults') ?? [];

    // FilePond UI options - use defaults from config.
    $form['container']['upload']['#config'] = [
      'labelIdle' => $defaults['upload_prompt'] ?? NULL,
      // Disable reorder - irrelevant in this context.
      'allowReorder' => FALSE,
      'maxParallelUploads' => $defaults['max_parallel_uploads'] ?? 3,
      'styleItemPanelAspectRatio' => $defaults['item_panel_aspect_ratio'] ?? '1:1',
      'previewFitMode' => $defaults['preview_fit_mode'] ?? 'contain',
      'processUrl' => $process_url,
      'patchUrl' => $patch_url,
      'revertUrl' => $revert_url,
    ];

    // Hidden element for AJAX auto-select functionality.
    // When FilePond finishes processing all files, it triggers this handler.
    $form['auto_select_handler'] = [
      '#type' => 'hidden',
      '#name' => 'auto_select_handler',
      '#id' => 'auto_select_handler',
      '#attributes' => ['id' => 'auto_select_handler'],
      '#submit' => ['::uploadButtonSubmit'],
      '#executes_submit_callback' => TRUE,
      '#ajax' => [
        'callback' => '::updateFormCallback',
        'wrapper' => 'media-library-wrapper',
        'event' => 'auto_select_media_library_widget',
        // Add a fixed URL to post the form since AJAX forms are automatically
        // posted to <current> instead of $form['#action'].
        // @see https://www.drupal.org/project/drupal/issues/2504115
        'url' => Url::fromRoute('media_library.ui'),
        'options' => [
          'query' => $this->getMediaLibraryState($form_state)->all() + [
            FormBuilderInterface::AJAX_FORM_REQUEST => TRUE,
          ],
        ],
      ],
    ];

    $form['#attached']['library'][] = 'filepond/filepond.media_library';

    return $form;
  }

  /**
   * Validates the upload element.
   *
   * @param array $element
   *   The upload element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The processed upload element.
   */
  public function validateUploadElement(array $element, FormStateInterface $form_state) {
    if ($form_state::hasAnyErrors()) {
      // When an error occurs, clear the value so user can re-upload.
      $element['#value'] = ['fids' => []];
    }

    // Get file IDs from the FilePond element.
    $values = $form_state->getValue('upload', []);
    $fids = $values['fids'] ?? [];

    // Check cardinality if not unlimited.
    if (count($fids) > $element['#cardinality'] && $element['#cardinality'] !== FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED) {
      $form_state->setError($element, $this->t('A maximum of @count files can be uploaded.', [
        '@count' => $element['#cardinality'],
      ]));
      $form_state->setValue('upload', ['fids' => []]);
      $element['#value'] = ['fids' => []];
    }

    return $element;
  }

  /**
   * Submit handler for the upload button / auto-select handler.
   *
   * @param array $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function uploadButtonSubmit(array $form, FormStateInterface $form_state) {
    $files = $this->getFiles($form, $form_state);
    $this->processInputValues($files, $form, $form_state);
  }

  /**
   * Gets uploaded files from the FilePond element.
   *
   * FilePond creates file entities during upload, so we just need to load
   * them by ID. This is simpler than dropzonejs which has temp files.
   *
   * @param array $form
   *   Form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   *
   * @return \Drupal\file\FileInterface[]
   *   Array of uploaded files.
   */
  protected function getFiles(array $form, FormStateInterface $form_state): array {
    // Check if we already have files in form state.
    // Prevents double processing.
    $files = $form_state->get(['filepond', $this->getFormId(), 'files']);
    if ($files) {
      return $files;
    }

    $files = [];

    // Get file IDs from the FilePond element value.
    $values = $form_state->getValue(['upload'], []);
    $fids = $values['fids'] ?? [];

    if (!empty($fids)) {
      /** @var \Drupal\file\FileStorageInterface $file_storage */
      $file_storage = $this->entityTypeManager->getStorage('file');
      $loaded_files = $file_storage->loadMultiple($fids);

      // Filter to only include valid file entities.
      foreach ($loaded_files as $file) {
        if ($file instanceof FileInterface) {
          $files[] = $file;
        }
      }
    }

    // Limit to max_files if set.
    $max_files = $form['container']['upload']['#max_files'] ?? 0;
    if ($max_files > 0) {
      $files = array_slice($files, -$max_files);
    }

    // Store in form state.
    $form_state->set(['filepond', $this->getFormId(), 'files'], $files);

    return $files;
  }

  /**
   * {@inheritdoc}
   *
   * Override parent to skip file move - FilePond already placed files at
   * their final destination during upload.
   */
  protected function createMediaFromValue(
    MediaTypeInterface $media_type,
    EntityStorageInterface $media_storage,
    $source_field_name,
    $file,
  ) {
    if (!($file instanceof FileInterface)) {
      throw new \InvalidArgumentException('Cannot create a media item without a file entity.');
    }

    // Skip the file move that FileUploadForm does - FilePond already uploaded
    // to the correct location. Just create the media entity using AddFormBase.
    // We call the grandparent method to avoid FileUploadForm's file move.
    $media = $media_storage->create([
      'bundle' => $media_type->id(),
      $source_field_name => $this->buildSourceFieldValue($file, $media_type),
    ]);
    $media->setName($media->getName());

    return $media;
  }

  /**
   * {@inheritdoc}
   */
  public function updateFormCallback(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $added_media = $form_state->get('media');
    $response = new AjaxResponse();

    // If auto_select_handler triggered but no media was added, refresh form.
    if (empty($added_media) && end($triggering_element['#parents']) === 'auto_select_handler') {
      $response->addCommand(new ReplaceCommand('#media-library-add-form-wrapper', $form));
    }
    else {
      $response = parent::updateFormCallback($form, $form_state);
    }

    return $response;
  }

}
