<?php

declare(strict_types=1);

namespace Drupal\filepond;

use Drupal\file\FileInterface;
use Drupal\media\MediaTypeInterface;

/**
 * Provides helper methods for building media source field values.
 */
trait MediaSourceFieldTrait {

  /**
   * Builds the source field value for a media entity.
   *
   * For image fields, includes dimensions to avoid ImageItem::preSave()
   * downloading the entire file from S3 just to read header bytes.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   * @param \Drupal\media\MediaTypeInterface $media_type
   *   The media type.
   *
   * @return array|\Drupal\file\FileInterface
   *   Array with target_id/width/height for image fields, or file entity.
   */
  protected function buildSourceFieldValue(FileInterface $file, MediaTypeInterface $media_type): array|FileInterface {
    $source = $media_type->getSource();
    $field_definition = $source->getSourceFieldDefinition($media_type);

    if ($field_definition && $field_definition->getType() === 'image') {
      $dimensions = ImageDimensionHelper::getDimensions($file);
      return [
        'target_id' => $file->id(),
        'width' => $dimensions['width'] ?? NULL,
        'height' => $dimensions['height'] ?? NULL,
      ];
    }

    return $file;
  }

}
