/**
 * @file
 * FilePond Views integration.
 *
 * Refreshes the view when all uploads are complete and optionally
 * auto-selects newly created media in Entity Browser.
 */

/* global FilePond */

(function (Drupal, drupalSettings, once) {
  /**
   * Extracts media ID from FilePond server response.
   *
   * For regular uploads, serverId is JSON with mediaId.
   * For chunked uploads, serverId is the transfer ID, and mediaId
   * is stored in the global Drupal.filepond.transferToMediaId map.
   *
   * @param {string} serverId
   *   The server ID returned by FilePond.
   *
   * @return {string|null}
   *   The media ID or null if not present.
   */
  function extractMediaId(serverId) {
    if (!serverId) {
      return null;
    }

    // Try parsing as JSON first (regular uploads).
    try {
      const data = JSON.parse(serverId);
      if (data.mediaId) {
        return String(data.mediaId);
      }
    } catch (e) {
      // Not JSON - likely a transfer ID from chunked upload.
      // Check the global mapping.
      if (Drupal.filepond?.transferToMediaId?.[serverId]) {
        return String(Drupal.filepond.transferToMediaId[serverId]);
      }
    }

    return null;
  }

  /**
   * Auto-selects media items in Entity Browser view.
   *
   * @param {Array} mediaIds
   *   Array of media IDs to select.
   * @param {string} viewDomId
   *   The view's DOM ID.
   */
  function autoSelectMedia(mediaIds, viewDomId) {
    if (!mediaIds.length) {
      return;
    }

    const viewContainer = document.querySelector(
      `.js-view-dom-id-${viewDomId}`,
    );
    if (!viewContainer) {
      return;
    }

    // Set auto-select mode flag so eb_extras knows not to show alerts.
    // It will silently skip items that exceed cardinality.
    const $form = jQuery('form.entity-browser-enhanced.multiselect');
    $form.data('autoSelectMode', true);

    // Find and click each media item's checkbox/selection element.
    mediaIds.forEach((mediaId) => {
      // Entity Browser uses data-entity-id on selectable items.
      const selector = `[data-entity-id="media:${mediaId}"]`;
      const item = viewContainer.querySelector(selector);

      if (item) {
        // Trigger click on the item to select it.
        item.click();
      } else {
        // Try alternate selector - some views use input checkboxes.
        const checkbox = viewContainer.querySelector(
          `input[name*="entity_browser_select"][value="media:${mediaId}"]`,
        );
        if (checkbox && !checkbox.checked) {
          checkbox.click();
        }
      }
    });

    // Clear auto-select mode flag.
    $form.data('autoSelectMode', false);
  }

  /**
   * Handles the processfiles event - refreshes view after uploads complete.
   *
   * @param {Object} pond
   *   The FilePond instance.
   * @param {string} viewDomId
   *   The view's DOM ID for refresh targeting.
   * @param {HTMLElement} element
   *   The uploader element for message context.
   * @param {Object} settings
   *   View settings from drupalSettings.
   */
  function handleProcessFiles(pond, viewDomId, element, settings) {
    const files = pond.getFiles();
    const processedFiles = files.filter(
      (file) => file.status === FilePond.FileStatus.PROCESSING_COMPLETE,
    );

    if (processedFiles.length === 0) {
      return;
    }

    // Collect media IDs from processed files for auto-select.
    const mediaIds = [];
    if (settings.autoSelect) {
      processedFiles.forEach((file) => {
        const mediaId = extractMediaId(file.serverId);
        if (mediaId) {
          mediaIds.push(mediaId);
        }
      });
    }

    // Clear FilePond after successful uploads.
    pond.removeFiles();

    // Trigger view refresh.
    const viewContainer = document.querySelector(
      `.js-view-dom-id-${viewDomId}`,
    );
    if (viewContainer && typeof jQuery !== 'undefined') {
      const $view = jQuery(viewContainer);

      // If auto-select is enabled, store media IDs in drupalSettings.
      // The behavior will pick these up when it re-attaches to the refreshed view.
      if (settings.autoSelect && mediaIds.length) {
        drupalSettings.filepondViews = drupalSettings.filepondViews || {};
        drupalSettings.filepondViews.pendingAutoSelect =
          drupalSettings.filepondViews.pendingAutoSelect || {};
        drupalSettings.filepondViews.pendingAutoSelect[viewDomId] = mediaIds;
      }

      $view.trigger('RefreshView');
    }

    // Show success message.
    // Message wrapper is placed outside the view, so it persists through refresh.
    const message = Drupal.t('@count file(s) uploaded successfully.', {
      '@count': processedFiles.length,
    });
    Drupal.filepond.showMessage(message, {
      type: 'status',
      context: element,
    });
  }

  /**
   * Handles FilePond initialization - sets up processfiles listener.
   *
   * @param {CustomEvent} e
   *   The filepond:init event.
   * @param {string} viewDomId
   *   The view's DOM ID.
   * @param {HTMLElement} element
   *   The uploader element.
   */
  function handleFilePondInit(e, viewDomId, element) {
    const { pond } = e.detail;
    if (!pond) {
      return;
    }

    // Get settings for this view.
    const settings =
      Drupal.settings?.filepondViews?.[viewDomId] ||
      drupalSettings?.filepondViews?.[viewDomId] ||
      {};

    // Get cardinality limit from Entity Browser (remaining slots).
    // 0 = unlimited, positive = limit.
    const cardinality = drupalSettings?.filepondViews?.cardinality || 0;

    // Apply max files limit using core utility.
    // This handles allowMultiple, beforeAddFile, and message display.
    if (cardinality > 0) {
      Drupal.filepond.setMaxFiles(pond, cardinality);
    }

    pond.on('processfiles', () =>
      handleProcessFiles(pond, viewDomId, element, settings),
    );
  }

  /**
   * Behavior for FilePond views area.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.filepondViews = {
    attach(context) {
      const uploaders = once('filepond-views', '[data-views-dom-id]', context);

      uploaders.forEach((element) => {
        const viewDomId = element.dataset.viewsDomId;
        if (!viewDomId) {
          return;
        }

        element.addEventListener('filepond:init', (e) =>
          handleFilePondInit(e, viewDomId, element),
        );
      });
    },
  };

  /**
   * Behavior for auto-selecting media after view refresh.
   *
   * When views refresh via AJAX, behaviors re-attach. This behavior checks
   * for pending auto-select media IDs and selects them.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.filepondViewsAutoSelect = {
    attach(context) {
      // Check if we have any pending auto-select.
      const pending = drupalSettings.filepondViews?.pendingAutoSelect;

      if (!pending || !Object.keys(pending).length) {
        return;
      }

      // Check if context itself is a view (AJAX refresh passes view as context).
      // Also check for views inside context (initial page load).
      let views = [];
      if (context.className?.includes('js-view-dom-id-')) {
        views = [context];
      } else if (context.querySelectorAll) {
        views = Array.from(
          context.querySelectorAll('[class*="js-view-dom-id-"]'),
        );
      }

      views.forEach((viewContainer) => {
        // Extract view DOM ID from class name.
        const classMatch = viewContainer.className.match(
          /js-view-dom-id-([a-f0-9]+)/,
        );
        if (!classMatch) {
          return;
        }

        const viewDomId = classMatch[1];

        // Check if we have pending media IDs to select for this view.
        if (pending[viewDomId]?.length) {
          const mediaIds = pending[viewDomId];

          // Clear pending before selecting to avoid re-triggering.
          delete pending[viewDomId];

          // Small delay to ensure DOM is fully ready.
          setTimeout(() => autoSelectMedia(mediaIds, viewDomId), 100);
        }
      });
    },
  };
})(Drupal, drupalSettings, once);
