<?php

declare(strict_types=1);

namespace Drupal\filepond_views\Plugin\views\area;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\Core\Utility\Token;
use Drupal\filepond\FilePondConfigFormTrait;
use Drupal\views\Attribute\ViewsArea;
use Drupal\views\Plugin\views\area\AreaPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Views area handler to display a FilePond uploader.
 *
 * Creates media entities on upload and refreshes the view to display them.
 *
 * @ingroup views_area_handlers
 */
#[ViewsArea("filepond_upload")]
class FilePondUploadArea extends AreaPluginBase {

  use FilePondConfigFormTrait;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The token service.
   */
  protected Token $token;

  /**
   * The current user.
   */
  protected AccountProxyInterface $currentUser;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->token = $container->get('token');
    $instance->currentUser = $container->get('current_user');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions(): array {
    $options = parent::defineOptions();

    // Use defaults from the trait.
    foreach ($this->getFilePondDefaults() as $key => $value) {
      $options[$key] = ['default' => $value];
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);

    // Build media type options.
    $media_types = $this->entityTypeManager->getStorage('media_type')->loadMultiple();
    $media_type_options = [];
    foreach ($media_types as $type) {
      $media_type_options[$type->id()] = $type->label();
    }

    // Use shared form builder from trait.
    $form = $this->buildFilePondConfigForm($form, $this->options, [
      'media_type_options' => $media_type_options,
      'show_auto_select' => TRUE,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::validateOptionsForm($form, $form_state);
    $values = $form_state->getValue('options');
    $this->validateFilePondConfigForm($values, $form_state, 'options][');
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::submitOptionsForm($form, $form_state);
    // Process values (e.g., append 'M' to max_filesize).
    $options = $form_state->getValue('options');
    $options = $this->processFilePondConfigValues($options);
    $form_state->setValue('options', $options);
  }

  /**
   * Gets the media type to use for uploads.
   *
   * @return string|null
   *   The media type ID or NULL if none configured.
   */
  protected function getMediaType(): ?string {
    return $this->options['media_type'] ?: NULL;
  }

  /**
   * Gets upload settings based on inherit_settings option.
   *
   * @param \Drupal\media\MediaTypeInterface $media_type
   *   The media type entity.
   *
   * @return array
   *   Array with extensions, max_filesize, and upload_location.
   */
  protected function getUploadSettings($media_type): array {
    if (!empty($this->options['inherit_settings'])) {
      // Get settings from media type's source field.
      $source = $media_type->getSource();
      $source_field = $source->getSourceFieldDefinition($media_type);
      if ($source_field) {
        $field_settings = $source_field->getSettings();
        return [
          'extensions' => $field_settings['file_extensions'] ?? 'jpg jpeg gif png',
          'max_filesize' => $field_settings['max_filesize'] ?? NULL,
        ];
      }
    }

    // Use manual settings from plugin options.
    return [
      'extensions' => $this->options['extensions'] ?? 'jpg jpeg gif png',
      'max_filesize' => $this->options['max_filesize'] ?? NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render($empty = FALSE): array {
    // Respect the "empty" setting.
    if ($empty && empty($this->options['empty'])) {
      return [];
    }

    // Check permission.
    if (!$this->currentUser->hasPermission('filepond upload files')) {
      return [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [new TranslatableMarkup("You don't have permission to upload files.")],
        ],
      ];
    }

    $media_type_id = $this->getMediaType();
    if (!$media_type_id) {
      return [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [new TranslatableMarkup('FilePond Upload: No media type configured or detected.')],
        ],
      ];
    }

    // Verify media type exists.
    /** @var \Drupal\media\MediaTypeInterface|null $media_type */
    $media_type = $this->entityTypeManager->getStorage('media_type')->load($media_type_id);
    if (!$media_type) {
      return [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [new TranslatableMarkup('FilePond Upload: Media type "@type" not found.', ['@type' => $media_type_id])],
        ],
      ];
    }

    // Get upload settings (from media type or manual config).
    $upload_settings = $this->getUploadSettings($media_type);

    // Build upload URLs using view_id and display_id.
    $view_id = $this->view->storage->id();
    $display_id = $this->view->current_display;
    $view_dom_id = $this->view->dom_id;

    $process_url = Url::fromRoute('filepond_views.process', [
      'view_id' => $view_id,
      'display_id' => $display_id,
    ])->toString();
    $patch_url = Url::fromRoute('filepond_views.patch', [
      'view_id' => $view_id,
      'display_id' => $display_id,
      'transferId' => '__TRANSFER_ID__',
    ])->toString();
    // Remove the placeholder suffix - JS will append the actual transfer ID.
    $patch_url = str_replace('/__TRANSFER_ID__', '', $patch_url);
    $revert_url = Url::fromRoute('filepond_views.revert', [
      'view_id' => $view_id,
      'display_id' => $display_id,
    ])->toString();

    // Generate unique element ID for this instance.
    $element_id = 'filepond-views-' . $view_dom_id;

    // Get chunk size - config stores MB, element expects bytes.
    $chunk_size_mb = $this->options['chunk_size'] ?? 5;
    $chunk_size_bytes = $chunk_size_mb * 1024 * 1024;

    // Build FilePond config from options.
    $config = array_filter([
      'labelIdle' => $this->options['upload_prompt'] ?: NULL,
      'chunkSize' => $chunk_size_bytes ?: NULL,
      'maxParallelUploads' => $this->options['max_parallel_uploads'] ?: NULL,
      'styleItemPanelAspectRatio' => $this->options['item_panel_aspect_ratio'] ?: NULL,
      'previewFitMode' => $this->options['preview_fit_mode'] ?: NULL,
      'allowReorder' => FALSE,
    ], fn($v) => $v !== NULL);

    // Use the FilePondUploader render element from the main module.
    $build = [
      '#type' => 'filepond_uploader',
      '#id' => $element_id,
      '#extensions' => $upload_settings['extensions'],
      '#max_files' => (int) ($this->options['max_files'] ?? 0),
      '#max_filesize' => $upload_settings['max_filesize'],
      '#process_url' => $process_url,
      '#patch_url' => $patch_url,
      '#revert_url' => $revert_url,
      '#config' => $config,
      '#columns' => $this->options['columns'] ?? 5,
      '#max_width' => $this->options['max_width'] ?? NULL,
      '#attributes' => [
        'class' => ['filepond-views-area'],
        'data-views-dom-id' => $view_dom_id,
      ],
      // Attach views-specific library and settings for refresh behavior.
      '#attached' => [
        'library' => [
          'filepond_views/filepond_views',
        ],
        'drupalSettings' => [
          'filepondViews' => [
            $view_dom_id => [
              'mediaType' => $media_type_id,
              'autoSelect' => !empty($this->options['auto_select']),
            ],
          ],
        ],
      ],
    ];

    // Add description if provided.
    if (!empty($this->options['description'])) {
      $build = [
        'description' => [
          '#type' => 'html_tag',
          '#tag' => 'div',
          '#value' => $this->options['description'],
          '#attributes' => ['class' => ['filepond-description']],
        ],
        'uploader' => $build,
      ];
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    // The uploader is never empty - it always shows.
    return FALSE;
  }

}
