<?php

declare(strict_types=1);

namespace Drupal\filepond_views;

use Drupal\views\ViewExecutable;

/**
 * Provides a method for detecting media type from view's bundle filter.
 *
 * This trait consolidates the logic used by both FilePondUploadArea and
 * ViewsUploadController to detect the media type from a view's filters.
 */
trait ViewsMediaTypeDetectorTrait {

  /**
   * Detects media type from the view's bundle filter.
   *
   * For views based on media_field_data, looks for a bundle filter and
   * returns the bundle value if exactly one is configured. If multiple
   * bundles are allowed, returns NULL to require explicit configuration.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view executable.
   *
   * @return string|null
   *   The media type ID, or NULL if not detectable or multiple types allowed.
   */
  protected function detectMediaTypeFromView(ViewExecutable $view): ?string {
    // Only works for media views.
    if ($view->storage->get('base_table') !== 'media_field_data') {
      return NULL;
    }

    $filters = $view->display_handler->getOption('filters');
    if (empty($filters)) {
      return NULL;
    }

    foreach ($filters as $filter) {
      if (($filter['field'] ?? '') === 'bundle' && !empty($filter['value'])) {
        $values = is_array($filter['value']) ? array_keys($filter['value']) : [$filter['value']];

        // Only auto-detect if exactly one bundle is configured.
        // Multiple bundles require explicit media type selection.
        // @todo Future enhancement: auto-detect media type from uploaded
        //   file's MIME type (image/* → image, video/* → video, etc.).
        if (count($values) === 1) {
          return reset($values) ?: NULL;
        }

        return NULL;
      }
    }

    return NULL;
  }

}
