<?php

declare(strict_types=1);

namespace Drupal\filepond\Element;

use Drupal\Component\Utility\Bytes;
use Drupal\Core\Render\Attribute\RenderElement;
use Drupal\Core\Render\Element\RenderElementBase;
use Drupal\Core\StringTranslation\ByteSizeMarkup;

/**
 * Provides a FilePond render element for non-form contexts.
 *
 * This is a render-only element that builds the FilePond HTML structure
 * and attaches the necessary JavaScript configuration. Use this in Views
 * areas or other non-form contexts where you don't need form value handling.
 *
 * For form contexts, use the 'filepond' form element instead.
 *
 * Properties:
 * - #id: (required) Unique element ID for JS targeting.
 * - #extensions: Space-separated allowed file extensions.
 * - #max_files: Maximum files (0 = unlimited).
 * - #max_filesize: Maximum file size (e.g., '10M' or bytes).
 * - #process_url: Upload endpoint URL.
 * - #patch_url: Chunked upload PATCH endpoint URL (without trailing slash).
 * - #revert_url: Revert/cancel endpoint URL.
 * - #config: Additional FilePond config options to merge.
 * - #columns: Number of columns for the grid layout (default 5).
 * - #max_width: CSS max-width for upload items (e.g., '200px').
 *
 * Usage:
 *
 * @code
 * $build['uploader'] = [
 *   '#type' => 'filepond_uploader',
 *   '#id' => 'my-uploader',
 *   '#extensions' => 'jpg jpeg png gif',
 *   '#max_files' => 10,
 *   '#process_url' => '/filepond/process',
 *   '#patch_url' => '/filepond/patch',
 *   '#revert_url' => '/filepond/revert',
 * ];
 * @endcode
 */
#[RenderElement('filepond_uploader')]
class FilePondUploader extends RenderElementBase {

  /**
   * Default allowed extensions.
   */
  public const DEFAULT_EXTENSIONS = 'jpg jpeg png gif';

  /**
   * {@inheritdoc}
   */
  public function getInfo(): array {
    return [
      '#pre_render' => [
        [static::class, 'preRenderFilePondUploader'],
      ],
      '#id' => NULL,
      '#extensions' => self::DEFAULT_EXTENSIONS,
      '#max_files' => 0,
      '#max_filesize' => NULL,
      '#process_url' => NULL,
      '#patch_url' => NULL,
      '#revert_url' => NULL,
      '#config' => [],
      '#attributes' => [],
      '#columns' => 5,
      '#max_width' => NULL,
    ];
  }

  /**
   * Pre-render callback to build the FilePond uploader structure.
   *
   * @param array $element
   *   The render element.
   *
   * @return array
   *   The processed render element.
   */
  public static function preRenderFilePondUploader(array $element): array {
    $element_id = $element['#id'];

    if (empty($element_id)) {
      throw new \InvalidArgumentException('FilePond uploader requires #id property.');
    }

    // Validate required URLs.
    if (empty($element['#process_url']) || empty($element['#patch_url']) || empty($element['#revert_url'])) {
      throw new \InvalidArgumentException('FilePond uploader requires #process_url, #patch_url, and #revert_url.');
    }

    // Build FilePond config.
    $extensions = $element['#extensions'] ?? self::DEFAULT_EXTENSIONS;
    $max_files = (int) ($element['#max_files'] ?? 0);

    $config = [
      'processUrl' => $element['#process_url'],
      'patchUrl' => $element['#patch_url'],
      'revertUrl' => $element['#revert_url'],
      'allowReorder' => FALSE,
      'maxFiles' => $max_files ?: NULL,
      'acceptedFileTypes' => array_values(self::extensionsToMimeTypes($extensions)),
      'extensions' => $extensions,
      'files' => [],
      'chunkUploads' => TRUE,
    ];

    // Add max file size if set.
    if (!empty($element['#max_filesize'])) {
      $max_size = $element['#max_filesize'];
      if (is_numeric($max_size)) {
        $mb = round($max_size / (1024 * 1024), 2);
      }
      else {
        $bytes = Bytes::toNumber($max_size);
        $mb = round($bytes / (1024 * 1024), 2);
      }
      $config['maxFileSize'] = $mb ? "{$mb}MB" : NULL;
    }

    // Merge any additional config.
    if (!empty($element['#config'])) {
      $config = array_merge($config, $element['#config']);
    }

    // Build wrapper attributes. Add 'filepond-element' here because render
    // elements don't wrap children with #attributes - only container renders.
    $wrapper_attributes = $element['#attributes'] ?? [];
    $wrapper_attributes['class'] = array_merge(
      $wrapper_attributes['class'] ?? [],
      ['filepond-element', 'filepond-wrapper', 'filepond-uploader']
    );

    // Add CSS variables for grid layout.
    $css_vars = [];
    $columns = (int) ($element['#columns'] ?? 5);
    if ($columns > 0) {
      // @todo make the gap configurable too.
      // Calculate width as percentage minus gap for grid spacing.
      $width_percent = round(100 / $columns, 2);
      $css_vars[] = '--filepond-item-width: calc(' . $width_percent . '% - 0.5em)';
    }
    if (!empty($element['#max_width'])) {
      $max_width = $element['#max_width'];
      if (is_numeric($max_width)) {
        $max_width .= 'px';
      }
      $css_vars[] = '--filepond-item-max-width: ' . $max_width;
    }
    if ($css_vars) {
      $existing_style = $wrapper_attributes['style'] ?? '';
      $wrapper_attributes['style'] = trim($existing_style . ' ' . implode('; ', $css_vars) . ';');
    }

    // Build the HTML structure.
    $element['wrapper'] = [
      '#type' => 'container',
      '#attributes' => $wrapper_attributes,
      // Hidden field for file IDs.
      'fids' => [
        '#type' => 'hidden',
        '#attributes' => [
          'class' => ['filepond-fids'],
        ],
      ],
      // File input that FilePond transforms.
      'input' => [
        '#type' => 'html_tag',
        '#tag' => 'input',
        '#attributes' => [
          'type' => 'file',
          'class' => ['filepond--input'],
          'id' => $element_id . '-input',
          'data-filepond-id' => $element_id,
        ],
      ],
    ];

    // Build upload requirements summary.
    $element['requirements'] = static::buildRequirementsSummary(
      $extensions,
      $max_files,
      $element['#max_filesize'] ?? NULL
    );

    // Attach libraries and settings.
    $element['#attached']['library'][] = 'filepond/filepond.element';
    $element['#attached']['drupalSettings']['filepond']['instances'][$element_id] = $config;

    return $element;
  }

  /**
   * Builds the upload requirements summary render array.
   *
   * @param string $extensions
   *   Space-separated list of allowed extensions.
   * @param int $max_files
   *   Maximum number of files (0 = unlimited).
   * @param int|string|null $max_filesize
   *   Maximum file size (e.g., '10M' or bytes), or NULL if not set.
   *
   * @return array
   *   Render array for the requirements summary.
   */
  public static function buildRequirementsSummary(string $extensions, int $max_files, int|string|null $max_filesize = NULL): array {
    $items = [];

    // File types - show MIME types instead of extensions.
    $mime_types = static::extensionsToMimeTypes($extensions);
    if (!empty($mime_types)) {
      $items[] = t('Allowed types: @types', [
        '@types' => implode(', ', $mime_types),
      ]);
    }

    // Max file size.
    if (!empty($max_filesize)) {
      if (is_numeric($max_filesize)) {
        $size_str = ByteSizeMarkup::create($max_filesize);
      }
      else {
        $bytes = Bytes::toNumber($max_filesize);
        $size_str = ByteSizeMarkup::create($bytes);
      }
      $items[] = t('@size per file', ['@size' => $size_str]);
    }

    // Max files.
    if ($max_files > 0) {
      $items[] = \Drupal::translation()->formatPlural(
        $max_files,
        '1 file',
        '@count files max'
      );
    }

    if (empty($items)) {
      return [];
    }

    return [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['filepond-requirements', 'description'],
      ],
      'list' => [
        '#theme' => 'item_list',
        '#items' => $items,
        '#attributes' => ['class' => ['filepond-requirements__list']],
      ],
    ];
  }

  /**
   * Converts file extensions to MIME types for FilePond validation.
   *
   * Uses Drupal's MIME type guesser service for accurate mapping.
   *
   * @param string $extensions
   *   Space-separated list of extensions.
   *
   * @return array
   *   Array of MIME types.
   */
  public static function extensionsToMimeTypes(string $extensions): array {
    $guesser = \Drupal::service('file.mime_type.guesser');
    $ext_array = array_filter(explode(' ', strtolower($extensions)));
    $mime_types = [];

    foreach ($ext_array as $ext) {
      $mime_type = $guesser->guessMimeType('file.' . $ext);
      if ($mime_type) {
        $mime_types[] = $mime_type;
      }
    }

    return array_unique($mime_types);
  }

}
