<?php

declare(strict_types=1);

namespace Drupal\filepond\Event;

use Drupal\Component\EventDispatcher\Event;
use Drupal\file\FileInterface;

/**
 * Event dispatched when a FilePond upload completes.
 *
 * Subscribe to this event to perform actions after a file is uploaded,
 * such as creating media entities or processing EXIF data.
 */
class FilePondUploadCompleteEvent extends Event {

  /**
   * Event name.
   */
  public const EVENT_NAME = 'filepond.upload_complete';

  /**
   * The file entity.
   */
  protected FileInterface $file;

  /**
   * The transfer ID.
   */
  protected string $transferId;

  /**
   * The original filename.
   */
  protected string $originalFilename;

  /**
   * Optional context data from the upload request.
   *
   * @var array<string, mixed>
   */
  protected array $context;

  /**
   * Result data that can be modified by subscribers.
   *
   * @var array<string, mixed>
   */
  protected array $result = [];

  /**
   * Constructs a FilePondUploadCompleteEvent.
   *
   * @param \Drupal\file\FileInterface $file
   *   The uploaded file entity.
   * @param string $transfer_id
   *   The transfer ID used during upload.
   * @param string $original_filename
   *   The original filename from the client.
   * @param array $context
   *   Optional context data (e.g., field name, entity type).
   */
  public function __construct(
    FileInterface $file,
    string $transfer_id,
    string $original_filename,
    array $context = [],
  ) {
    $this->file = $file;
    $this->transferId = $transfer_id;
    $this->originalFilename = $original_filename;
    $this->context = $context;
  }

  /**
   * Gets the uploaded file entity.
   *
   * @return \Drupal\file\FileInterface
   *   The file entity.
   */
  public function getFile(): FileInterface {
    return $this->file;
  }

  /**
   * Gets the transfer ID.
   *
   * @return string
   *   The transfer ID.
   */
  public function getTransferId(): string {
    return $this->transferId;
  }

  /**
   * Gets the original filename.
   *
   * @return string
   *   The original filename from the client.
   */
  public function getOriginalFilename(): string {
    return $this->originalFilename;
  }

  /**
   * Gets the upload context.
   *
   * @return array<string, mixed>
   *   The context data.
   */
  public function getContext(): array {
    return $this->context;
  }

  /**
   * Gets a specific context value.
   *
   * @param string $key
   *   The context key.
   * @param mixed $default
   *   Default value if key doesn't exist.
   *
   * @return mixed
   *   The context value.
   */
  public function getContextValue(string $key, mixed $default = NULL): mixed {
    return $this->context[$key] ?? $default;
  }

  /**
   * Sets a result value.
   *
   * Subscribers can use this to pass data back, such as a media entity ID.
   *
   * @param string $key
   *   The result key.
   * @param mixed $value
   *   The result value.
   *
   * @return $this
   */
  public function setResult(string $key, mixed $value): static {
    $this->result[$key] = $value;
    return $this;
  }

  /**
   * Gets a result value.
   *
   * @param string $key
   *   The result key.
   * @param mixed $default
   *   Default value if key doesn't exist.
   *
   * @return mixed
   *   The result value.
   */
  public function getResult(string $key, mixed $default = NULL): mixed {
    return $this->result[$key] ?? $default;
  }

  /**
   * Gets all result values.
   *
   * @return array<string, mixed>
   *   All result values.
   */
  public function getResults(): array {
    return $this->result;
  }

}
