<?php

declare(strict_types=1);

namespace Drupal\filepond;

use Drupal\Component\Utility\Bytes;
use Drupal\Component\Utility\Environment;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\filepond\Element\FilePond as FilePondElement;

/**
 * Provides shared form building for FilePond configuration.
 *
 * This trait provides consistent configuration form elements for FilePond
 * uploaders across different contexts (Entity Browser widget, Views area, etc).
 */
trait FilePondConfigFormTrait {

  /**
   * Gets default FilePond configuration values.
   *
   * @return array
   *   Default configuration array.
   */
  protected function getFilePondDefaults(): array {
    return [
      'media_type' => '',
      'inherit_settings' => TRUE,
      'upload_location' => 'public://[date:custom:Y]-[date:custom:m]',
      'max_filesize' => (int) (Environment::getUploadMaxSize() / pow(Bytes::KILOBYTE, 2)) . 'M',
      'extensions' => 'jpg jpeg gif png',
      'upload_prompt' => '',
      'description' => '',
      'max_files' => 0,
      'chunk_size' => 5,
      'max_parallel_uploads' => 3,
      'item_panel_aspect_ratio' => '1:1',
      'preview_fit_mode' => 'contain',
      'columns' => 5,
      'max_width' => '',
      'auto_select' => FALSE,
    ];
  }

  /**
   * Builds the FilePond configuration form elements.
   *
   * @param array $form
   *   The form array to add elements to.
   * @param array $config
   *   Current configuration values.
   * @param array $options
   *   Options for form building:
   *   - media_type_options: Array of media type options for select.
   *   - show_auto_select: Whether to show auto-select option (default FALSE).
   *
   * @return array
   *   The form array with FilePond elements added.
   */
  protected function buildFilePondConfigForm(array $form, array $config, array $options = []): array {
    $media_type_options = $options['media_type_options'] ?? [];

    // Media type selection - always required.
    // @todo Future enhancement: auto-detect media type from uploaded file's
    //   MIME type (image/* → image, video/* → video, etc.).
    $form['media_type'] = [
      '#type' => 'select',
      '#title' => new TranslatableMarkup('Media type'),
      '#required' => TRUE,
      '#default_value' => $config['media_type'] ?? '',
      '#options' => ['' => new TranslatableMarkup('- Select -')] + $media_type_options,
      '#description' => new TranslatableMarkup('The type of media entity to create from uploaded files.'),
      '#weight' => 0,
    ];

    // Inherit settings from the media type's source field.
    $form['inherit_settings'] = [
      '#type' => 'checkbox',
      '#title' => new TranslatableMarkup('Inherit settings from the media type'),
      '#description' => new TranslatableMarkup('When enabled, upload location, file size, and extensions are taken from the media type source field configuration.'),
      '#default_value' => $config['inherit_settings'] ?? TRUE,
      '#weight' => 10,
      '#attributes' => ['class' => ['filepond-inherit-settings']],
    ];

    // States to show manual config fields only when inherit_settings is off.
    $manual_states = [
      'visible' => [
        '.filepond-inherit-settings' => ['checked' => FALSE],
      ],
    ];

    $form['upload_location'] = [
      '#type' => 'textfield',
      '#title' => new TranslatableMarkup('Upload location'),
      '#default_value' => $config['upload_location'] ?? 'public://[date:custom:Y]-[date:custom:m]',
      '#description' => new TranslatableMarkup('Destination folder for uploaded files. Supports tokens like [date:custom:Y-m].'),
      '#weight' => 11,
      '#states' => $manual_states,
    ];

    // Extract numeric value from max_filesize (e.g., "10M" -> 10).
    $max_filesize = $config['max_filesize'] ?? '10M';
    preg_match('%\d+%', $max_filesize, $matches);
    $max_filesize_num = !empty($matches) ? array_shift($matches) : '10';

    $form['max_filesize'] = [
      '#type' => 'number',
      '#title' => new TranslatableMarkup('Maximum file size'),
      '#min' => 1,
      '#field_suffix' => new TranslatableMarkup('MB'),
      '#default_value' => $max_filesize_num,
      '#weight' => 12,
      '#states' => $manual_states,
    ];

    $form['extensions'] = [
      '#type' => 'textfield',
      '#title' => new TranslatableMarkup('Allowed file extensions'),
      '#description' => new TranslatableMarkup('Space-separated list of allowed extensions.'),
      '#default_value' => $config['extensions'] ?? 'jpg jpeg gif png',
      '#weight' => 13,
      '#states' => $manual_states,
    ];

    $form['upload_prompt'] = [
      '#type' => 'textfield',
      '#title' => new TranslatableMarkup('Upload prompt'),
      '#description' => new TranslatableMarkup('Text shown in the upload area prompting users to drag or click. Leave empty for default.'),
      '#default_value' => $config['upload_prompt'] ?? '',
      '#weight' => 20,
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => new TranslatableMarkup('Description'),
      '#description' => new TranslatableMarkup('Help text displayed above the upload area. Supports basic HTML.'),
      '#default_value' => $config['description'] ?? '',
      '#rows' => 2,
      '#weight' => 21,
    ];

    $form['max_files'] = [
      '#type' => 'number',
      '#title' => new TranslatableMarkup('Maximum number of files per upload'),
      '#min' => 0,
      '#default_value' => $config['max_files'] ?? 0,
      '#description' => new TranslatableMarkup('0 for unlimited.'),
      '#weight' => 22,
    ];

    $form['chunk_size'] = [
      '#type' => 'number',
      '#title' => new TranslatableMarkup('Chunk size'),
      '#min' => 1,
      '#max' => 50,
      '#field_suffix' => new TranslatableMarkup('MB'),
      '#default_value' => $config['chunk_size'] ?? 5,
      '#description' => new TranslatableMarkup('Size of each upload chunk for large files.'),
      '#weight' => 23,
    ];

    $form['max_parallel_uploads'] = [
      '#type' => 'number',
      '#title' => new TranslatableMarkup('Max parallel uploads'),
      '#min' => 1,
      '#max' => 10,
      '#default_value' => $config['max_parallel_uploads'] ?? 3,
      '#description' => new TranslatableMarkup('Number of files to upload simultaneously.'),
      '#weight' => 24,
    ];

    // Display options fieldset.
    $form['display'] = [
      '#type' => 'details',
      '#title' => new TranslatableMarkup('Display options'),
      '#weight' => 25,
      '#open' => TRUE,
    ];

    $form['display']['item_panel_aspect_ratio'] = [
      '#type' => 'textfield',
      '#title' => new TranslatableMarkup('Panel aspect ratio'),
      '#default_value' => $config['item_panel_aspect_ratio'] ?? '1:1',
      '#description' => new TranslatableMarkup('Aspect ratio for file panels (e.g., "16:9", "4:3", "1:1").'),
      '#size' => 10,
      '#element_validate' => [[FilePondElement::class, 'validateAspectRatio']],
    ];

    $form['display']['preview_fit_mode'] = [
      '#type' => 'select',
      '#title' => new TranslatableMarkup('Preview fit mode'),
      '#options' => [
        'contain' => new TranslatableMarkup('Contain (letterbox)'),
        'cover' => new TranslatableMarkup('Cover (crop to fill)'),
      ],
      '#default_value' => $config['preview_fit_mode'] ?? 'contain',
      '#description' => new TranslatableMarkup('How image previews fit within the panel.'),
    ];

    $form['display']['columns'] = [
      '#type' => 'number',
      '#title' => new TranslatableMarkup('Columns'),
      '#default_value' => $config['columns'] ?? 5,
      '#description' => new TranslatableMarkup('Number of columns for the upload grid.'),
      '#min' => 1,
      '#max' => 12,
    ];

    $form['display']['max_width'] = [
      '#type' => 'textfield',
      '#title' => new TranslatableMarkup('Max width'),
      '#default_value' => $config['max_width'] ?? '',
      '#description' => new TranslatableMarkup('Optional max-width for items (e.g., "200px").'),
      '#size' => 10,
    ];

    // Auto-select is primarily useful in Entity Browser/Views contexts.
    $show_auto_select = $options['show_auto_select'] ?? FALSE;
    if ($show_auto_select) {
      $form['auto_select'] = [
        '#type' => 'checkbox',
        '#title' => new TranslatableMarkup('Auto-select uploaded items'),
        '#description' => new TranslatableMarkup('When used in an Entity Browser, automatically select newly uploaded media items.'),
        '#default_value' => $config['auto_select'] ?? FALSE,
        '#weight' => 30,
      ];
    }

    return $form;
  }

  /**
   * Validates the FilePond configuration form.
   *
   * @param array $values
   *   The form values to validate.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $error_prefix
   *   Prefix for error element names.
   */
  protected function validateFilePondConfigForm(array $values, FormStateInterface $form_state, string $error_prefix = ''): void {
    if (!empty($values['extensions'])) {
      $extensions = explode(' ', $values['extensions']);
      foreach ($extensions as $extension) {
        if (preg_match('%^\w*$%', $extension) !== 1) {
          $form_state->setErrorByName($error_prefix . 'extensions', new TranslatableMarkup('Invalid extension list format.'));
          break;
        }
      }
    }
  }

  /**
   * Processes FilePond config values after form submission.
   *
   * @param array $values
   *   The submitted values.
   *
   * @return array
   *   The processed values with nested display values flattened.
   */
  protected function processFilePondConfigValues(array $values): array {
    // Append 'M' to max_filesize if it's just a number.
    if (!empty($values['max_filesize']) && is_numeric($values['max_filesize'])) {
      $values['max_filesize'] = $values['max_filesize'] . 'M';
    }

    // Flatten display fieldset values to root level.
    if (isset($values['display']) && is_array($values['display'])) {
      foreach ($values['display'] as $key => $value) {
        $values[$key] = $value;
      }
      unset($values['display']);
    }

    // Append 'px' to max_width if it's just a number.
    if (!empty($values['max_width']) && is_numeric($values['max_width'])) {
      $values['max_width'] = $values['max_width'] . 'px';
    }

    return $values;
  }

}
