<?php

declare(strict_types=1);

namespace Drupal\filepond\Form;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\filepond\Element\FilePond;
use Drupal\media\MediaSourceManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for FilePond module settings.
 *
 * Settings include:
 * - Enable/disable Media Library widget replacement
 * - Default upload settings (max files, extensions, file size, etc.)
 * - UI customization (aspect ratio, upload prompt)
 */
class FilePondSettingsForm extends ConfigFormBase {

  /**
   * The media source plugin manager.
   */
  protected MediaSourceManager $mediaSourcePluginManager;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->mediaSourcePluginManager = $container->get('plugin.manager.media.source');
    $instance->moduleHandler = $container->get('module_handler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'filepond_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['filepond.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('filepond.settings');

    $form['use_cdn'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Load libraries from CDN'),
      '#description' => $this->t('Load FilePond JavaScript and CSS from jsDelivr CDN instead of local library files. Useful for quick setup or when library installation is not possible. <strong>Note:</strong> Using a CDN means requests are made to a third-party server (jsDelivr).'),
      '#default_value' => $config->get('use_cdn'),
    ];

    // Check if local libraries are installed.
    $missing_libraries = $this->checkMissingLibraries();
    if (!empty($missing_libraries)) {
      $form['library_warning'] = [
        '#type' => 'container',
        '#states' => [
          'visible' => [
            ':input[name="use_cdn"]' => ['checked' => FALSE],
          ],
        ],
        'message' => [
          '#theme' => 'status_messages',
          '#message_list' => [
            'warning' => [
              $this->t('Local libraries are not installed: @missing. Either enable CDN above or <a href=":readme" target="_blank">install the libraries</a>.', [
                '@missing' => implode(', ', $missing_libraries),
                ':readme' => 'https://www.drupal.org/project/filepond',
              ]),
            ],
          ],
        ],
      ];
    }

    $form['enable_media_library_widget'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable FilePond widget in Media Library'),
      '#description' => $this->t('When enabled, the Media Library upload form will use FilePond instead of the default file upload widget. This applies to image, video, audio, and file media types.'),
      '#default_value' => $config->get('enable_media_library_widget'),
    ];

    $defaults = $config->get('defaults') ?? [];

    $form['defaults'] = [
      '#type' => 'details',
      '#title' => $this->t('Default element settings'),
      '#description' => $this->t('These defaults apply when the property is not set on the element. Individual elements can override these values.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form['defaults']['max_files'] = [
      '#type' => 'number',
      '#title' => $this->t('Max files'),
      '#description' => $this->t('Maximum number of files per upload. Set to 0 for unlimited.'),
      '#default_value' => $defaults['max_files'] ?? 10,
      '#min' => 0,
      '#max' => 100,
    ];

    $form['defaults']['extensions'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Allowed extensions'),
      '#description' => $this->t('Space-separated list of allowed file extensions.'),
      '#default_value' => $defaults['extensions'] ?? 'jpg jpeg gif png webp',
    ];

    $form['defaults']['max_filesize'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Max file size'),
      '#description' => $this->t('Maximum file size per file (e.g., "10M", "500K"). Leave empty to use PHP upload_max_filesize setting.'),
      '#default_value' => $defaults['max_filesize'] ?? '',
      '#size' => 10,
    ];

    $form['defaults']['chunk_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Chunk size (MB)'),
      '#description' => $this->t('Chunk size for chunked uploads in megabytes. Larger chunks = fewer requests but more memory. 5MB is recommended.'),
      '#default_value' => $defaults['chunk_size'] ?? 5,
      '#min' => 1,
      '#max' => 50,
      '#field_suffix' => 'MB',
    ];

    $form['defaults']['max_parallel_uploads'] = [
      '#type' => 'number',
      '#title' => $this->t('Max parallel uploads'),
      '#description' => $this->t('Number of files to upload simultaneously. Higher values speed up bulk uploads but use more bandwidth/memory.'),
      '#default_value' => $defaults['max_parallel_uploads'] ?? 3,
      '#min' => 1,
      '#max' => 10,
    ];

    $form['defaults']['upload_location'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Upload location'),
      '#description' => $this->t('Destination URI for uploaded files. Supports tokens like [current-user:uid]. Example: public://filepond-uploads/[current-user:uid]'),
      '#default_value' => $defaults['upload_location'] ?? 'public://filepond-uploads',
    ];

    // Show available tokens if token module is enabled.
    if ($this->moduleHandler->moduleExists('token')) {
      $form['defaults']['token_help'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['current-user'],
      ];
    }

    $form['defaults']['upload_prompt'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Upload prompt'),
      '#description' => $this->t('Text shown in the drop area. Use <code><span class="filepond--label-action">Click here</span></code> for clickable styling.'),
      '#default_value' => $defaults['upload_prompt'] ?? 'Drag & drop files or <span class="filepond--label-action">Browse</span>',
    ];

    $form['defaults']['item_panel_aspect_ratio'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Item panel aspect ratio'),
      '#description' => $this->t('Aspect ratio for file item panels in ratio format (e.g., "16:9", "4:3", "1:1"). Can be overridden per element.'),
      '#default_value' => $defaults['item_panel_aspect_ratio'] ?? '1:1',
      '#size' => 10,
      '#element_validate' => [[FilePond::class, 'validateAspectRatio']],
    ];

    $form['defaults']['preview_fit_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Preview fit mode'),
      '#description' => $this->t('How images are fitted within the preview panel.'),
      '#options' => [
        'contain' => $this->t('Contain (show entire image)'),
        'cover' => $this->t('Cover (fill panel, may crop)'),
      ],
      '#default_value' => $defaults['preview_fit_mode'] ?? 'contain',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $defaults = $form_state->getValue('defaults');

    $this->config('filepond.settings')
      ->set('use_cdn', (bool) $form_state->getValue('use_cdn'))
      ->set('enable_media_library_widget', $form_state->getValue('enable_media_library_widget'))
      ->set('defaults', [
        'max_files' => (int) $defaults['max_files'],
        'extensions' => $defaults['extensions'],
        'max_filesize' => $defaults['max_filesize'],
        'chunk_size' => (int) $defaults['chunk_size'],
        'upload_location' => $defaults['upload_location'],
        'upload_prompt' => $defaults['upload_prompt'],
        'max_parallel_uploads' => (int) $defaults['max_parallel_uploads'],
        'item_panel_aspect_ratio' => $defaults['item_panel_aspect_ratio'],
        'preview_fit_mode' => $defaults['preview_fit_mode'],
      ])
      ->save();

    // Clear media source plugin cache so the form override takes effect.
    $this->mediaSourcePluginManager->clearCachedDefinitions();

    // Clear library discovery cache so CDN setting change takes effect.
    // Uses cache tag invalidation to avoid deprecated service/method APIs.
    Cache::invalidateTags(['library_info']);

    parent::submitForm($form, $form_state);
  }

  /**
   * Checks which FilePond libraries are missing.
   *
   * @return array
   *   List of missing library names.
   */
  protected function checkMissingLibraries(): array {
    $libraries = [
      'filepond' => 'FilePond',
      'filepond-plugin-file-validate-type' => 'File Validate Type',
      'filepond-plugin-file-validate-size' => 'File Validate Size',
      'filepond-plugin-image-preview' => 'Image Preview',
      'filepond-plugin-file-poster' => 'File Poster',
      'filepond-plugin-image-crop' => 'Image Crop',
    ];

    $missing = [];
    foreach ($libraries as $library_dir => $library_name) {
      $path = DRUPAL_ROOT . '/libraries/' . $library_dir . '/dist';
      if (!is_dir($path)) {
        $missing[] = $library_name;
      }
    }

    return $missing;
  }

}
