<?php

declare(strict_types=1);

namespace Drupal\filepond;

use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Utility\Bytes;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Utility\Token;
use Drupal\filepond\Element\FilePond;
use Drupal\filepond\Element\FilePondUploader;
use Drupal\views\Views;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Resolves upload settings from various configuration sources.
 *
 * This service consolidates the common pattern of extracting upload
 * configuration (extensions, max size, destination) from different sources.
 * Each source (State API, field definitions, media types, etc.) stores
 * settings differently, but they all need to be converted to the same
 * standardized format for FilePondUploadHandler.
 */
class UploadSettingsResolver implements UploadSettingsResolverInterface {

  /**
   * Default file extensions if none specified.
   */
  protected const DEFAULT_EXTENSIONS = 'png gif jpg jpeg';

  /**
   * Constructs an UploadSettingsResolver.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected Token $token,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function resolveFromState(string $formId, string $elementName): ?UploadOptions {
    $config = FilePond::getConfig($formId, $elementName);
    if ($config === NULL) {
      return NULL;
    }

    // Extract extensions.
    $extensionsString = $config['extensions'] ?? self::DEFAULT_EXTENSIONS;
    $extensions = $this->parseExtensions($extensionsString);

    // Convert max filesize to bytes.
    $maxSize = $this->parseMaxSize($config['max_filesize'] ?? NULL);

    // Replace tokens in upload location at request time.
    $destination = $config['upload_location'] ?? 'public://';
    $destination = $this->token->replace($destination, [], ['clear' => TRUE]);

    return new UploadOptions(
      allowedExtensions: $extensions,
      allowedMimeTypes: $this->extensionsToMimeTypes($extensions),
      destination: $destination,
      maxSize: $maxSize,
      context: $config['context'] ?? [],
    );
  }

  /**
   * {@inheritdoc}
   */
  public function resolveFromField(string $entityType, string $bundle, string $fieldName): UploadOptions {
    // Validate entity type exists.
    if (!$this->entityTypeManager->hasDefinition($entityType)) {
      throw new NotFoundHttpException('Entity type not found');
    }

    $fieldDefinitions = $this->entityFieldManager->getFieldDefinitions($entityType, $bundle);

    if (!isset($fieldDefinitions[$fieldName])) {
      throw new NotFoundHttpException('Field not found');
    }

    $fieldDefinition = $fieldDefinitions[$fieldName];
    $settings = $fieldDefinition->getSettings();
    $storageSettings = $fieldDefinition->getFieldStorageDefinition()->getSettings();

    return $this->buildOptionsFromFieldSettings($settings, $storageSettings, [
      'entity_type' => $entityType,
      'bundle' => $bundle,
      'field_name' => $fieldName,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function resolveFromMediaType(string $mediaTypeId): UploadOptions {
    /** @var \Drupal\media\MediaTypeInterface|null $mediaType */
    $mediaType = $this->entityTypeManager->getStorage('media_type')->load($mediaTypeId);
    if (!$mediaType) {
      throw new NotFoundHttpException('Media type not found');
    }

    $source = $mediaType->getSource();
    $sourceField = $source->getSourceFieldDefinition($mediaType);
    if (!$sourceField) {
      throw new NotFoundHttpException('Media type has no source field');
    }

    $settings = $sourceField->getSettings();
    $storageSettings = $sourceField->getFieldStorageDefinition()->getSettings();

    return $this->buildOptionsFromFieldSettings($settings, $storageSettings, [
      'media_type' => $mediaTypeId,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function resolveFromWidget(string $entityBrowserId, string $widgetUuid): UploadOptions {
    /** @var \Drupal\entity_browser\EntityBrowserInterface|null $entityBrowser */
    $entityBrowser = $this->entityTypeManager
      ->getStorage('entity_browser')
      ->load($entityBrowserId);

    if (!$entityBrowser) {
      throw new NotFoundHttpException('Entity browser not found');
    }

    try {
      $widget = $entityBrowser->getWidgets()->get($widgetUuid);
    }
    catch (PluginNotFoundException $e) {
      throw new NotFoundHttpException('Widget not found');
    }
    if (!$widget) {
      throw new NotFoundHttpException('Widget not found');
    }

    if ($widget->getPluginId() !== 'filepond_media') {
      throw new AccessDeniedHttpException('Invalid widget type');
    }

    $config = $widget->getConfiguration();
    $widgetSettings = $config['settings'] ?? [];

    $mediaTypeId = $widgetSettings['media_type'] ?? '';
    if (empty($mediaTypeId)) {
      throw new AccessDeniedHttpException('Media type not configured');
    }

    /** @var \Drupal\media\MediaTypeInterface|null $mediaType */
    $mediaType = $this->entityTypeManager
      ->getStorage('media_type')
      ->load($mediaTypeId);

    if (!$mediaType) {
      throw new AccessDeniedHttpException('Media type not found');
    }

    $source = $mediaType->getSource();
    $sourceField = $source->getSourceFieldDefinition($mediaType);
    if (!$sourceField) {
      throw new AccessDeniedHttpException('Source field not found');
    }

    // Build options from field settings (if inherit) or widget config.
    if (!empty($widgetSettings['inherit_settings'])) {
      $fieldSettings = $sourceField->getSettings();
      $storageSettings = $sourceField->getFieldStorageDefinition()->getSettings();

      $extensionsString = $fieldSettings['file_extensions'] ?? self::DEFAULT_EXTENSIONS;
      $maxFilesize = $fieldSettings['max_filesize'] ?? NULL;

      $uriScheme = $storageSettings['uri_scheme'] ?? 'public';
      $fileDirectory = $this->token->replace($fieldSettings['file_directory'] ?? '');
      $destination = $uriScheme . '://' . $fileDirectory;
    }
    else {
      $extensionsString = $widgetSettings['extensions'] ?? self::DEFAULT_EXTENSIONS;
      $maxFilesize = $widgetSettings['max_filesize'] ?? NULL;
      $destination = $this->token->replace($widgetSettings['upload_location'] ?? 'public://');
    }

    $extensions = $this->parseExtensions($extensionsString);
    $maxSize = $this->parseMaxSize($maxFilesize);

    return new UploadOptions(
      allowedExtensions: $extensions,
      allowedMimeTypes: $this->extensionsToMimeTypes($extensions),
      destination: $destination,
      maxSize: $maxSize,
      context: [
        'entity_browser' => $entityBrowserId,
        'widget_uuid' => $widgetUuid,
        'media_type' => $mediaTypeId,
      ],
    );
  }

  /**
   * {@inheritdoc}
   */
  public function resolveFromViewsArea(string $viewId, string $displayId): UploadOptions {
    $view = Views::getView($viewId);
    if (!$view) {
      throw new NotFoundHttpException('View not found');
    }

    if (!$view->setDisplay($displayId)) {
      throw new NotFoundHttpException('Display not found');
    }

    // Find the FilePond area plugin in this display.
    $areaOptions = $this->findViewsAreaPluginOptions($view, $displayId);
    if (!$areaOptions) {
      throw new NotFoundHttpException('FilePond area plugin not found in view');
    }

    // Get media type from area options.
    $mediaTypeId = $areaOptions['media_type'] ?? NULL;
    if (empty($mediaTypeId)) {
      throw new NotFoundHttpException('Media type not configured in views area');
    }

    /** @var \Drupal\media\MediaTypeInterface|null $mediaType */
    $mediaType = $this->entityTypeManager->getStorage('media_type')->load($mediaTypeId);
    if (!$mediaType) {
      throw new NotFoundHttpException('Media type not found');
    }

    $source = $mediaType->getSource();
    $sourceField = $source->getSourceFieldDefinition($mediaType);
    if (!$sourceField) {
      throw new NotFoundHttpException('Media type has no source field');
    }

    $settings = $sourceField->getSettings();
    $storageSettings = $sourceField->getFieldStorageDefinition()->getSettings();

    // Build options with views_area context flag.
    return $this->buildOptionsFromFieldSettings($settings, $storageSettings, [
      'view_id' => $viewId,
      'display_id' => $displayId,
      'media_type' => $mediaTypeId,
      'views_area' => TRUE,
    ]);
  }

  /**
   * Builds upload options from field settings.
   *
   * This is the core method that converts field/storage settings to
   * standardized UploadOptions.
   *
   * @param array $settings
   *   Field instance settings.
   * @param array $storageSettings
   *   Field storage settings.
   * @param array $context
   *   Additional context for event subscribers.
   *
   * @return \Drupal\filepond\UploadOptions
   *   Upload options.
   */
  protected function buildOptionsFromFieldSettings(array $settings, array $storageSettings, array $context): UploadOptions {
    // Build upload location from field settings.
    $fileDirectory = $this->token->replace($settings['file_directory'] ?? '');
    $uriScheme = $storageSettings['uri_scheme'] ?? 'public';
    $destination = $uriScheme . '://' . $fileDirectory;

    // Get extensions.
    $extensionsString = $settings['file_extensions'] ?? self::DEFAULT_EXTENSIONS;
    $extensions = $this->parseExtensions($extensionsString);

    // Get max size.
    $maxSize = $this->parseMaxSize($settings['max_filesize'] ?? NULL);

    return new UploadOptions(
      allowedExtensions: $extensions,
      allowedMimeTypes: $this->extensionsToMimeTypes($extensions),
      destination: $destination,
      maxSize: $maxSize,
      context: $context,
    );
  }

  /**
   * Parses an extensions string into an array.
   *
   * @param string $extensionsString
   *   Space-separated extensions (e.g., 'jpg jpeg png gif').
   *
   * @return array
   *   Array of lowercase extensions.
   */
  protected function parseExtensions(string $extensionsString): array {
    return array_filter(
      array_map('trim', explode(' ', strtolower($extensionsString)))
    );
  }

  /**
   * Parses max filesize to bytes.
   *
   * @param string|int|null $maxFilesize
   *   Max filesize as string (e.g., '10M') or bytes.
   *
   * @return int|null
   *   Max size in bytes, or NULL if not set.
   */
  protected function parseMaxSize(string|int|null $maxFilesize): ?int {
    if (empty($maxFilesize)) {
      return NULL;
    }

    if (is_numeric($maxFilesize)) {
      return (int) $maxFilesize;
    }

    // Bytes::toNumber returns float, cast to int.
    return (int) Bytes::toNumber($maxFilesize);
  }

  /**
   * Converts file extensions to MIME types.
   *
   * @param array $extensions
   *   Array of file extensions.
   *
   * @return array
   *   Array of MIME types.
   */
  protected function extensionsToMimeTypes(array $extensions): array {
    return FilePondUploader::extensionsToMimeTypes(implode(' ', $extensions));
  }

  /**
   * Finds the FilePond area plugin options in a view display.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view.
   * @param string $displayId
   *   The display ID.
   *
   * @return array|null
   *   The area plugin options, or NULL if not found.
   */
  protected function findViewsAreaPluginOptions($view, string $displayId): ?array {
    $display = $view->displayHandlers->get($displayId);
    if (!$display) {
      return NULL;
    }

    // Check all area types.
    foreach (['header', 'footer', 'empty'] as $areaType) {
      $areas = $display->getOption($areaType);
      if (empty($areas)) {
        continue;
      }

      foreach ($areas as $area) {
        if (($area['plugin_id'] ?? '') === 'filepond_upload') {
          return $area;
        }
      }
    }

    return NULL;
  }

}
