<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\Role;
use Drupal\user\RoleInterface;

/**
 * Tests the FilePond form element.
 *
 * @group filepond
 */
class FilePondElementTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'file',
    'user',
    'filepond',
    'filepond_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installConfig(['filepond']);

    // Grant anonymous user permission to upload.
    $role = Role::create([
      'id' => RoleInterface::ANONYMOUS_ID,
      'label' => 'Anonymous',
    ]);
    $role->grantPermission('filepond upload files');
    $role->save();
  }

  /**
   * Tests that the FilePond element renders correctly.
   */
  public function testFilePondElementRendering(): void {
    $this->container->get('router.builder')->rebuild();

    $form = \Drupal::formBuilder()->getForm('\Drupal\filepond_test\Form\FilePondTestForm');
    $this->render($form);

    $xpath_base = "//div[contains(@class, 'js-form-type-filepond')]";

    // Label should be present.
    $this->assertNotEmpty(
      $this->xpath("$xpath_base/label[contains(text(), 'FilePond element')]"),
      'FilePond element label should be present.'
    );

    // FilePond wrapper should be present (class from template).
    $this->assertNotEmpty(
      $this->xpath("$xpath_base//div[contains(@class, 'filepond-wrapper')]"),
      'FilePond wrapper element should be present.'
    );

    // Hidden file input should be present.
    $this->assertNotEmpty(
      $this->xpath("$xpath_base//input[@type='file']"),
      'File input should be present.'
    );

    // Hidden fids input should be present.
    $this->assertNotEmpty(
      $this->xpath("$xpath_base//input[@type='hidden'][contains(@name, '[fids]')]"),
      'Hidden fids input should be present.'
    );
  }

  /**
   * Tests element properties are passed correctly.
   */
  public function testFilePondElementProperties(): void {
    $this->container->get('router.builder')->rebuild();

    $form = \Drupal::formBuilder()->getForm('\Drupal\filepond_test\Form\FilePondTestForm');

    // Check element properties are set.
    $element = $form['filepond'];
    $this->assertEquals('jpg png gif', $element['#extensions']);
    $this->assertEquals('10M', $element['#max_filesize']);
    $this->assertEquals(5, $element['#max_files']);
    $this->assertEquals('public://filepond-test', $element['#upload_location']);
    $this->assertTrue($element['#required']);
  }

  /**
   * Tests that element info has defaults defined.
   */
  public function testFilePondElementDefaults(): void {
    /** @var \Drupal\Core\Render\ElementInfoManagerInterface $element_info */
    $element_info = $this->container->get('plugin.manager.element_info');

    $info = $element_info->getInfo('filepond');

    // Element info should have default properties defined.
    $this->assertArrayHasKey('#extensions', $info);
    $this->assertArrayHasKey('#max_filesize', $info);
    $this->assertArrayHasKey('#max_files', $info);
    $this->assertArrayHasKey('#upload_location', $info);
    $this->assertArrayHasKey('#config', $info);
    $this->assertIsArray($info['#config']);
  }

}
