/**
 * @file
 * FilePond Drupal integration helpers.
 *
 * Provides common functionality for FilePond integration with Drupal:
 * - CSRF token handling
 * - Server configuration builder
 * - Common initialization patterns
 */

/* global FilePond, FilePondPluginFileValidateType, FilePondPluginFileValidateSize, FilePondPluginImagePreview, FilePondPluginFilePoster, FilePondPluginImageCrop */

(function (Drupal, drupalSettings) {
  /**
   * FilePond Drupal integration namespace.
   *
   * @namespace
   */
  Drupal.filepond = Drupal.filepond || {};

  /**
   * Cached CSRF token.
   *
   * @type {string|null}
   */
  Drupal.filepond.csrfToken = null;

  /**
   * Map of transfer ID to media ID for chunked uploads.
   *
   * When uploads create media entities, the mediaId is returned in the
   * final PATCH response. This map allows other scripts (like filepond_views.js)
   * to look up the media ID by transfer ID.
   *
   * @type {Object}
   */
  Drupal.filepond.transferToMediaId = {};

  /**
   * Map of transfer ID to file ID for chunked uploads.
   *
   * Chunked uploads return a transfer ID initially, then the final PATCH
   * response returns the file ID. This global map is populated by element.js
   * and used for testing purposes.
   *
   * @type {Object}
   */
  Drupal.filepond.transferToFileId = {};

  /**
   * Fetches the CSRF token from Drupal.
   *
   * @return {Promise<string>}
   *   Promise resolving to the CSRF token.
   */
  Drupal.filepond.fetchCsrfToken = function () {
    // Return cached token if available.
    if (Drupal.filepond.csrfToken) {
      return Promise.resolve(Drupal.filepond.csrfToken);
    }

    // Use Drupal.url() to respect base path (e.g., when Drupal is in a subdir).
    return fetch(Drupal.url('session/token'))
      .then(function (response) {
        return response.text();
      })
      .then(function (token) {
        Drupal.filepond.csrfToken = token;
        return token;
      });
  };

  /**
   * Registers standard FilePond plugins.
   *
   * Call this before creating FilePond instances.
   * Safely checks if plugins are available before registering.
   */
  Drupal.filepond.registerPlugins = function () {
    if (
      typeof FilePondPluginFileValidateType !== 'undefined' &&
      !FilePond.find('filepond-plugin-file-validate-type')
    ) {
      FilePond.registerPlugin(FilePondPluginFileValidateType);
    }
    if (
      typeof FilePondPluginFileValidateSize !== 'undefined' &&
      !FilePond.find('filepond-plugin-file-validate-size')
    ) {
      FilePond.registerPlugin(FilePondPluginFileValidateSize);
    }
    if (
      typeof FilePondPluginImagePreview !== 'undefined' &&
      !FilePond.find('filepond-plugin-image-preview')
    ) {
      FilePond.registerPlugin(FilePondPluginImagePreview);
    }
    if (
      typeof FilePondPluginFilePoster !== 'undefined' &&
      !FilePond.find('filepond-plugin-file-poster')
    ) {
      FilePond.registerPlugin(FilePondPluginFilePoster);
    }
    if (
      typeof FilePondPluginImageCrop !== 'undefined' &&
      !FilePond.find('filepond-plugin-image-crop')
    ) {
      FilePond.registerPlugin(FilePondPluginImageCrop);
    }
  };

  /**
   * Builds server configuration with CSRF headers.
   *
   * @param {Object} settings
   *   Settings object with endpoint URLs.
   * @param {string} settings.processUrl
   *   The upload process endpoint.
   * @param {string} settings.revertUrl
   *   The revert/cancel endpoint.
   * @param {string} settings.patchUrl
   *   The chunked upload patch endpoint.
   * @param {string} csrfToken
   *   The CSRF token.
   *
   * @return {Object}
   *   FilePond server configuration object.
   */
  Drupal.filepond.buildServerConfig = function (settings, csrfToken) {
    const headers = {};
    if (csrfToken) {
      headers['X-CSRF-Token'] = csrfToken;
    }

    return {
      process: {
        url: settings.processUrl,
        headers,
      },
      revert: {
        url: settings.revertUrl,
        headers,
      },
      patch: {
        url: `${settings.patchUrl}/`,
        headers,
      },
    };
  };

  /**
   * Gets default FilePond labels (English).
   *
   * @return {Object}
   *   Object with label properties.
   */
  Drupal.filepond.getDefaultLabels = function () {
    return {
      labelIdle:
        'Drag & Drop your files or <span class="filepond--label-action">Browse</span>',
      labelFileProcessing: 'Uploading...',
      labelFileProcessingComplete: 'Upload complete',
      labelFileProcessingError: 'Upload error',
      labelTapToCancel: 'tap to cancel',
      labelTapToRetry: 'tap to retry',
    };
  };

  /**
   * Creates a FilePond instance with Drupal integration.
   *
   * Automatically handles maxFiles using beforeAddFile callback instead of
   * FilePond's native maxFiles (which rejects entire batches).
   *
   * @param {HTMLElement} element
   *   The input element to enhance.
   * @param {Object} options
   *   FilePond options. Required properties:
   *   - processUrl: Upload endpoint.
   *   - revertUrl: Revert endpoint.
   *   - patchUrl: Chunked upload endpoint.
   *
   * @return {Promise<Object>}
   *   Promise resolving to the FilePond instance.
   */
  Drupal.filepond.create = function (element, options) {
    return Drupal.filepond.fetchCsrfToken().then(function (csrfToken) {
      // Register plugins if not already done.
      Drupal.filepond.registerPlugins();

      // Build server config with CSRF token.
      const serverConfig = Drupal.filepond.buildServerConfig(
        {
          processUrl: options.processUrl,
          revertUrl: options.revertUrl,
          patchUrl: options.patchUrl,
        },
        csrfToken,
      );

      // Merge default labels with provided options.
      const labels = Drupal.filepond.getDefaultLabels();

      // Extract maxFiles before creating - we'll handle it ourselves.
      const maxFiles = parseInt(options.maxFiles, 10) || 0;

      // Build final config.
      const config = {
        // Defaults.
        credits: false,
        allowMultiple: true,
        maxParallelUploads: 3,
        chunkUploads: true,
        chunkForce: false,
        imagePreviewMaxFileSize: '7MB',
        ...labels,
        ...options,
        // Server config always uses our built version.
        server: serverConfig,
        // Don't use native maxFiles - we handle it via setMaxFiles().
        maxFiles: null,
      };

      // Create the instance.
      const pond = FilePond.create(element, config);

      // Automatically configure max files limit if specified.
      // This uses our custom beforeAddFile approach instead of native maxFiles.
      if (maxFiles > 0) {
        Drupal.filepond.setMaxFiles(pond, maxFiles);
      }

      return pond;
    });
  };

  /**
   * Gets or creates a message wrapper for the given context.
   *
   * Handles message placement consistently across different FilePond contexts:
   *
   * 1. Inside a View: Creates wrapper OUTSIDE the view (as preceding sibling).
   *    This ensures messages persist through AJAX view refreshes.
   *
   * 2. Not in a View: Creates wrapper INSIDE the FilePond wrapper.
   *    Messages appear directly above the FilePond element.
   *
   * 3. Fallback: Form-item wrapper for non-FilePond contexts.
   *
   * @param {HTMLElement} contextElement
   *   An element within the context (typically pond.element or uploader wrapper).
   *
   * @return {HTMLElement|null}
   *   The message wrapper element, or null for page-level messages.
   */
  Drupal.filepond.getMessageWrapper = function (contextElement) {
    if (!contextElement) {
      return null;
    }

    // Check if we're inside a view (messages would be lost on view refresh).
    // If so, create a message container OUTSIDE the view as a sibling.
    const viewElement = contextElement.closest('[class*="js-view-dom-id-"]');
    if (viewElement) {
      // Look for existing message wrapper before the view.
      let wrapper = viewElement.previousElementSibling;
      if (!wrapper || !wrapper.hasAttribute('data-filepond-messages')) {
        wrapper = document.createElement('div');
        wrapper.setAttribute('data-filepond-messages', '');
        wrapper.setAttribute('data-drupal-messages', '');
        wrapper.className = 'filepond-messages';
        viewElement.parentNode.insertBefore(wrapper, viewElement);
      }
      return wrapper;
    }

    // Not in a view - use FilePond wrapper.
    let pondWrapper = null;

    // Try finding wrapper by looking UP from context (context is inside wrapper).
    const pondRoot = contextElement.closest('.filepond--root');
    if (pondRoot) {
      pondWrapper = pondRoot.closest('.filepond-wrapper');
    }

    // If not found, try looking UP from context directly.
    if (!pondWrapper) {
      pondWrapper = contextElement.closest('.filepond-wrapper');
    }

    // If still not found, try looking DOWN (context contains the wrapper).
    if (!pondWrapper) {
      pondWrapper = contextElement.querySelector('.filepond-wrapper');
    }

    if (pondWrapper) {
      let wrapper = pondWrapper.querySelector('[data-filepond-messages]');
      if (!wrapper) {
        wrapper = document.createElement('div');
        wrapper.setAttribute('data-filepond-messages', '');
        wrapper.setAttribute('data-drupal-messages', '');
        wrapper.className = 'filepond-messages';
        pondWrapper.insertBefore(wrapper, pondWrapper.firstChild);
      }
      return wrapper;
    }

    // Fallback: form-item wrapper for non-FilePond contexts.
    const formItem = contextElement.closest('.js-form-item, .form-item');
    if (formItem) {
      let formWrapper = formItem.querySelector('[data-drupal-messages]');
      if (!formWrapper) {
        formWrapper = document.createElement('div');
        formWrapper.setAttribute('data-drupal-messages', '');
        formWrapper.classList.add('filepond-messages');
        formItem.insertBefore(formWrapper, formItem.firstChild);
      }
      return formWrapper;
    }

    return null;
  };

  /**
   * Gets or creates a message wrapper for modal context.
   *
   * @param {HTMLElement} contextElement
   *   An element within the modal (typically pond.element, the FilePond root).
   *
   * @return {HTMLElement|null}
   *   The message wrapper element, or null if not in modal.
   *
   * @deprecated Use getMessageWrapper() instead.
   */
  Drupal.filepond.getModalMessageWrapper = function (contextElement) {
    return Drupal.filepond.getMessageWrapper(contextElement);
  };

  /**
   * Helper to show a message using Drupal.Message.
   *
   * Messages are displayed inside the FilePond wrapper, directly above
   * the FilePond element. Previous messages are cleared automatically.
   *
   * @param {string} message
   *   The message to display.
   * @param {Object} options
   *   Options object.
   * @param {string} options.type
   *   Message type: 'warning', 'error', 'status'. Default 'warning'.
   * @param {HTMLElement} options.context
   *   Context element to determine message placement (typically pond.element).
   *
   * @return {string|null}
   *   The message ID, or null if Drupal.Message not available.
   */
  Drupal.filepond.showMessage = function (message, options) {
    options = options || {};
    const type = options.type || 'warning';

    if (typeof Drupal.Message === 'undefined') {
      return null;
    }

    // Determine message wrapper - use widget/modal wrapper if context provided.
    let wrapper = null;
    if (options.context) {
      wrapper = Drupal.filepond.getMessageWrapper(options.context);
    }

    // Clear previous messages to prevent stacking.
    // FilePond should only show one message at a time per instance.
    if (wrapper) {
      wrapper.innerHTML = '';
    }

    const messages = wrapper
      ? new Drupal.Message(wrapper)
      : new Drupal.Message();

    const messageId = messages.add(message, { type });

    // Trigger fade-in animation after a brief delay for DOM render.
    if (messageId && wrapper) {
      setTimeout(function () {
        const messageEl = wrapper.querySelector(
          `[data-drupal-message-id="${messageId}"]`,
        );
        if (messageEl) {
          messageEl.classList.add('fade-in');
        }
      }, 10);
    }

    return messageId;
  };

  /**
   * Utility to check if FilePond is processing files.
   *
   * @param {Object} pond
   *   The FilePond instance.
   *
   * @return {boolean}
   *   TRUE if any files are processing.
   */
  Drupal.filepond.isProcessing = function (pond) {
    const files = pond.getFiles();
    return files.some(function (file) {
      const { status } = file;
      return (
        status === FilePond.FileStatus.PROCESSING ||
        status === FilePond.FileStatus.PROCESSING_QUEUED ||
        status === FilePond.FileStatus.LOADING
      );
    });
  };

  /**
   * Utility to count pending files (not yet completed).
   *
   * @param {Object} pond
   *   The FilePond instance.
   *
   * @return {number}
   *   Count of non-completed files.
   */
  Drupal.filepond.getPendingCount = function (pond) {
    const files = pond.getFiles();
    return files.filter(function (file) {
      return file.status !== FilePond.FileStatus.PROCESSING_COMPLETE;
    }).length;
  };

  /**
   * Registry of submit state validators.
   *
   * Modules can add validators that return TRUE if submit should be disabled.
   * All validators are checked - if any returns TRUE, submit is disabled.
   *
   * @type {Array}
   */
  Drupal.filepond.submitValidators = [];

  /**
   * Currently hovered FilePond instance for context-aware paste.
   *
   * @type {Object|null}
   */
  Drupal.filepond.hoveredPond = null;

  /**
   * Configures max files limit on a FilePond instance.
   *
   * Used by Entity Browser to enforce cardinality limits that may differ
   * from the field's configured maxFiles. For example, a field allowing 5
   * images may only have 1 slot remaining in the current EB context.
   *
   * This function:
   * - Installs a beforeAddFile callback that accepts up to maxFiles
   * - Sets allowMultiple based on maxFiles (false when maxFiles === 1)
   * - Adds drop listener to show message when multi-dropping on single-file limit
   *
   * NOTE: We use a custom beforeAddFile callback instead of FilePond's native
   * maxFiles option because native maxFiles rejects the ENTIRE batch when the
   * total exceeds the limit. Our approach accepts files up to the limit and
   * rejects only the excess.
   *
   * @param {Object} pond
   *   The FilePond instance.
   * @param {number} maxFiles
   *   Maximum allowed files (0 = unlimited).
   */
  Drupal.filepond.setMaxFiles = function (pond, maxFiles) {
    // Store on pond for reference.
    pond._maxFilesLimit = maxFiles;

    // Unlimited - no callback needed.
    if (maxFiles === 0) {
      return;
    }

    // Track rejected files within a batch for showing a single warning.
    let rejectedInBatch = 0;
    let batchMessageTimer = null;

    // Don't use FilePond's native maxFiles - it rejects the ENTIRE batch
    // when the total exceeds the limit. Instead, use beforeAddFile to
    // accept files up to the limit and reject the rest.
    pond.setOptions({
      allowMultiple: maxFiles !== 1,
      maxFiles: null,
      beforeAddFile() {
        // Check current file count vs limit.
        // Note: getFiles() includes the file being added (in pending state),
        // so we use > instead of >= to allow exactly maxFiles.
        const currentCount = pond.getFiles().length;

        // When currentCount exceeds maxFiles, reject.
        // Example: limit=5, adding file makes currentCount=6, 6>5=true, reject.
        if (currentCount > maxFiles) {
          // Track rejection count for batch message.
          rejectedInBatch += 1;

          // Debounce the warning message - show once after all files
          // in the batch have been processed.
          if (batchMessageTimer) {
            clearTimeout(batchMessageTimer);
          }
          batchMessageTimer = setTimeout(function () {
            if (rejectedInBatch > 0) {
              const message = Drupal.formatPlural(
                rejectedInBatch,
                'You can only upload @max files. 1 file was not added.',
                'You can only upload @max files. @count files were not added.',
                { '@max': maxFiles },
              );
              Drupal.filepond.showMessage(message, {
                type: 'warning',
                context: pond.element,
              });
              rejectedInBatch = 0;
            }
            batchMessageTimer = null;
          }, 100);

          return false;
        }

        return true;
      },
    });

    // For single-file limit, add drop listener to show message on multi-drop.
    // FilePond silently rejects all files when allowMultiple: false and
    // multiple files are dropped. We want to show a helpful message.
    if (maxFiles === 1 && !pond._hasMultiDropListener) {
      pond._hasMultiDropListener = true;
      pond.element.addEventListener('drop', function (e) {
        const files = e.dataTransfer?.files;
        if (files && files.length > 1) {
          // Show message after a brief delay for better UX.
          setTimeout(function () {
            Drupal.filepond.showMessage(
              Drupal.t('Only 1 file allowed. Please drop one file at a time.'),
              {
                type: 'warning',
                context: pond.element,
              },
            );
          }, 100);
        }
      });
    }
  };

  // Global paste listener - adds to hovered FilePond instance only.
  document.addEventListener('paste', function (e) {
    const pond = Drupal.filepond.hoveredPond;
    if (!pond) {
      return;
    }

    const { items } = e.clipboardData || e.originalEvent.clipboardData;
    let hasImage = false;

    for (let i = 0; i < items.length; i++) {
      if (items[i].type.indexOf('image') !== -1) {
        const file = items[i].getAsFile();
        if (file) {
          pond.addFile(file);
          hasImage = true;
        }
      }
    }

    if (hasImage) {
      e.preventDefault();
    }
  });

  /**
   * Registers a submit state validator.
   *
   * @param {Function} validator
   *   Function that receives (pond, form) and returns TRUE to disable submit.
   */
  Drupal.filepond.addSubmitValidator = function (validator) {
    Drupal.filepond.submitValidators.push(validator);
  };

  // Register the default validator: disable while processing.
  Drupal.filepond.addSubmitValidator(function (pond) {
    const files = pond.getFiles();
    return files.some(function (file) {
      const { status } = file;
      return (
        status === FilePond.FileStatus.PROCESSING ||
        status === FilePond.FileStatus.PROCESSING_QUEUED ||
        status === FilePond.FileStatus.LOADING
      );
    });
  });

  /**
   * Updates form submit button state based on registered validators.
   *
   * Runs all registered validators - if any returns TRUE, submit is disabled.
   *
   * @param {Object} pond
   *   The FilePond instance.
   */
  Drupal.filepondUpdateSubmitState = function (pond) {
    // Find the containing form.
    const form = pond.element.closest('form');
    if (!form) {
      return;
    }

    // Run all validators - disable if any returns true.
    const shouldDisable = Drupal.filepond.submitValidators.some(
      function (validator) {
        return validator(pond, form);
      },
    );

    // Find all submit buttons in the form.
    const submitButtons = form.querySelectorAll(
      'input[type="submit"], button[type="submit"], .form-submit',
    );

    submitButtons.forEach(function (button) {
      if (shouldDisable) {
        button.disabled = true;
        button.classList.add('filepond-disabled');
      } else {
        button.disabled = false;
        button.classList.remove('filepond-disabled');
      }
    });
  };
})(Drupal, drupalSettings);
