<?php

declare(strict_types=1);

namespace Drupal\filepond_eb_widget\Plugin\EntityBrowser\Widget;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\Core\Utility\Token;
use Drupal\entity_browser\WidgetBase;
use Drupal\file\FileInterface;
use Drupal\filepond\FilePondConfigFormTrait;
use Drupal\filepond\MediaSourceFieldTrait;
use Drupal\media\MediaTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an Entity Browser widget that uploads and creates media entities.
 *
 * Uses the FilePond form element which creates file entities during upload.
 * On submit, this widget creates media entities from those files.
 *
 * @EntityBrowserWidget(
 *   id = "filepond_media",
 *   label = @Translation("FilePond Media Upload"),
 *   description = @Translation("Upload files with FilePond and create media entities."),
 *   auto_select = TRUE
 * )
 */
class FilePondMediaWidget extends WidgetBase {

  use FilePondConfigFormTrait;
  use MediaSourceFieldTrait;

  /**
   * The current user.
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The token service.
   */
  protected Token $token;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->currentUser = $container->get('current_user');
    $instance->token = $container->get('token');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return array_merge(parent::defaultConfiguration(), $this->getFilePondDefaults());
  }

  /**
   * Gets the configured media type.
   *
   * @return \Drupal\media\MediaTypeInterface|null
   *   The media type entity or NULL if not configured/found.
   */
  protected function getMediaType(): ?MediaTypeInterface {
    if (empty($this->configuration['media_type'])) {
      return NULL;
    }
    return $this->entityTypeManager
      ->getStorage('media_type')
      ->load($this->configuration['media_type']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $media_types = filepond_get_file_based_media_types();

    // Use shared form builder from trait.
    // Extensions always come from media type, so hide that field.
    $form = $this->buildFilePondConfigForm($form, $this->configuration, [
      'media_type_options' => $media_types,
      'media_type_required' => TRUE,
      'show_extensions' => FALSE,
      'use_details' => TRUE,
    ]);

    // Add note when no media types exist.
    if (empty($media_types)) {
      $form['create']['media_type']['#disabled'] = TRUE;
      $form['create']['media_type']['#description'] = $this->t('You must @create_media_type before using this widget.', [
        '@create_media_type' => Link::createFromRoute($this->t('create a media type'), 'entity.media_type.add_form')->toString(),
      ]);
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // Get raw form values from nested structure.
    // Parent's submitConfigurationForm() won't work because our form uses
    // fieldsets (create, constraints, etc.) which don't match flat config keys.
    $values = $form_state->getValues()['table'][$this->uuid()]['form'] ?? [];

    // Flatten fieldsets and process values (append 'M' to filesize, etc.).
    $processed = $this->processFilePondConfigValues($values);

    // Update configuration with processed values.
    foreach ($processed as $key => $value) {
      if (array_key_exists($key, $this->configuration)) {
        $this->configuration[$key] = $value;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies(): array {
    $dependencies = parent::calculateDependencies();

    $media_type = $this->getMediaType();
    if ($media_type) {
      $dependencies[$media_type->getConfigDependencyKey()][] = $media_type->getConfigDependencyName();
    }
    $dependencies['module'][] = 'media';

    return $dependencies;
  }

  /**
   * {@inheritdoc}
   */
  public function getForm(array &$original_form, FormStateInterface $form_state, array $additional_widget_parameters): array {
    $form = parent::getForm($original_form, $form_state, $additional_widget_parameters);

    // Check permission - show message instead of upload widget if no access.
    if (!$this->currentUser->hasPermission('filepond upload files')) {
      $form['no_access'] = [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [$this->t("You don't have permission to upload files.")],
        ],
      ];
      return $form;
    }

    $config = $this->getConfiguration();

    // Resolve max files against field cardinality.
    $validators = $form_state->get(['entity_browser', 'validators']);
    $cardinality = $validators['cardinality']['cardinality'] ?? 0;
    $max_files = $this->resolveMaxFiles($config['settings']['max_files'] ?? 0, $cardinality);

    // Get entity browser ID from the form object for EB-specific routes.
    /** @var \Drupal\entity_browser\Form\EntityBrowserForm $form_object */
    $form_object = $form_state->getFormObject();
    $entity_browser = $form_object->getEntityBrowser();
    $entity_browser_id = $entity_browser->id();
    $widget_uuid = $this->uuid();

    // Build EB-specific upload URLs that bypass tempstore. The controller
    // will load config directly from the EB widget and media type.
    $process_url = Url::fromRoute('filepond_eb_widget.process', [
      'entity_browser' => $entity_browser_id,
      'widget_uuid' => $widget_uuid,
    ])->toString();

    // Build patch URL with placeholder for transfer ID.
    $patch_url = Url::fromRoute('filepond_eb_widget.patch', [
      'entity_browser' => $entity_browser_id,
      'widget_uuid' => $widget_uuid,
      'transferId' => '__TRANSFER_ID__',
    ])->toString();
    // Remove the placeholder - JS will append the actual transfer ID.
    $patch_url = str_replace('/__TRANSFER_ID__', '', $patch_url);

    $revert_url = Url::fromRoute('filepond_eb_widget.revert', [
      'entity_browser' => $entity_browser_id,
      'widget_uuid' => $widget_uuid,
    ])->toString();

    // Get chunk size - config stores MB, element expects bytes.
    // Empty value inherits from filepond.settings at element level.
    $chunk_size_mb = $config['settings']['chunk_size'] ?? NULL;
    $chunk_size_bytes = $chunk_size_mb ? $chunk_size_mb * 1024 * 1024 : NULL;

    // FilePond upload element with EB-specific routes.
    $form['upload'] = [
      '#type' => 'filepond',
      '#title' => $this->t('Upload files'),
      '#description' => $config['settings']['description'] ?? NULL,
      '#required' => TRUE,
      // Server-side properties (root level).
      // Use ?: NULL so empty values fall back to filepond.settings defaults.
      // Note: #max_files uses 0 for unlimited, so don't use ?: for it.
      '#extensions' => $config['settings']['extensions'] ?: NULL,
      '#max_filesize' => $config['settings']['max_filesize'] ?: NULL,
      '#max_files' => $max_files,
      '#upload_location' => $this->getUploadLocation() ?: NULL,
      // CSS grid sizing - empty values inherit from filepond.settings.
      '#columns' => $config['settings']['columns'] ?: NULL,
      '#max_width' => $config['settings']['max_width'] ?: NULL,
      // FilePond UI options.
      // Only set values if not empty - empty values fall back to
      // filepond.settings.
      '#config' => array_filter([
        'labelIdle' => $config['settings']['upload_prompt'] ?: NULL,
        'chunkSize' => $chunk_size_bytes ?: NULL,
        'maxParallelUploads' => $config['settings']['max_parallel_uploads'] ?: NULL,
        'styleItemPanelAspectRatio' => $config['settings']['item_panel_aspect_ratio'] ?: NULL,
        'previewFitMode' => $config['settings']['preview_fit_mode'] ?: NULL,
        // Disable reorder - files processed immediately, order irrelevant.
        'allowReorder' => FALSE,
        // Use EB routes instead of tempstore routes.
        'processUrl' => $process_url,
        'patchUrl' => $patch_url,
        'revertUrl' => $revert_url,
      ], fn($v) => $v !== NULL),
    ];

    // Attach widget styles.
    $form['#attached']['library'][] = 'filepond_eb_widget/widget';

    return $form;
  }

  /**
   * Gets the token-replaced upload location.
   *
   * @return string
   *   The upload location URI.
   */
  protected function getUploadLocation(): string {
    return $this->token->replace($this->configuration['upload_location']);
  }

  /**
   * {@inheritdoc}
   */
  public function validate(array &$form, FormStateInterface $form_state): void {
    $trigger = $form_state->getTriggeringElement();

    // Only validate on actual submit.
    if ($trigger['#type'] !== 'submit' || $trigger['#name'] !== 'op') {
      return;
    }

    // Get file IDs from the filepond element.
    $upload_value = $form_state->getValue(['upload']);
    $fids = $upload_value['fids'] ?? [];

    if (empty($fids)) {
      $form_state->setError($form['widget']['upload'], $this->t('Please upload at least one file.'));
      return;
    }

    // Verify media type is configured.
    $media_type = $this->getMediaType();
    if (!$media_type) {
      $form_state->setError($form['widget']['upload'], $this->t('No media type configured for this widget.'));
      return;
    }

    parent::validate($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function prepareEntities(array $form, FormStateInterface $form_state): array {
    $upload_value = $form_state->getValue(['upload']);
    $fids = $upload_value['fids'] ?? [];

    if (empty($fids)) {
      return [];
    }

    $media_type = $this->getMediaType();
    if (!$media_type) {
      return [];
    }

    // Load file entities.
    $file_storage = $this->entityTypeManager->getStorage('file');
    $files = $file_storage->loadMultiple($fids);

    // Create media entities (not saved yet).
    $media_entities = [];
    foreach ($files as $file) {
      if (!$file instanceof FileInterface) {
        continue;
      }

      $values = $this->buildMediaValues($file, $media_type);
      $media_entities[] = $this->entityTypeManager->getStorage('media')->create($values);
    }

    return $media_entities;
  }

  /**
   * {@inheritdoc}
   */
  public function submit(array &$element, array &$form, FormStateInterface $form_state): void {
    $media_entities = $this->prepareEntities($form, $form_state);

    if (empty($media_entities)) {
      return;
    }

    $media_type = $this->getMediaType();
    $source_field = $media_type->getSource()->getConfiguration()['source_field'];

    // Save all media entities.
    foreach ($media_entities as $media_entity) {
      // Mark file as permanent before saving media.
      $file = $media_entity->get($source_field)->entity;
      if ($file instanceof FileInterface && !$file->isPermanent()) {
        $file->setPermanent();
        $file->save();
      }
      $media_entity->save();
    }

    $this->selectEntities($media_entities, $form_state);
    $this->clearFormValues($element, $form_state);
  }

  /**
   * Clears form values after submission.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function clearFormValues(array &$element, FormStateInterface $form_state): void {
    $form_state->setValueForElement($element['upload'], ['fids' => []]);
    NestedArray::setValue(
      $form_state->getUserInput(),
      array_merge($element['upload']['#parents'], ['fids']),
      ''
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function handleWidgetContext($widget_context): void {
    // Handle default widget context from parent.
    foreach ($this->defaultConfiguration() as $key => $value) {
      if (isset($widget_context[$key]) && isset($this->configuration[$key])) {
        $this->configuration[$key] = $widget_context[$key];
      }
    }

    // If inherit_settings is enabled, get settings from the media type.
    if ($this->configuration['inherit_settings']) {
      $media_type = $this->getMediaType();
      if ($media_type) {
        $source = $media_type->getSource();
        $field = $source->getSourceFieldDefinition($media_type);
        if ($field) {
          $field_storage = $field->getFieldStorageDefinition();
          $field_settings = $field->getSettings();

          // Build upload location from uri_scheme + file_directory.
          $uri_scheme = $field_storage->getSettings()['uri_scheme'] ?? 'public';
          $file_directory = $field_settings['file_directory'] ?? '';
          // Replace tokens in file_directory.
          $file_directory = $this->token->replace($file_directory);
          $this->configuration['upload_location'] = $uri_scheme . '://' . $file_directory;

          // Get max filesize from field settings.
          if (!empty($field_settings['max_filesize'])) {
            $this->configuration['max_filesize'] = $field_settings['max_filesize'];
          }

          // Get extensions from field settings.
          if (!empty($field_settings['file_extensions'])) {
            $this->configuration['extensions'] = $field_settings['file_extensions'];
          }
        }
      }
    }
  }

}
