<?php

declare(strict_types=1);

namespace Drupal\filepond_views\Controller;

use Drupal\filepond\Controller\UploadController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Handles FilePond upload requests for Views area context.
 *
 * Extends the main UploadController and uses the settings resolver for
 * views area option resolution. The resolver adds views_area flag to context
 * so the event subscriber knows to create media entities on upload.
 *
 * Routes:
 * - POST /filepond/views/{view_id}/{display_id}/process
 * - PATCH /filepond/views/{view_id}/{display_id}/patch/{transferId}
 * - DELETE /filepond/views/{view_id}/{display_id}/revert
 */
class ViewsUploadController extends UploadController {

  /**
   * Handles Views process requests (POST).
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $view_id
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function viewsProcess(Request $request, string $view_id, string $display_id): Response {
    $options = $this->settingsResolver->resolveFromViewsArea($view_id, $display_id);
    return $this->handleProcess($request, $options->toArray());
  }

  /**
   * Handles Views PATCH requests for chunked uploads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $view_id
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   * @param string $transferId
   *   The transfer ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function viewsPatch(Request $request, string $view_id, string $display_id, string $transferId): Response {
    $options = $this->settingsResolver->resolveFromViewsArea($view_id, $display_id);
    return $this->handlePatch($request, $transferId, $options->toArray());
  }

  /**
   * Handles Views DELETE requests to revert/cancel uploads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $view_id
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function viewsRevert(Request $request, string $view_id, string $display_id): Response {
    // Validate view and display exist.
    $this->settingsResolver->resolveFromViewsArea($view_id, $display_id);
    return $this->handleRevert($request);
  }

}
