<?php

declare(strict_types=1);

namespace Drupal\filepond_views\EventSubscriber;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\filepond\Event\FilePondUploadCompleteEvent;
use Drupal\filepond\MediaSourceFieldTrait;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Creates media entities when files are uploaded via Views area.
 *
 * Listens for FilePondUploadCompleteEvent and creates media when
 * the upload context includes 'views_area: true'.
 */
class ViewsAreaMediaSubscriber implements EventSubscriberInterface {

  use MediaSourceFieldTrait;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a ViewsAreaMediaSubscriber.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      FilePondUploadCompleteEvent::EVENT_NAME => 'onUploadComplete',
    ];
  }

  /**
   * Handles the upload complete event.
   *
   * Creates a media entity when the upload came from a Views area.
   *
   * @param \Drupal\filepond\Event\FilePondUploadCompleteEvent $event
   *   The upload complete event.
   */
  public function onUploadComplete(FilePondUploadCompleteEvent $event): void {
    $context = $event->getContext();

    // Only handle views_area uploads.
    if (empty($context['views_area'])) {
      return;
    }

    $media_type_id = $context['media_type'] ?? NULL;
    if (!$media_type_id) {
      return;
    }

    // Load media type.
    /** @var \Drupal\media\MediaTypeInterface|null $media_type */
    $media_type = $this->entityTypeManager->getStorage('media_type')->load($media_type_id);
    if (!$media_type) {
      return;
    }

    $file = $event->getFile();

    // Create media using trait (handles dimensions, thumbnail optimization).
    $values = $this->buildMediaValues($file, $media_type);
    /** @var \Drupal\media\MediaInterface $media */
    $media = $this->entityTypeManager->getStorage('media')->create($values);

    // Mark file as permanent.
    $file->setPermanent();
    $file->save();

    // Save media.
    $media->save();

    // Return media ID to JS.
    $event->setResult('media_id', $media->id());
  }

}
