<?php

declare(strict_types=1);

namespace Drupal\filepond_views;

use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\media\MediaTypeInterface;
use Drupal\views\Views;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Lazy builder for FilePond Views area plugin.
 *
 * This allows the surrounding view to be cached while the uploader
 * renders dynamically based on user permissions, Entity Browser context,
 * and request parameters.
 */
class FilePondUploaderLazyBuilder implements TrustedCallbackInterface {

  use StringTranslationTrait;

  /**
   * The service container.
   *
   * Used for checking optional services like entity_browser.selection_storage.
   */
  protected ContainerInterface $container;

  /**
   * Constructs a FilePondUploaderLazyBuilder.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected AccountProxyInterface $currentUser,
    protected RequestStack $requestStack,
    protected ConfigFactoryInterface $configFactory,
    ContainerInterface $container,
  ) {
    $this->container = $container;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks(): array {
    return ['renderUploader'];
  }

  /**
   * Renders the FilePond uploader.
   *
   * @param string $view_id
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   * @param string $view_dom_id
   *   The view's DOM ID.
   * @param string $area_type
   *   The area type (e.g., 'header', 'footer', 'empty').
   * @param string $handler_id
   *   The handler ID within the area.
   *
   * @return array
   *   A render array.
   */
  public function renderUploader(string $view_id, string $display_id, string $view_dom_id, string $area_type, string $handler_id): array {
    // Load options from the view configuration.
    $options = $this->getAreaPluginOptions($view_id, $display_id, $area_type, $handler_id);
    if ($options === NULL) {
      return [
        '#theme' => 'status_messages',
        '#message_list' => [
          'error' => [new TranslatableMarkup('FilePond Upload: Could not load view configuration.')],
        ],
      ];
    }

    // Check permission.
    if (!$this->currentUser->hasPermission('filepond upload files')) {
      return [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [new TranslatableMarkup("You don't have permission to upload files.")],
        ],
      ];
    }

    // @todo support creating files only, not just media.
    $media_type_id = $options['media_type'] ?? NULL;
    if (!$media_type_id) {
      return [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [new TranslatableMarkup('FilePond Upload: No media type configured or detected.')],
        ],
      ];
    }

    // Verify media type exists.
    /** @var \Drupal\media\MediaTypeInterface|null $media_type */
    $media_type = $this->entityTypeManager->getStorage('media_type')->load($media_type_id);
    if (!$media_type) {
      return [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [new TranslatableMarkup('FilePond Upload: Media type "@type" not found.', ['@type' => $media_type_id])],
        ],
      ];
    }

    // Get upload settings (from media type or manual config).
    $upload_settings = $this->getUploadSettings($options, $media_type);

    // Start with configured max_files (0 = unlimited).
    $max_files = (int) ($options['max_files'] ?? 0);

    // Check Entity Browser context for cardinality limits.
    $eb_context = $this->getEntityBrowserContext();
    if ($eb_context['limit_reached']) {
      return [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [$this->t('Maximum file limit reached for this field.')],
        ],
      ];
    }
    // Apply EB remaining slots if more restrictive than configured max.
    if ($eb_context['remaining'] !== NULL) {
      if ($max_files === 0 || $eb_context['remaining'] < $max_files) {
        $max_files = $eb_context['remaining'];
      }
    }

    // Build upload URLs using view_id and display_id.
    $process_url = Url::fromRoute('filepond_views.process', [
      'view_id' => $view_id,
      'display_id' => $display_id,
    ])->toString();
    $patch_url = Url::fromRoute('filepond_views.patch', [
      'view_id' => $view_id,
      'display_id' => $display_id,
      'transferId' => '__TRANSFER_ID__',
    ])->toString();
    // Remove the placeholder suffix - JS will append the actual transfer ID.
    $patch_url = str_replace('/__TRANSFER_ID__', '', $patch_url);
    $revert_url = Url::fromRoute('filepond_views.revert', [
      'view_id' => $view_id,
      'display_id' => $display_id,
    ])->toString();

    // Generate unique element ID for this instance.
    $element_id = Html::getUniqueId('filepond-views-' . $view_dom_id);

    // Get chunk size - config stores MB, element expects bytes.
    $chunk_size_mb = $options['chunk_size'] ?? NULL;
    $chunk_size_bytes = $chunk_size_mb * 1024 * 1024;

    // Build FilePond config from options.
    // Use ?: for string options so empty strings are treated as NULL
    // (allowing FilePond defaults to apply).
    $config = array_filter([
      'labelIdle' => $options['upload_prompt'] ?: NULL,
      'chunkSize' => $chunk_size_bytes ?: NULL,
      'maxParallelUploads' => $options['max_parallel_uploads'] ?: NULL,
      'styleItemPanelAspectRatio' => $options['item_panel_aspect_ratio'] ?: NULL,
      'previewFitMode' => $options['preview_fit_mode'] ?: NULL,
      'allowReorder' => FALSE,
    ], fn($v) => $v !== NULL);

    // Use the FilePondUploader render element from the main module.
    $build = [
      '#type' => 'filepond_uploader',
      '#id' => $element_id,
      '#extensions' => $upload_settings['extensions'],
      '#max_files' => (int) $max_files,
      '#max_filesize' => $upload_settings['max_filesize'],
      '#process_url' => $process_url,
      '#patch_url' => $patch_url,
      '#revert_url' => $revert_url,
      '#config' => $config,
      '#columns' => $options['columns'] ?? 5,
      '#max_width' => $options['max_width'] ?? NULL,
      '#attributes' => [
        'class' => ['filepond-views-area'],
        'data-views-dom-id' => $view_dom_id,
      ],
      // Attach views-specific library and settings for refresh behavior.
      '#attached' => [
        'library' => [
          'filepond_views/filepond_views',
        ],
        'drupalSettings' => [
          'filepondViews' => [
            $element_id => [
              'viewsDomId' => $view_dom_id,
              'mediaType' => $media_type_id,
              'autoSelect' => !empty($options['auto_select']),
              // Pass max_files for JS-side limit enforcement.
              'maxFiles' => $max_files,
            ],
          ],
        ],
      ],
    ];

    return $build;
  }

  /**
   * Gets upload settings based on inherit_settings option.
   *
   * @param array $options
   *   The area plugin options.
   * @param \Drupal\media\MediaTypeInterface|null $media_type
   *   The media type entity, or NULL if not configured.
   *
   * @return array
   *   Array with extensions and max_filesize.
   */
  protected function getUploadSettings(array $options, ?MediaTypeInterface $media_type = NULL): array {
    $extensions = $options['extensions'] ?? NULL;
    $max_filesize = $options['max_filesize'] ?? NULL;
    $upload_location = $options['upload_location'] ?? NULL;

    if ($media_type) {
      $source = $media_type->getSource();
      $source_field = $source->getSourceFieldDefinition($media_type);
      if ($source_field) {
        $field_settings = $source_field->getSettings();
        $extensions = !empty($field_settings['file_extensions']) ? $field_settings['file_extensions'] : $extensions;
        if (!empty($options['inherit_settings'])) {
          $max_filesize = !empty($field_settings['max_filesize']) ? $field_settings['max_filesize'] : $max_filesize;
          $upload_location = !empty($field_settings['upload_location']) ? $field_settings['upload_location'] : $upload_location;
        }
      }
    }

    return [
      'extensions' => $extensions,
      'max_filesize' => $max_filesize,
      'upload_location' => $upload_location,
    ];
  }

  /**
   * Gets Entity Browser context from selection storage.
   *
   * When embedded in an Entity Browser, the widget context contains
   * cardinality limits and currently selected entity IDs. This method
   * retrieves that context and calculates remaining upload slots.
   *
   * @return array
   *   Array with keys:
   *   - 'remaining': int|null - Remaining slots (NULL if no EB context)
   *   - 'limit_reached': bool - TRUE if no more uploads allowed
   *
   * @see \Drupal\entity_browser\Plugin\views\argument_default\EntityBrowserWidgetContext
   */
  protected function getEntityBrowserContext(): array {
    $result = [
      'remaining' => NULL,
      'limit_reached' => FALSE,
    ];

    $request = $this->requestStack->getCurrentRequest();
    if (!$request || !$request->query->has('uuid')) {
      return $result;
    }

    $uuid = $request->query->get('uuid');

    // Entity Browser is an optional module - check if service exists.
    if (!$this->container->has('entity_browser.selection_storage')) {
      return $result;
    }

    $storage = $this->container->get('entity_browser.selection_storage')->get($uuid);
    if (!$storage) {
      return $result;
    }

    $widget_context = $storage['widget_context'] ?? [];
    $current_ids = $widget_context['current_ids'] ?? [];
    $cardinality = $widget_context['cardinality'] ?? -1;

    // Cardinality values: positive = limit, -1 = unlimited.
    if ($cardinality > 0) {
      $remaining = $cardinality - count($current_ids);
      $result['remaining'] = $remaining;
      $result['limit_reached'] = $remaining <= 0;
    }

    return $result;
  }

  /**
   * Gets the area plugin options from view configuration.
   *
   * @param string $view_id
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   * @param string $area_type
   *   The area type (header, footer, or empty).
   * @param string $handler_id
   *   The handler ID within the area.
   *
   * @return array|null
   *   The plugin options or NULL if not found.
   */
  protected function getAreaPluginOptions(string $view_id, string $display_id, string $area_type, string $handler_id): ?array {
    $view = Views::getView($view_id);
    if (!$view) {
      return NULL;
    }

    $view->setDisplay($display_id);
    $display = $view->getDisplay();

    // Get the handler by its ID.
    $handler = $display->getHandler($area_type, $handler_id);
    if ($handler && $handler->getPluginId() === 'filepond_upload') {
      return $handler->options;
    }

    return NULL;
  }

}
