<?php

declare(strict_types=1);

namespace Drupal\filepond\Controller;

use Drupal\Component\Serialization\Json;
use Drupal\Component\Utility\Crypt;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\PrivateKey;
use Drupal\Core\Site\Settings;
use Drupal\filepond\FilePondUploadHandler;
use Drupal\filepond\UploadSettingsResolverInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Handles FilePond upload requests.
 *
 * This controller handles the FilePond server protocol with multiple route
 * types:
 *
 * Form-based routes (State API):
 * - POST /filepond/form/{form_id}/{element_name}/process
 * - PATCH /filepond/form/{form_id}/{element_name}/patch/{transferId}
 * - DELETE /filepond/form/{form_id}/{element_name}/revert
 *
 * Field widget routes (from field definition):
 * - POST /filepond/field/{entity_type}/{bundle}/{field_name}/process
 * - etc.
 *
 * Media library routes (from media type):
 * - POST /filepond/media/{media_type}/process
 * - etc.
 *
 * Form-based routes store config in State API keyed by form_id:element_name.
 * This is stored once per form element (not per user), making it efficient
 * for high-traffic forms.
 */
class UploadController extends ControllerBase {

  /**
   * The upload handler service.
   */
  protected FilePondUploadHandler $uploadHandler;

  /**
   * The settings resolver service.
   */
  protected UploadSettingsResolverInterface $settingsResolver;

  /**
   * The private key service.
   */
  protected PrivateKey $privateKey;

  /**
   * Constructs an UploadController.
   *
   * @param \Drupal\filepond\FilePondUploadHandler $upload_handler
   *   The upload handler service.
   * @param \Drupal\filepond\UploadSettingsResolverInterface $settings_resolver
   *   The settings resolver service.
   * @param \Drupal\Core\PrivateKey $private_key
   *   The private key service.
   */
  public function __construct(
    FilePondUploadHandler $upload_handler,
    UploadSettingsResolverInterface $settings_resolver,
    PrivateKey $private_key,
  ) {
    $this->uploadHandler = $upload_handler;
    $this->settingsResolver = $settings_resolver;
    $this->privateKey = $private_key;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('filepond.upload_handler'),
      $container->get('filepond.settings_resolver'),
      $container->get('private_key'),
    );
  }

  /**
   * Handles form-based process requests (POST).
   *
   * For chunked uploads: Creates transfer ID when Upload-Length header present.
   * For regular uploads: Processes the file directly.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $form_id
   *   The form ID.
   * @param string $element_name
   *   The element name.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response with transfer ID or file ID.
   */
  public function formProcess(Request $request, string $form_id, string $element_name): Response {
    $options = $this->settingsResolver->resolveFromState($form_id, $element_name);
    if ($options === NULL) {
      return new Response('Upload configuration not found for this form element', 404);
    }
    return $this->handleProcess($request, $options->toArray());
  }

  /**
   * Handles a process request (regular upload or chunked init).
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param array $options
   *   Upload options.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response with transfer ID or file ID.
   */
  protected function handleProcess(Request $request, array $options): Response {
    // Check for file in request FIRST.
    // If a file is present, this is a regular (non-chunked) upload.
    $file = $request->files->get('filepond');
    if ($file) {
      return $this->handleRegularUpload($request, $file, $options);
    }

    // No file in request - check if this is a chunked upload initialization.
    // FilePond sends Upload-Length header (without file) for chunked uploads.
    $uploadLength = $request->headers->get('Upload-Length', '');
    if (!empty($uploadLength)) {
      return $this->initializeChunkedUpload($request, $options);
    }

    // No file and no Upload-Length - invalid request.
    return new Response('No file uploaded and no Upload-Length header', 400);
  }

  /**
   * Initializes a chunked upload.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param array $options
   *   Upload options.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response with transfer ID.
   */
  protected function initializeChunkedUpload(Request $request, array $options): Response {
    $uploadLength = (int) $request->headers->get('Upload-Length', '0');
    // FilePond URL-encodes the filename. Default to .jpg if header missing.
    $rawUploadName = $request->headers->get('Upload-Name');
    $uploadName = urldecode($rawUploadName ?? 'unknown.jpg');

    if ($uploadLength <= 0) {
      return new Response('Invalid upload length', 400);
    }

    $result = $this->uploadHandler->initializeChunkedUpload(
      $uploadLength,
      $uploadName,
      $options
    );

    if (!$result['success']) {
      return new Response($result['error'] ?? 'Upload failed', $result['code'] ?? 500);
    }

    return new Response($result['transfer_id'], 200, [
      'Content-Type' => 'text/plain',
    ]);
  }

  /**
   * Handles a regular (non-chunked) file upload.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param \Symfony\Component\HttpFoundation\File\UploadedFile $file
   *   The uploaded file.
   * @param array $options
   *   Upload options.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response with file ID (and media ID if created by event subscriber).
   */
  protected function handleRegularUpload(Request $request, $file, array $options): Response {
    $result = $this->uploadHandler->handleUpload($file, $options);

    if (!$result['success']) {
      return new Response($result['error'] ?? 'Upload failed', $result['code'] ?? 500);
    }

    $signedFileId = $this->signFileId($result['file']->id());

    // If event subscribers set results (e.g., media_id), return JSON.
    if (!empty($result['event_results'])) {
      $response = [
        'fileId' => $signedFileId,
      ];
      // Include media_id if set by subscriber (e.g., ViewsAreaMediaSubscriber).
      if (!empty($result['event_results']['media_id'])) {
        $response['mediaId'] = $result['event_results']['media_id'];
      }
      return new Response(Json::encode($response), 200, [
        'Content-Type' => 'application/json',
      ]);
    }

    // Return signed file ID - prevents attackers from guessing IDs.
    return new Response($signedFileId, 200, [
      'Content-Type' => 'text/plain',
    ]);
  }

  /**
   * Handles form-based PATCH requests for chunked uploads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $form_id
   *   The form ID.
   * @param string $element_name
   *   The element name.
   * @param string $transferId
   *   The transfer ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function formPatch(Request $request, string $form_id, string $element_name, string $transferId): Response {
    $options = $this->settingsResolver->resolveFromState($form_id, $element_name);
    if ($options === NULL) {
      return new Response('Upload configuration not found for this form element', 404);
    }
    return $this->handlePatch($request, $transferId, $options->toArray());
  }

  /**
   * Handles form-based DELETE requests to revert/cancel uploads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $form_id
   *   The form ID.
   * @param string $element_name
   *   The element name.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function formRevert(Request $request, string $form_id, string $element_name): Response {
    $options = $this->settingsResolver->resolveFromState($form_id, $element_name);
    if ($options === NULL) {
      return new Response('Upload configuration not found for this form element', 404);
    }
    return $this->handleRevert($request);
  }

  /**
   * Handles a PATCH (or HEAD) request for chunked uploads.
   *
   * HEAD requests are used by FilePond to check the current upload offset
   * when resuming a failed upload. The server responds with Upload-Offset
   * header indicating where to resume from.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request containing chunk data.
   * @param string $transferId
   *   The transfer ID.
   * @param array $options
   *   Upload options.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response with offset or file ID on completion.
   */
  protected function handlePatch(Request $request, string $transferId, array $options): Response {
    // HEAD requests are used to check current upload offset for resume.
    // Return the current offset without processing any chunk data.
    if ($request->isMethod('HEAD')) {
      $offset = $this->uploadHandler->getUploadOffset($transferId, $options);
      return new Response('', 200, [
        'Upload-Offset' => (string) $offset,
      ]);
    }

    $offset = (int) $request->headers->get('Upload-Offset', '0');
    $chunk = $request->getContent();

    // FilePond sends Upload-Name and Upload-Length in PATCH requests.
    $uploadName = $request->headers->get('Upload-Name');
    if ($uploadName) {
      $options['upload_name'] = urldecode($uploadName);
    }
    $uploadLength = $request->headers->get('Upload-Length');
    if ($uploadLength) {
      $options['upload_length'] = (int) $uploadLength;
    }

    $result = $this->uploadHandler->processChunk($transferId, $chunk, $offset, $options);

    if (!$result['success']) {
      return new Response($result['error'] ?? 'Chunk failed', $result['code'] ?? 500);
    }

    // If complete, return JSON with transfer ID and signed file ID.
    if (!empty($result['complete']) && !empty($result['file'])) {
      $response = [
        'transferId' => $transferId,
        'fileId' => $this->signFileId($result['file']->id()),
      ];
      // Include media_id if set by event subscriber.
      if (isset($result['event_results']['media_id'])) {
        $response['mediaId'] = $result['event_results']['media_id'];
      }
      return new Response(Json::encode($response), 200, [
        'Content-Type' => 'application/json',
      ]);
    }

    // Return new offset for next chunk.
    return new Response('', 204, [
      'Upload-Offset' => (string) $result['offset'],
    ]);
  }

  /**
   * Handles a DELETE request to revert/cancel uploads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request containing the server ID in the body.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Empty response on success.
   */
  protected function handleRevert(Request $request): Response {
    $serverId = trim($request->getContent());
    if (empty($serverId)) {
      return new Response('No server ID', 400);
    }

    // If serverId is JSON (from uploads that returned mediaId), handle media
    // cleanup. This happens when ViewsAreaMediaSubscriber creates media on
    // upload. We can safely delete the media if all security checks pass.
    if (str_starts_with($serverId, '{')) {
      $data = Json::decode($serverId);
      if (isset($data['fileId']) && isset($data['mediaId'])) {
        return $this->handleMediaRevert($data['fileId'], $data['mediaId']);
      }
      // Fall through to normal file revert if only fileId present.
      if (isset($data['fileId'])) {
        $serverId = $data['fileId'];
      }
    }

    // Determine the type of server ID and validate accordingly.
    // 1. Signed file ID (fid:hmac) - verify HMAC, extract file ID.
    // 2. Transfer ID (32-char hex) - temp file cleanup during chunked upload.
    // 3. Plain numeric ID - reject to prevent guessing attacks.
    if (str_contains($serverId, ':')) {
      // Signed file ID - verify HMAC.
      $fileId = $this->verifySignedFileId($serverId);
      if ($fileId === NULL) {
        return new Response('Invalid file token', 403);
      }
      // Pass verified file ID to handler.
      $serverId = $fileId;
    }
    elseif (is_numeric($serverId)) {
      // Plain numeric ID without signature - reject to prevent guessing.
      // All file IDs from our upload endpoints are signed. Existing files
      // use plain IDs but they're permanent so revert is a no-op anyway.
      // Return success so FilePond removes from UI.
      return new Response('skipped_unsigned', 200);
    }
    // Transfer ID (hex string) passes through to handler for temp cleanup.
    $result = $this->uploadHandler->revert($serverId);

    if (!$result['success']) {
      return new Response($result['error'] ?? 'Revert failed', $result['code'] ?? 500);
    }

    return new Response($result['action'] ?? 'success', 200);
  }

  /**
   * Handles revert for uploads that created media entities.
   *
   * When ViewsAreaMediaSubscriber creates media on upload, the response
   * includes both fileId and mediaId. On revert, we can safely delete the
   * media if all security checks pass:
   * - fileId has valid HMAC (proves we issued it)
   * - Current user owns the media
   * - Media's source field references this exact file (binding check)
   *
   * @param string $signedFileId
   *   The signed file ID (fid:hmac format).
   * @param int|string $mediaId
   *   The media entity ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response indicating success or failure.
   */
  protected function handleMediaRevert(string $signedFileId, int|string $mediaId): Response {
    // Verify file ID signature.
    $fileId = $this->verifySignedFileId($signedFileId);
    if ($fileId === NULL) {
      return new Response('Invalid file token', 403);
    }

    // Load media and verify delete access.
    /** @var \Drupal\media\MediaInterface|null $media */
    $media = $this->entityTypeManager()->getStorage('media')->load($mediaId);
    if (!$media) {
      // Media already deleted or never existed - that's fine.
      return new Response('media_not_found', 200);
    }

    if (!$media->access('delete')) {
      return new Response('Access denied', 403);
    }

    // Verify media actually references this file (prevents mediaId swapping).
    $source = $media->getSource();
    $source_field_name = $source->getSourceFieldDefinition($media->bundle->entity)->getName();
    /** @var \Drupal\file\FileInterface|null $media_file */
    $media_file = $media->get($source_field_name)->entity;

    if (!$media_file || $media_file->id() != $fileId) {
      return new Response('File/media mismatch', 403);
    }

    // All checks passed - safe to delete media and file.
    $media->delete();
    $media_file->delete();

    return new Response('media_deleted', 200);
  }

  /**
   * Handles field-based process requests (POST).
   */
  public function fieldProcess(Request $request, string $entity_type, string $bundle, string $field_name): Response {
    $options = $this->settingsResolver->resolveFromField($entity_type, $bundle, $field_name);
    return $this->handleProcess($request, $options->toArray());
  }

  /**
   * Handles field-based PATCH requests for chunked uploads.
   */
  public function fieldPatch(Request $request, string $entity_type, string $bundle, string $field_name, string $transferId): Response {
    $options = $this->settingsResolver->resolveFromField($entity_type, $bundle, $field_name);
    return $this->handlePatch($request, $transferId, $options->toArray());
  }

  /**
   * Handles field-based DELETE requests to revert/cancel uploads.
   */
  public function fieldRevert(Request $request, string $entity_type, string $bundle, string $field_name): Response {
    // Validate the field exists.
    $this->settingsResolver->resolveFromField($entity_type, $bundle, $field_name);
    return $this->handleRevert($request);
  }

  /**
   * Handles media-type-based process requests (POST).
   */
  public function mediaProcess(Request $request, string $media_type): Response {
    $options = $this->settingsResolver->resolveFromMediaType($media_type);
    return $this->handleProcess($request, $options->toArray());
  }

  /**
   * Handles media-type-based PATCH requests for chunked uploads.
   */
  public function mediaPatch(Request $request, string $media_type, string $transferId): Response {
    $options = $this->settingsResolver->resolveFromMediaType($media_type);
    return $this->handlePatch($request, $transferId, $options->toArray());
  }

  /**
   * Handles media-type-based DELETE requests to revert/cancel uploads.
   */
  public function mediaRevert(Request $request, string $media_type): Response {
    // Validate the media type exists.
    $this->settingsResolver->resolveFromMediaType($media_type);
    return $this->handleRevert($request);
  }

  /**
   * Signs a file ID to create a tamper-proof token.
   *
   * This prevents attackers from guessing file IDs and deleting other users'
   * uploads. The token format is "fid:hmac" where the HMAC is computed using
   * the site's private key.
   *
   * @param int|string $fileId
   *   The file entity ID.
   *
   * @return string
   *   The signed token in format "fid:hmac".
   */
  public function signFileId(int|string $fileId): string {
    $hmac = Crypt::hmacBase64(
      'filepond-file-' . $fileId,
      $this->privateKey->get() . Settings::getHashSalt()
    );
    return $fileId . ':' . $hmac;
  }

  /**
   * Verifies a signed file ID token and extracts the file ID.
   *
   * @param string $signedToken
   *   The signed token in format "fid:hmac".
   *
   * @return string|null
   *   The file ID if valid, or NULL if verification fails.
   */
  protected function verifySignedFileId(string $signedToken): ?string {
    // Check for colon separator.
    if (!str_contains($signedToken, ':')) {
      return NULL;
    }

    [$fileId, $providedHmac] = explode(':', $signedToken, 2);

    // File ID must be numeric.
    if (!is_numeric($fileId)) {
      return NULL;
    }

    // Compute expected HMAC and compare.
    $expectedHmac = Crypt::hmacBase64(
      'filepond-file-' . $fileId,
      $this->privateKey->get() . Settings::getHashSalt()
    );

    if (!hash_equals($expectedHmac, $providedHmac)) {
      return NULL;
    }

    return $fileId;
  }

}
