<?php

declare(strict_types=1);

namespace Drupal\filepond;

/**
 * Gets image dimensions by reading only header bytes.
 *
 * This is much faster than getimagesize() for remote files (S3, HTTP)
 * because it only downloads the bytes needed to determine dimensions:
 * - PNG: 25 bytes
 * - GIF: 11 bytes
 * - BMP: 29 bytes
 * - JPEG: Variable (scans for SOF marker)
 *
 * Based on the Ruby implementation by Steven Sykes:
 * https://github.com/sdsykes/fastimage
 *
 * PHP port by Tom Moor (2012): http://tommoor.com
 *
 * @see https://github.com/tommoor/fastimage
 * @see https://www.drupal.org/project/image_field_repair/issues/2957577#comment-12555735
 *
 * @internal
 *   This class is for internal use by the FilePond module.
 */
class FastImage {

  /**
   * Current position in the buffered string.
   */
  private int $strpos = 0;

  /**
   * Buffered bytes read from the file.
   */
  private string $str = '';

  /**
   * Detected image type.
   */
  private ?string $type = NULL;

  /**
   * File handle.
   *
   * @var resource|null
   */
  private $handle = NULL;

  /**
   * Constructs a FastImage instance.
   *
   * @param string|null $uri
   *   Optional URI to load immediately.
   */
  public function __construct(?string $uri = NULL) {
    if ($uri) {
      $this->load($uri);
    }
  }

  /**
   * Loads a file for dimension reading.
   *
   * @param string $uri
   *   The file URI to load.
   *
   * @return bool
   *   TRUE if file was opened successfully.
   */
  public function load(string $uri): bool {
    if ($this->handle) {
      $this->close();
    }

    $handle = @fopen($uri, 'r');
    if ($handle === FALSE) {
      return FALSE;
    }

    $this->handle = $handle;
    return TRUE;
  }

  /**
   * Closes the file handle and resets state.
   */
  public function close(): void {
    if ($this->handle) {
      fclose($this->handle);
      $this->handle = NULL;
      $this->type = NULL;
      $this->str = '';
      $this->strpos = 0;
    }
  }

  /**
   * Gets the image dimensions.
   *
   * @return array|null
   *   Array with [width, height] or NULL if not determinable.
   */
  public function getSize(): ?array {
    if (!$this->handle) {
      return NULL;
    }

    $this->strpos = 0;

    if ($this->getType()) {
      $size = $this->parseSize();
      if ($size) {
        return array_values($size);
      }
    }

    return NULL;
  }

  /**
   * Detects the image type from header bytes.
   *
   * @return string|null
   *   The image type ('png', 'gif', 'jpeg', 'bmp') or NULL.
   */
  public function getType(): ?string {
    if (!$this->handle) {
      return NULL;
    }

    $this->strpos = 0;

    if (!$this->type) {
      $chars = $this->getChars(2);
      if ($chars === NULL) {
        return NULL;
      }

      switch ($chars) {
        case "BM":
          $this->type = 'bmp';
          break;

        case "GI":
          $this->type = 'gif';
          break;

        case chr(0xFF) . chr(0xD8):
          $this->type = 'jpeg';
          break;

        case chr(0x89) . 'P':
          $this->type = 'png';
          break;

        default:
          return NULL;
      }
    }

    return $this->type;
  }

  /**
   * Parses dimensions based on detected type.
   *
   * @return array|null
   *   Associative array with dimensions or NULL.
   */
  private function parseSize(): ?array {
    $this->strpos = 0;

    switch ($this->type) {
      case 'png':
        return $this->parseSizeForPng();

      case 'gif':
        return $this->parseSizeForGif();

      case 'bmp':
        return $this->parseSizeForBmp();

      case 'jpeg':
        return $this->parseSizeForJpeg();
    }

    return NULL;
  }

  /**
   * Parses PNG dimensions (needs 25 bytes).
   */
  private function parseSizeForPng(): ?array {
    $chars = $this->getChars(25);
    if ($chars === NULL || strlen($chars) < 24) {
      return NULL;
    }

    $data = unpack("N*", substr($chars, 16, 8));
    if (!$data || count($data) < 2) {
      return NULL;
    }

    return array_values($data);
  }

  /**
   * Parses GIF dimensions (needs 11 bytes).
   */
  private function parseSizeForGif(): ?array {
    $chars = $this->getChars(11);
    if ($chars === NULL || strlen($chars) < 10) {
      return NULL;
    }

    $data = unpack("S*", substr($chars, 6, 4));
    if (!$data || count($data) < 2) {
      return NULL;
    }

    return array_values($data);
  }

  /**
   * Parses BMP dimensions (needs 29 bytes).
   */
  private function parseSizeForBmp(): ?array {
    $chars = $this->getChars(29);
    if ($chars === NULL || strlen($chars) < 28) {
      return NULL;
    }

    $chars = substr($chars, 14, 14);
    $type = unpack('C', $chars);

    if (!$type) {
      return NULL;
    }

    $format = (reset($type) == 40) ? 'L*' : 'L*';
    $offset = (reset($type) == 40) ? 4 : 4;
    $length = (reset($type) == 40) ? NULL : 8;

    $substr = $length ? substr($chars, $offset, $length) : substr($chars, $offset);
    $data = unpack($format, $substr);

    if (!$data || count($data) < 2) {
      return NULL;
    }

    return array_values($data);
  }

  /**
   * Parses JPEG dimensions (variable bytes, scans for SOF marker).
   */
  private function parseSizeForJpeg(): ?array {
    $state = NULL;
    $skip = 0;

    while (TRUE) {
      switch ($state) {
        default:
          $this->getChars(2);
          $state = 'started';
          break;

        case 'started':
          $b = $this->getByte();
          if ($b === NULL) {
            return NULL;
          }
          $state = ($b == 0xFF) ? 'sof' : 'started';
          break;

        case 'sof':
          $b = $this->getByte();
          if ($b === NULL) {
            return NULL;
          }

          if ($b >= 0xE0 && $b <= 0xEF) {
            $state = 'skipframe';
          }
          elseif (in_array($b, array_merge(
            range(0xC0, 0xC3),
            range(0xC5, 0xC7),
            range(0xC9, 0xCB),
            range(0xCD, 0xCF)
          ))) {
            $state = 'readsize';
          }
          elseif ($b == 0xFF) {
            $state = 'sof';
          }
          else {
            $state = 'skipframe';
          }
          break;

        case 'skipframe':
          $chars = $this->getChars(2);
          if ($chars === NULL) {
            return NULL;
          }
          $skip = $this->readInt($chars) - 2;
          $state = 'doskip';
          break;

        case 'doskip':
          $this->getChars($skip);
          $state = 'started';
          break;

        case 'readsize':
          $c = $this->getChars(7);
          if ($c === NULL || strlen($c) < 7) {
            return NULL;
          }
          return [
            $this->readInt(substr($c, 5, 2)),
            $this->readInt(substr($c, 3, 2)),
          ];
      }
    }
  }

  /**
   * Reads N characters from the file, buffering as needed.
   *
   * @param int $n
   *   Number of characters to read.
   *
   * @return string|null
   *   The characters or NULL on failure.
   */
  private function getChars(int $n): ?string {
    if (!$this->handle) {
      return NULL;
    }

    // Do we need more data?
    if ($this->strpos + $n - 1 >= strlen($this->str)) {
      $end = $this->strpos + $n;

      while (strlen($this->str) < $end) {
        $need = $end - ftell($this->handle);
        $response = fread($this->handle, $need);

        if ($response === FALSE || $response === '') {
          return NULL;
        }

        $this->str .= $response;
      }
    }

    $result = substr($this->str, $this->strpos, $n);
    $this->strpos += $n;

    return $result;
  }

  /**
   * Reads a single byte.
   *
   * @return int|null
   *   The byte value or NULL on failure.
   */
  private function getByte(): ?int {
    $c = $this->getChars(1);
    if ($c === NULL) {
      return NULL;
    }

    $b = unpack("C", $c);
    return is_array($b) ? reset($b) : NULL;
  }

  /**
   * Reads a 2-byte big-endian integer.
   *
   * @param string $str
   *   Two bytes.
   *
   * @return int
   *   The integer value.
   */
  private function readInt(string $str): int {
    $size = unpack("C*", $str);
    return ($size[1] << 8) + $size[2];
  }

  /**
   * Destructor ensures file handle is closed.
   */
  public function __destruct() {
    $this->close();
  }

  /**
   * Gets image dimensions, using FastImage for remote URIs.
   *
   * This is a convenience method that handles the common pattern of:
   * - Using FastImage for remote/S3 files (downloads only header bytes)
   * - Falling back to getimagesize() for local files (fast on local disk)
   *
   * @param string $uri
   *   The file URI (e.g., 's3://bucket/image.jpg' or 'public://image.jpg').
   *
   * @return array
   *   Array with 'width' and 'height' keys, or empty array if not determinable.
   */
  public static function getDimensions(string $uri): array {
    $isRemote = !@stream_is_local($uri);

    // For remote URIs (S3), use FastImage which only downloads header bytes.
    if ($isRemote) {
      $fastImage = new static();
      if ($fastImage->load($uri)) {
        $size = $fastImage->getSize();
        $fastImage->close();

        if ($size && count($size) >= 2) {
          return [
            'width' => (int) $size[0],
            'height' => (int) $size[1],
          ];
        }
      }
    }

    // For local files or as fallback, use getimagesize() which is fast locally.
    $size = @getimagesize($uri);
    if ($size) {
      return [
        'width' => (int) $size[0],
        'height' => (int) $size[1],
      ];
    }

    return [];
  }

}
