<?php

declare(strict_types=1);

namespace Drupal\filepond;

/**
 * Value object representing upload configuration options.
 *
 * This class encapsulates all settings needed for file uploads, providing
 * a type-safe alternative to ad-hoc arrays. It's immutable once created.
 *
 * Used by UploadSettingsResolver to return resolved settings and by
 * FilePondUploadHandler to process uploads.
 */
final class UploadOptions {

  /**
   * Constructs an UploadOptions object.
   *
   * @param array $allowedExtensions
   *   Array of allowed file extensions (lowercase, no dots).
   * @param array $allowedMimeTypes
   *   Array of allowed MIME types.
   * @param string $destination
   *   Upload destination URI (e.g., 'public://media/images').
   * @param int|null $maxSize
   *   Maximum file size in bytes, or NULL for no limit.
   * @param array $context
   *   Additional context for event subscribers.
   */
  public function __construct(
    public readonly array $allowedExtensions,
    public readonly array $allowedMimeTypes,
    public readonly string $destination,
    public readonly ?int $maxSize,
    public readonly array $context,
  ) {}

  /**
   * Converts to array format expected by FilePondUploadHandler.
   *
   * @return array
   *   Handler options array.
   */
  public function toArray(): array {
    $options = [
      'allowed_extensions' => $this->allowedExtensions,
      'allowed_mime_types' => $this->allowedMimeTypes,
      'destination' => $this->destination,
      'context' => $this->context,
    ];

    if ($this->maxSize !== NULL) {
      $options['max_size'] = $this->maxSize;
    }

    return $options;
  }

}
