<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\FunctionalJavascript;

/**
 * Tests chunked FilePond uploads.
 *
 * Chunked uploads follow a different protocol:
 * 1. POST with Upload-Length header → returns transfer ID
 * 2. PATCH requests with chunks → returns file ID on final chunk.
 *
 * These tests verify that file IDs are correctly captured from PATCH responses.
 * This is critical - the CDN version bug (4.31.4) broke this functionality.
 *
 * Uses the built-in test form at /admin/config/media/filepond/test.
 *
 * @group filepond
 * @group filepond_core
 * @group filepond_chunked
 */
class FilePondChunkedUploadTest extends FilePondTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set chunk size to 1MB so our 2MB test files are chunked.
    // Default is 5MB which would make our test files upload as single POST.
    $this->config('filepond.settings')
      ->set('defaults.chunk_size', 1)
      ->save();
  }

  /**
   * Tests chunked upload completes and captures file ID.
   *
   * This is the most critical test - it would have caught the CDN bug.
   */
  public function testChunkedUploadComplete(): void {
    $this->drupalGet('/admin/config/media/filepond/test');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    // Verify chunk configuration is applied (1MB = 1048576 bytes).
    $this->assertChunkSizeConfigured(1048576);

    // Create a large file to force chunked upload (2MB > 1MB chunk size).
    $filePath = $this->createLargeTestFile('large-file', 2);
    $this->dropFileToFilePond($filePath);

    // Wait for chunked upload (longer timeout).
    $this->waitForFilePondComplete('.filepond--root', 60000);
    $this->assertFilesUploadedSuccessfully(1);

    // Submit form and verify file was processed.
    $this->getSession()->getPage()->pressButton('Submit');
    $this->assertSession()->waitForElementVisible('css', '.messages--status', 10000);
    $this->assertSession()->pageTextContains('Uploaded 1 file(s)');
    $this->assertSession()->pageTextContains('large-file');
  }

  /**
   * Tests transfer ID to file ID mapping is populated.
   *
   * The JavaScript must maintain a mapping from transfer IDs to file IDs
   * for chunked uploads because the initial POST returns a transfer ID,
   * but the form needs the final file ID.
   */
  public function testTransferIdMapping(): void {
    $this->drupalGet('/admin/config/media/filepond/test');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();
    $this->assertChunkSizeConfigured(1048576);

    $filePath = $this->createLargeTestFile('chunked-test', 2);
    $this->dropFileToFilePond($filePath);
    $this->waitForFilePondComplete('.filepond--root', 60000);
    $this->assertFilesUploadedSuccessfully(1);

    // Verify JS tracking map was populated.
    $this->assertTrue(
      $this->hasTransferIdMapping(),
      'Transfer ID to file ID mapping must exist after chunked upload'
    );
  }

  /**
   * Tests chunked upload UI shows file.
   *
   * Note: Multiple chunked uploads have the same hidden field issue as
   * regular uploads. This test verifies the UI behavior.
   */
  public function testChunkedUploadUi(): void {
    $this->drupalGet('/admin/config/media/filepond/test');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();
    $this->assertChunkSizeConfigured(1048576);

    // Upload a large file.
    $filePath = $this->createLargeTestFile('large1', 2);
    $this->dropFileToFilePond($filePath);

    // Wait for it to complete (longer timeout for chunked).
    $this->waitForFilePondComplete('.filepond--root', 60000);
    $this->assertFilesUploadedSuccessfully(1);

    // Verify file shows in UI.
    $this->assertEquals(1, $this->getFilePondFileCount(), 'Should have 1 file');
  }

  /**
   * Tests chunked upload removal.
   */
  public function testChunkedUploadRemoval(): void {
    $this->drupalGet('/admin/config/media/filepond/test');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();
    $this->assertChunkSizeConfigured(1048576);

    $filePath = $this->createLargeTestFile('remove-test', 2);
    $this->dropFileToFilePond($filePath);
    $this->waitForFilePondComplete('.filepond--root', 60000);
    $this->assertFilesUploadedSuccessfully(1);

    // Verify file shows in FilePond.
    $this->assertEquals(1, $this->getFilePondFileCount(), 'Should have 1 file');

    // Remove the file via FilePond API.
    $this->getSession()->executeScript(
      "FilePond.find(document.querySelector('.filepond--root')).removeFile();"
    );
    $this->waitForFilePondEmpty();

    // Submit form to verify no files.
    $this->getSession()->getPage()->pressButton('Submit');
    $this->assertSession()->waitForElementVisible('css', '.messages--warning', 10000);
    $this->assertSession()->pageTextContains('No files were uploaded');
  }

}
