<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\FunctionalJavascript;

use Drupal\entity_browser\Element\EntityBrowserElement;
use Drupal\entity_browser\Entity\EntityBrowser;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\media\Entity\MediaType;
use Drupal\node\Entity\NodeType;

/**
 * Tests FilePond Entity Browser widget integration.
 *
 * Tests the filepond_media widget when used in Entity Browser:
 * - Entity browser opens with FilePond widget
 * - Files can be uploaded via FilePond
 * - Media entities are created and selected
 * - Selected media appears in the node form.
 *
 * @group filepond
 * @group filepond_entity_browser
 */
class FilePondEntityBrowserTest extends FilePondTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'file',
    'image',
    'media',
    'entity_browser',
    'filepond',
    'filepond_eb_widget',
  ];

  /**
   * A test user with entity browser permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * The entity browser iframe name.
   *
   * @var string
   */
  protected $iframeName = 'entity_browser_iframe_filepond_eb_test';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create media type.
    $this->createImageMediaType();

    // Create entity browser with FilePond widget.
    $this->createFilePondEntityBrowser();

    // Create node type with media reference field.
    $this->createNodeTypeWithMediaField();

    // Create user with permissions.
    $this->testUser = $this->drupalCreateUser([
      'access content',
      'create eb_test_content content',
      'edit own eb_test_content content',
      'administer media',
      'create media',
      'view media',
      'filepond upload files',
      'access filepond_eb_test entity browser pages',
    ]);
    $this->drupalLogin($this->testUser);
  }

  /**
   * Creates an image media type.
   */
  protected function createImageMediaType(): void {
    if (MediaType::load('eb_test_image')) {
      return;
    }

    $media_type = MediaType::create([
      'id' => 'eb_test_image',
      'label' => 'EB Test Image',
      'source' => 'image',
    ]);
    $media_type->save();

    // Create the source field.
    $source = $media_type->getSource();
    $source_field = $source->createSourceField($media_type);
    $source_field->getFieldStorageDefinition()->save();
    $source_field->save();

    // Set the source field on the media type.
    $media_type->set('source_configuration', [
      'source_field' => $source_field->getName(),
    ]);
    $media_type->save();
  }

  /**
   * Creates an entity browser with FilePond widget.
   */
  protected function createFilePondEntityBrowser(): void {
    if (EntityBrowser::load('filepond_eb_test')) {
      return;
    }

    $entity_browser = EntityBrowser::create([
      'name' => 'filepond_eb_test',
      'label' => 'FilePond EB Test',
      'display' => 'iframe',
      'display_configuration' => [
        'width' => '650',
        'height' => '500',
        'link_text' => 'Select media',
        'auto_open' => FALSE,
      ],
      'selection_display' => 'no_display',
      'selection_display_configuration' => [],
      'widget_selector' => 'single',
      'widget_selector_configuration' => [],
      'widgets' => [
        'filepond_widget_uuid' => [
          'uuid' => 'filepond_widget_uuid',
          'id' => 'filepond_media',
          'label' => 'FilePond Upload',
          'weight' => 0,
          'settings' => [
            'media_type' => 'eb_test_image',
            'upload_location' => 'public://test-eb-uploads',
            'extensions' => 'png gif jpg jpeg webp',
            'max_filesize' => '10M',
            'submit_text' => 'Select media',
            'auto_select' => FALSE,
          ],
        ],
      ],
    ]);
    $entity_browser->save();

    // Rebuild routes after creating entity browser.
    \Drupal::service('router.builder')->rebuild();
  }

  /**
   * Creates a node type with a media reference field using Entity Browser.
   */
  protected function createNodeTypeWithMediaField(): void {
    // Create node type.
    if (!NodeType::load('eb_test_content')) {
      NodeType::create([
        'type' => 'eb_test_content',
        'name' => 'EB Test Content',
      ])->save();
    }

    // Create media reference field storage.
    if (!FieldStorageConfig::loadByName('node', 'field_eb_media')) {
      FieldStorageConfig::create([
        'field_name' => 'field_eb_media',
        'entity_type' => 'node',
        'type' => 'entity_reference',
        'cardinality' => -1,
        'settings' => [
          'target_type' => 'media',
        ],
      ])->save();
    }

    // Create field instance.
    if (!FieldConfig::loadByName('node', 'eb_test_content', 'field_eb_media')) {
      FieldConfig::create([
        'field_name' => 'field_eb_media',
        'entity_type' => 'node',
        'bundle' => 'eb_test_content',
        'label' => 'Media',
        'settings' => [
          'handler' => 'default:media',
          'handler_settings' => [
            'target_bundles' => ['eb_test_image' => 'eb_test_image'],
          ],
        ],
      ])->save();
    }

    // Configure form display to use Entity Browser widget.
    /** @var \Drupal\Core\Entity\Display\EntityFormDisplayInterface $form_display */
    $form_display = $this->container->get('entity_type.manager')
      ->getStorage('entity_form_display')
      ->load('node.eb_test_content.default');

    if (!$form_display) {
      $form_display = $this->container->get('entity_type.manager')
        ->getStorage('entity_form_display')
        ->create([
          'targetEntityType' => 'node',
          'bundle' => 'eb_test_content',
          'mode' => 'default',
          'status' => TRUE,
        ]);
    }

    $form_display->setComponent('field_eb_media', [
      'type' => 'entity_browser_entity_reference',
      'settings' => [
        'entity_browser' => 'filepond_eb_test',
        'field_widget_display' => 'label',
        'field_widget_edit' => TRUE,
        'field_widget_remove' => TRUE,
        'field_widget_replace' => FALSE,
        'open' => FALSE,
        'selection_mode' => EntityBrowserElement::SELECTION_MODE_APPEND,
      ],
    ])->save();
  }

  /**
   * Tests that Entity Browser opens with FilePond widget.
   */
  public function testEntityBrowserOpensWithFilePond(): void {
    $this->drupalGet('/node/add/eb_test_content');

    // Check page loaded.
    $pageText = $this->getSession()->getPage()->getText();
    $this->assertStringContainsString('Create EB Test Content', $pageText, 'Node form should load');

    // Click the link to open the entity browser via JavaScript.
    $this->assertSession()->waitForElement('css', 'a[data-drupal-selector*="entity-browser-link"]', 5000);
    $this->getSession()->executeScript(
      "document.querySelector('a[data-drupal-selector*=\"entity-browser-link\"]').click();"
    );

    // Wait for the iframe to appear.
    $iframe = $this->assertSession()->waitForElement('css', 'iframe[name="' . $this->iframeName . '"]', 10000);
    $this->assertNotNull($iframe, 'Entity Browser iframe should appear');

    // Wait for iframe to load.
    usleep(2000000);

    // Switch to the iframe.
    $this->getSession()->switchToIFrame($this->iframeName);

    // Wait for form to load in iframe.
    $this->assertSession()->waitForElement('css', 'form', 15000);

    // Wait for FilePond to initialize in the iframe.
    $filepond = $this->assertSession()->waitForElement('css', '.filepond--root', 15000);
    $this->assertNotNull($filepond, 'FilePond should be present in Entity Browser');

    // Verify FilePond JS is loaded.
    $jsReady = $this->getSession()->wait(5000, 'typeof FilePond !== "undefined"');
    $this->assertTrue($jsReady, 'FilePond JS should be loaded in iframe');
  }

  /**
   * Tests uploading an image via FilePond in Entity Browser.
   */
  public function testEntityBrowserUpload(): void {
    $this->drupalGet('/node/add/eb_test_content');

    // Open Entity Browser.
    $this->openEntityBrowser();

    // Create and upload a test file.
    $filePath = $this->createTestFile('eb-test-image');
    $this->dropFileToFilePond($filePath);

    // Wait for upload to complete.
    $this->waitForFilePondComplete();

    // Verify file shows in FilePond.
    $this->assertEquals(1, $this->getFilePondFileCount());
  }

  /**
   * Tests selecting uploaded media in Entity Browser.
   */
  public function testEntityBrowserSelectMedia(): void {
    $this->drupalGet('/node/add/eb_test_content');

    // Open Entity Browser.
    $this->openEntityBrowser();

    // Upload a file.
    $filePath = $this->createTestFile('eb-select-test');
    $this->dropFileToFilePond($filePath);
    $this->waitForFilePondComplete();

    // Click the select button via JavaScript.
    $this->getSession()->executeScript(
      "document.querySelector('input[value=\"Select media\"]').click();"
    );

    // Wait for AJAX and switch back to main page.
    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->getSession()->switchToIFrame();

    // Wait for the iframe to be removed (EB closes).
    $this->assertSession()->waitForElementRemoved(
      'css',
      'iframe[name="' . $this->iframeName . '"]',
      10000
    );

    // Verify media item appears in the field widget.
    $mediaItem = $this->assertSession()->waitForElement('css', '.entities-list', 10000);
    $this->assertNotNull($mediaItem, 'Selected media should appear in the field');
  }

  /**
   * Tests multiple file uploads in Entity Browser.
   */
  public function testEntityBrowserMultipleUploads(): void {
    $this->drupalGet('/node/add/eb_test_content');

    // Open Entity Browser.
    $this->openEntityBrowser();

    // Upload first file and wait.
    $file1 = $this->createTestFile('eb-multi-1');
    $this->dropFileToFilePond($file1);
    $this->waitForFilePondComplete();

    // Upload second file and wait.
    $file2 = $this->createTestFile('eb-multi-2');
    $this->dropFileToFilePond($file2);
    $this->waitForFilePondFileCount(2);

    // Verify both files show in FilePond UI.
    $this->assertEquals(2, $this->getFilePondFileCount(), 'Two files should display in FilePond');
  }

  /**
   * Tests file removal before selection in Entity Browser.
   */
  public function testEntityBrowserFileRemoval(): void {
    $this->drupalGet('/node/add/eb_test_content');

    // Open Entity Browser.
    $this->openEntityBrowser();

    // Upload a file.
    $filePath = $this->createTestFile('eb-removal-test');
    $this->dropFileToFilePond($filePath);
    $this->waitForFilePondComplete();

    // Verify file is in FilePond.
    $this->assertEquals(1, $this->getFilePondFileCount());

    // Remove the file via FilePond API.
    $this->getSession()->executeScript(
      "FilePond.find(document.querySelector('.filepond--root')).removeFile();"
    );

    // Wait for removal.
    $this->waitForFilePondEmpty();
    $this->assertEquals(0, $this->getFilePondFileCount());
  }

  /**
   * Tests chunked upload in Entity Browser.
   *
   * This test verifies that large files requiring chunked upload work
   * correctly in the Entity Browser iframe context.
   */
  public function testEntityBrowserChunkedUpload(): void {
    // Set smaller chunk size to ensure chunking happens.
    $this->config('filepond.settings')
      ->set('defaults.chunk_size', 1)
      ->save();

    $this->drupalGet('/node/add/eb_test_content');

    // Open Entity Browser.
    $this->openEntityBrowser();

    // Create and upload a large test file (2MB > 1MB chunk size).
    $filePath = $this->createLargeTestFile('eb-chunked-test', 2);
    $this->dropFileToFilePond($filePath);

    // Wait for chunked upload to complete (longer timeout).
    $this->waitForFilePondComplete('.filepond--root', 60000);

    // Verify file shows in FilePond.
    $this->assertEquals(1, $this->getFilePondFileCount());

    // Verify the file state indicates successful processing.
    $fileState = $this->getSession()->evaluateScript(
      "return FilePond.find(document.querySelector('.filepond--root'))?.getFile()?.status || 0;"
    );
    // FileStatus.PROCESSING_COMPLETE = 5.
    $this->assertEquals(5, $fileState, 'Chunked upload should complete successfully');
  }

  /**
   * Opens the Entity Browser and switches to its iframe.
   */
  protected function openEntityBrowser(): void {
    // Click the link to open the entity browser via JavaScript.
    $this->assertSession()->waitForElement('css', 'a[data-drupal-selector*="entity-browser-link"]', 5000);
    $this->getSession()->executeScript(
      "document.querySelector('a[data-drupal-selector*=\"entity-browser-link\"]').click();"
    );

    // Wait for the iframe to appear.
    $this->assertSession()->waitForElement(
      'css',
      'iframe[name="' . $this->iframeName . '"]',
      10000
    );

    // Wait for iframe to load.
    usleep(2000000);

    // Switch to the iframe.
    $this->getSession()->switchToIFrame($this->iframeName);

    // Wait for form to load.
    $this->assertSession()->waitForElement('css', 'form', 15000);

    // Wait for FilePond to initialize.
    $this->waitForFilePondInit();
  }

  /**
   * {@inheritdoc}
   *
   * Override to handle Entity Browser iframe context.
   */
  protected function waitForFilePondInit(int $timeout = 10000): void {
    // Wait for the FilePond wrapper element.
    $wrapper = $this->assertSession()->waitForElement('css', '.filepond-wrapper', $timeout);
    if (!$wrapper) {
      // In EB iframe, the structure might be different.
      $wrapper = $this->assertSession()->waitForElement('css', '.filepond--root', $timeout);
    }
    $this->assertNotNull($wrapper, 'FilePond wrapper element should exist in Entity Browser');

    // Wait for FilePond JS global to be available.
    $jsReady = $this->getSession()->wait($timeout, 'typeof FilePond !== "undefined"');
    $this->assertTrue($jsReady, 'FilePond JS global should be available');
  }

}
