<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\FunctionalJavascript;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\media\Entity\MediaType;
use Drupal\node\Entity\NodeType;

/**
 * Tests FilePond integration with Media Library.
 *
 * When enabled, FilePond replaces the default file upload widget in
 * the Media Library modal. These tests verify that:
 * - Files can be uploaded via FilePond in the modal
 * - Media entities are created correctly
 * - The auto-select functionality works.
 *
 * @group filepond
 * @group filepond_media_library
 */
class FilePondMediaLibraryTest extends FilePondTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'file',
    'image',
    'media',
    'media_library',
    'filepond',
  ];

  /**
   * A test user with media permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Enable FilePond for Media Library.
    // This must happen before media type creation to ensure the alter hook
    // registers the FilePond form.
    $this->config('filepond.settings')
      ->set('enable_media_library_widget', TRUE)
      ->save();

    // Clear caches to ensure the media source info alter hook fires
    // with the new config value.
    drupal_flush_all_caches();

    // Create an image media type.
    $this->createImageMediaType();

    // Create a node type with a media reference field.
    $this->createNodeTypeWithMediaField();

    // Create admin user with full permissions.
    $this->adminUser = $this->drupalCreateUser([
      'access content',
      'create article content',
      'edit own article content',
      'administer media',
      'create media',
      'view media',
      'filepond upload files',
    ]);
    $this->drupalLogin($this->adminUser);
  }

  /**
   * Creates an image media type.
   */
  protected function createImageMediaType(): void {
    // Check if the media type already exists.
    if (MediaType::load('image')) {
      return;
    }

    $media_type = MediaType::create([
      'id' => 'image',
      'label' => 'Image',
      'source' => 'image',
    ]);
    $media_type->save();

    // Create the source field.
    $source = $media_type->getSource();
    $source_field = $source->createSourceField($media_type);
    $source_field->getFieldStorageDefinition()->save();
    $source_field->save();

    // Set the source field on the media type.
    $media_type->set('source_configuration', [
      'source_field' => $source_field->getName(),
    ]);
    $media_type->save();
  }

  /**
   * Creates a node type with a media reference field.
   */
  protected function createNodeTypeWithMediaField(): void {
    // Create article node type.
    if (!NodeType::load('article')) {
      NodeType::create([
        'type' => 'article',
        'name' => 'Article',
      ])->save();
    }

    // Create media reference field storage.
    if (!FieldStorageConfig::loadByName('node', 'field_media')) {
      FieldStorageConfig::create([
        'field_name' => 'field_media',
        'entity_type' => 'node',
        'type' => 'entity_reference',
        'cardinality' => -1,
        'settings' => [
          'target_type' => 'media',
        ],
      ])->save();
    }

    // Create field instance.
    if (!FieldConfig::loadByName('node', 'article', 'field_media')) {
      FieldConfig::create([
        'field_name' => 'field_media',
        'entity_type' => 'node',
        'bundle' => 'article',
        'label' => 'Media',
        'settings' => [
          'handler' => 'default:media',
          'handler_settings' => [
            'target_bundles' => ['image' => 'image'],
          ],
        ],
      ])->save();
    }

    // Configure the form display to use Media Library widget.
    /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $display_repository */
    $display_repository = \Drupal::service('entity_display.repository');
    $display_repository->getFormDisplay('node', 'article')
      ->setComponent('field_media', [
        'type' => 'media_library_widget',
        'settings' => [
          'media_types' => ['image'],
        ],
      ])
      ->save();
  }

  /**
   * Tests that FilePond appears in Media Library modal.
   */
  public function testFilePondInMediaLibrary(): void {
    $this->drupalGet('/node/add/article');

    // Open the Media Library modal.
    $this->getSession()->getPage()->pressButton('Add media');

    // Wait for the modal to open.
    $modal = $this->assertSession()->waitForElementVisible('css', '.media-library-widget-modal');
    $this->assertNotNull($modal, 'Media Library modal should open');

    // Wait for AJAX to complete and modal content to load.
    $this->assertSession()->assertWaitOnAjaxRequest();

    // The add form should be visible (uses .js-media-library-add-form).
    // When only one media type exists and no media, it shows directly.
    $addForm = $this->assertSession()->waitForElement('css', '.js-media-library-add-form', 10000);
    $this->assertNotNull($addForm, 'Media Library add form should appear');

    // Wait for FilePond to initialize in the modal.
    $this->assertSession()->assertWaitOnAjaxRequest();

    $filepond = $this->assertSession()->waitForElement('css', '.js-media-library-add-form .filepond--root', 15000);
    $this->assertNotNull($filepond, 'FilePond should be present in Media Library modal');

    // Verify FilePond JS is loaded.
    $jsReady = $this->getSession()->wait(5000, 'typeof FilePond !== "undefined"');
    $this->assertTrue($jsReady, 'FilePond JS should be loaded');
  }

  /**
   * Tests uploading an image via FilePond in Media Library.
   */
  public function testMediaLibraryUpload(): void {
    $this->drupalGet('/node/add/article');

    // Open Media Library modal.
    $this->openMediaLibraryModal();

    $selector = '.js-media-library-add-form .filepond--root';
    $filepond = $this->assertSession()->waitForElement('css', $selector, 15000);
    $this->assertNotNull($filepond, 'FilePond should appear in modal');

    // Wait for FilePond JS to be ready.
    $this->waitForFilePondInit();

    // Create and upload a test file.
    $filePath = $this->createTestFile('media-test-image');
    $this->dropFileToFilePond($filePath, $selector);

    // Wait for upload to complete.
    $this->waitForFilePondComplete($selector);

    // Wait for AJAX processing to complete.
    $this->assertSession()->assertWaitOnAjaxRequest();

    // The auto-select should trigger, showing the media item form.
    // Wait for either the media fields form or the selection area.
    $result = $this->getSession()->wait(15000,
      'document.querySelector("[data-drupal-selector*=\"edit-media-0-fields\"]") !== null || ' .
      'document.querySelector(".js-media-library-item") !== null'
    );
    $this->assertTrue($result, 'Media form or selection should appear after upload');
  }

  /**
   * Opens Media Library modal and navigates to the Image upload form.
   */
  protected function openMediaLibraryModal(): void {
    $this->getSession()->getPage()->pressButton('Add media');
    $this->assertSession()->waitForElementVisible('css', '.media-library-widget-modal');
    $this->assertSession()->assertWaitOnAjaxRequest();

    // Click on the "Image" tab to get to the image upload form.
    // The tab is in the .js-media-library-menu element.
    $page = $this->getSession()->getPage();
    $imageTab = $page->find('named', ['link', 'Image']);
    if ($imageTab) {
      $imageTab->click();
      $this->assertSession()->assertWaitOnAjaxRequest();
    }

    // Wait for the add form to appear.
    $this->assertSession()->waitForElement('css', '.js-media-library-add-form');
  }

  /**
   * Tests that uploaded media can be saved and selected.
   */
  public function testMediaLibrarySaveAndInsert(): void {
    $this->drupalGet('/node/add/article');

    // Open Media Library modal.
    $this->openMediaLibraryModal();

    $selector = '.js-media-library-add-form .filepond--root';
    $filepond = $this->assertSession()->waitForElement('css', $selector, 15000);
    $this->assertNotNull($filepond, 'FilePond should appear in modal');

    $this->waitForFilePondInit();

    // Upload a file.
    $filePath = $this->createTestFile('save-insert-test');
    $this->dropFileToFilePond($filePath, $selector);
    $this->waitForFilePondComplete($selector);

    // Wait for auto-select AJAX.
    $this->assertSession()->assertWaitOnAjaxRequest();

    // Wait for the media item form to appear.
    $fieldsForm = $this->assertSession()->waitForElement(
      'css',
      '[data-drupal-selector*="edit-media-0-fields"]',
      15000
    );

    if ($fieldsForm) {
      // Click "Save and insert" button in the dialog.
      $dialog = $this->assertSession()->elementExists('css', '.ui-dialog-buttonpane');
      $saveButton = $dialog->findButton('Save and insert');
      if ($saveButton) {
        $saveButton->press();

        // Wait for modal to close.
        $this->assertSession()->waitForElementRemoved('css', '.media-library-widget-modal');

        // Verify media item appears in the field.
        $mediaItem = $this->assertSession()->waitForElement('css', '.js-media-library-item');
        $this->assertNotNull($mediaItem, 'Media item should appear in field after insert');
      }
    }
  }

  /**
   * Tests file removal in Media Library before saving.
   */
  public function testMediaLibraryFileRemoval(): void {
    $this->drupalGet('/node/add/article');

    // Open Media Library modal.
    $this->openMediaLibraryModal();

    $selector = '.js-media-library-add-form .filepond--root';
    $filepond = $this->assertSession()->waitForElement('css', $selector, 15000);
    $this->assertNotNull($filepond, 'FilePond should appear in modal');

    $this->waitForFilePondInit();

    // Upload a file.
    $filePath = $this->createTestFile('removal-test');
    $this->dropFileToFilePond($filePath, $selector);
    $this->waitForFilePondComplete($selector);

    // Verify file is in FilePond.
    $this->assertEquals(1, $this->getFilePondFileCount($selector));

    // Remove the file via FilePond API.
    $this->getSession()->executeScript(
      "FilePond.find(document.querySelector('$selector')).removeFile();"
    );

    // Wait for removal.
    $this->waitForFilePondEmpty($selector);
    $this->assertEquals(0, $this->getFilePondFileCount($selector));
  }

  /**
   * {@inheritdoc}
   *
   * Override to handle Media Library context.
   */
  protected function waitForFilePondInit(int $timeout = 10000): void {
    // Wait for the FilePond wrapper element.
    $wrapper = $this->assertSession()->waitForElement('css', '.filepond-wrapper', $timeout);
    if (!$wrapper) {
      // In Media Library, the structure might be different.
      $wrapper = $this->assertSession()->waitForElement('css', '.filepond--root', $timeout);
    }
    $this->assertNotNull($wrapper, 'FilePond wrapper element should exist');

    // Wait for FilePond JS global to be available.
    $jsReady = $this->getSession()->wait($timeout, 'typeof FilePond !== "undefined"');
    $this->assertTrue($jsReady, 'FilePond JS global should be available');
  }

}
