<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\FunctionalJavascript;

/**
 * Tests regular (non-chunked) FilePond uploads.
 *
 * Uses the built-in test form at /admin/config/media/filepond/test.
 *
 * @group filepond
 * @group filepond_core
 */
class FilePondRegularUploadTest extends FilePondTestBase {

  /**
   * Tests single file upload creates a file entity.
   */
  public function testSingleFileUpload(): void {
    $this->drupalGet('/admin/config/media/filepond/test');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    // Create and upload a test file.
    $filePath = $this->createTestFile('test-image');
    $this->dropFileToFilePond($filePath);
    $this->waitForFilePondComplete();
    $this->assertFilesUploadedSuccessfully(1);

    // Submit the form and wait for response.
    $this->getSession()->getPage()->pressButton('Submit');

    // Wait for status message to appear.
    $this->assertSession()->waitForElementVisible('css', '.messages--status', 10000);

    // Check status message for the uploaded file.
    $this->assertSession()->pageTextContains('Uploaded 1 file(s)');
    $this->assertSession()->pageTextContains('test-image');
    $this->assertSession()->pageTextContains('ID:');
  }

  /**
   * Tests multiple file uploads display in FilePond UI.
   *
   * Note: There's a known issue where the hidden field may not capture
   * all file IDs when files are uploaded rapidly. This test verifies
   * the UI behavior. The hidden field bug is tracked separately.
   */
  public function testMultipleFileUpload(): void {
    $this->drupalGet('/admin/config/media/filepond/test');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    // Upload first file and wait for it to complete.
    $file1 = $this->createTestFile('image1');
    $this->dropFileToFilePond($file1);
    $this->waitForFilePondComplete();
    $this->assertFilesUploadedSuccessfully(1);

    // Verify first file is complete.
    $this->assertEquals(1, $this->getFilePondFileCount(), 'Should have 1 file after first upload');

    // Upload second file and wait for it to complete.
    $file2 = $this->createTestFile('image2');
    $this->dropFileToFilePond($file2);
    $this->waitForFilePondFileCount(2);

    // Wait for both to have processing-complete state.
    $this->assertSession()->waitForElementVisible(
      'css',
      '.filepond--item:nth-child(2)[data-filepond-item-state="processing-complete"]',
      30000
    );

    // Verify both files show in FilePond UI.
    $this->assertEquals(2, $this->getFilePondFileCount(), 'Should have 2 files');
  }

  /**
   * Tests file removal updates hidden field.
   */
  public function testFileRemoval(): void {
    $this->drupalGet('/admin/config/media/filepond/test');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    // Upload a file.
    $filePath = $this->createTestFile('test-image');
    $this->dropFileToFilePond($filePath);
    $this->waitForFilePondComplete();
    $this->assertFilesUploadedSuccessfully(1);

    // Verify file shows in FilePond.
    $this->assertEquals(1, $this->getFilePondFileCount(), 'Should have 1 file');

    // Remove the file via FilePond API (more reliable than click).
    $this->getSession()->executeScript(
      "FilePond.find(document.querySelector('.filepond--root')).removeFile();"
    );

    // Wait for item to be removed.
    $this->waitForFilePondEmpty();

    // Submit form to verify no files.
    $this->getSession()->getPage()->pressButton('Submit');
    $this->assertSession()->waitForElementVisible('css', '.messages--warning', 10000);
    $this->assertSession()->pageTextContains('No files were uploaded');
  }

  /**
   * Tests FilePond shows correct number of items.
   */
  public function testFilePondItemCount(): void {
    $this->drupalGet('/admin/config/media/filepond/test');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    $this->assertEquals(0, $this->getFilePondFileCount(), 'Should start with no files');

    // Add first file.
    $file1 = $this->createTestFile('image1');
    $this->dropFileToFilePond($file1);
    $this->waitForFilePondComplete();
    $this->assertFilesUploadedSuccessfully(1);
    $this->assertEquals(1, $this->getFilePondFileCount(), 'Should have 1 file');

    // Add second file (small delay to avoid rate limiting in CI).
    usleep(500000);
    $file2 = $this->createTestFile('image2');
    $this->dropFileToFilePond($file2);
    $this->waitForFilePondFileCount(2);
    $this->assertFilesUploadedSuccessfully(2);
    $this->assertEquals(2, $this->getFilePondFileCount(), 'Should have 2 files');
  }

}
