<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\Kernel;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;
use Drupal\user\RoleInterface;

/**
 * Tests the FilePond image field widget.
 *
 * @group filepond
 */
class FilePondImageWidgetTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'field',
    'file',
    'image',
    'user',
    'node',
    'text',
    'filter',
    'filepond',
  ];

  /**
   * A test user with permission to upload files.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['system', 'field', 'node', 'filepond', 'filter']);

    // Create authenticated role with filepond permission.
    $role = Role::create([
      'id' => RoleInterface::AUTHENTICATED_ID,
      'label' => 'Authenticated',
    ]);
    $role->grantPermission('filepond upload files');
    $role->save();

    // Create a test user.
    $this->testUser = User::create([
      'name' => 'test_user',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $this->testUser->save();
    $this->container->get('current_user')->setAccount($this->testUser);

    // Create a content type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Create an image field on the article content type.
    FieldStorageConfig::create([
      'field_name' => 'field_image',
      'entity_type' => 'node',
      'type' => 'image',
      'cardinality' => 1,
    ])->save();

    FieldConfig::create([
      'entity_type' => 'node',
      'bundle' => 'article',
      'field_name' => 'field_image',
      'label' => 'Image',
      'settings' => [
        'file_extensions' => 'png gif jpg jpeg',
        'max_filesize' => '5M',
      ],
    ])->save();

    // Configure form display to use FilePond widget.
    /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $display_repository */
    $display_repository = $this->container->get('entity_display.repository');
    $display_repository->getFormDisplay('node', 'article')
      ->setComponent('field_image', [
        'type' => 'filepond_image',
        'settings' => [
          'preview_image_style' => 'thumbnail',
          'upload_prompt' => '',
          'allow_reorder' => FALSE,
          'item_panel_aspect_ratio' => '1:1',
          'preview_fit_mode' => 'contain',
        ],
      ])
      ->save();
  }

  /**
   * Tests that the FilePond image widget renders on node form.
   */
  public function testImageWidgetRendering(): void {
    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);
    $rendered = $this->render($form);

    // The form should render without errors.
    $this->assertNotEmpty($rendered, 'Form should render successfully.');

    // Field wrapper should be present.
    $this->assertNotEmpty(
      $this->xpath("//div[contains(@class, 'field--name-field-image')]"),
      'Image field wrapper should be present.'
    );

    // FilePond wrapper should be present.
    $this->assertNotEmpty(
      $this->xpath("//div[contains(@class, 'filepond-wrapper')]"),
      'FilePond wrapper should be present inside image field.'
    );

    // File input element should be present.
    $this->assertNotEmpty(
      $this->xpath("//input[@type='file'][contains(@class, 'filepond--input')]"),
      'FilePond file input should be present.'
    );

    // Hidden fids input should be present.
    $this->assertNotEmpty(
      $this->xpath("//input[@type='hidden'][contains(@name, 'field_image')][contains(@name, '[fids]')]"),
      'Hidden fids input should be present for image field.'
    );
  }

  /**
   * Tests that field settings are passed to the widget.
   */
  public function testImageWidgetSettings(): void {
    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);

    // Find the filepond element within the field.
    $element = $form['field_image']['widget'][0] ?? NULL;
    $this->assertNotNull($element, 'Widget element should exist.');

    // Check that extensions are set from field config.
    $this->assertStringContainsString('png', $element['#extensions'] ?? '');
    $this->assertStringContainsString('jpg', $element['#extensions'] ?? '');
  }

  /**
   * Tests widget with multi-value image field.
   */
  public function testMultiValueImageWidget(): void {
    // Update field storage to allow multiple values.
    $field_storage = FieldStorageConfig::loadByName('node', 'field_image');
    $field_storage->setCardinality(5);
    $field_storage->save();

    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);
    $rendered = $this->render($form);

    // Form should still render.
    $this->assertNotEmpty($rendered, 'Form with multi-value field should render.');

    // FilePond should still be present (handles multi-value internally).
    $this->assertNotEmpty(
      $this->xpath("//div[contains(@class, 'filepond-wrapper')]"),
      'FilePond wrapper should be present for multi-value field.'
    );
  }

  /**
   * Tests widget renders without permission (preview mode).
   */
  public function testWidgetWithoutPermission(): void {
    // Remove permission from authenticated role.
    $role = Role::load(RoleInterface::AUTHENTICATED_ID);
    $role->revokePermission('filepond upload files');
    $role->save();

    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);
    $rendered = $this->render($form);

    // Form should still render (with warning message instead of uploader).
    $this->assertNotEmpty($rendered, 'Form should render even without permission.');

    // Should show warning message about no permission.
    $this->assertNotEmpty(
      $this->xpath("//*[contains(@class, 'messages--warning')]"),
      'Warning message should be shown when user lacks permission.'
    );
  }

  /**
   * Tests that reorder setting is passed to config.
   */
  public function testReorderSetting(): void {
    // Update form display to enable reordering.
    /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $display_repository */
    $display_repository = $this->container->get('entity_display.repository');
    $display_repository->getFormDisplay('node', 'article')
      ->setComponent('field_image', [
        'type' => 'filepond_image',
        'settings' => [
          'allow_reorder' => TRUE,
        ],
      ])
      ->save();

    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);

    $element = $form['field_image']['widget'][0] ?? NULL;
    $this->assertNotNull($element, 'Widget element should exist.');

    // Check that allowReorder is set in config.
    $this->assertTrue(
      $element['#config']['allowReorder'] ?? FALSE,
      'allowReorder should be TRUE in widget config.'
    );
  }

}
