<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\Kernel;

use Drupal\filepond\Element\FilePond;
use Drupal\filepond\UploadOptions;
use Drupal\filepond\UploadSettingsResolverInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\media\Entity\MediaType;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Tests the UploadSettingsResolver service.
 *
 * @group filepond
 * @coversDefaultClass \Drupal\filepond\UploadSettingsResolver
 */
class UploadSettingsResolverTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'file',
    'user',
    'field',
    'image',
    'media',
    'views',
    'filepond',
  ];

  /**
   * The resolver service under test.
   */
  protected UploadSettingsResolverInterface $resolver;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('media');
    $this->installConfig(['filepond', 'media']);

    $this->resolver = $this->container->get('filepond.settings_resolver');
  }

  /**
   * Stores form config in State API.
   *
   * @param string $form_id
   *   The form ID.
   * @param string $element_name
   *   The element name.
   * @param array $config
   *   Upload configuration.
   */
  protected function storeFormConfig(string $form_id, string $element_name, array $config): void {
    $key = FilePond::STATE_PREFIX . ":{$form_id}:{$element_name}";
    $this->container->get('state')->set($key, $config);
  }

  /**
   * Tests that resolveFromState returns NULL for non-existent config.
   *
   * @covers ::resolveFromState
   */
  public function testResolveFromStateReturnsNullForNonExistent(): void {
    $result = $this->resolver->resolveFromState('nonexistent_form', 'nonexistent_element');
    $this->assertNull($result);
  }

  /**
   * Tests that resolveFromState returns UploadOptions for valid config.
   *
   * @covers ::resolveFromState
   */
  public function testResolveFromStateReturnsOptionsForValidConfig(): void {
    // Store config using the helper method.
    $this->storeFormConfig('test_form', 'upload_field', [
      'extensions' => 'jpg png gif',
      'max_filesize' => '10M',
      'upload_location' => 'public://uploads',
      'context' => ['custom_key' => 'custom_value'],
    ]);

    $result = $this->resolver->resolveFromState('test_form', 'upload_field');

    $this->assertInstanceOf(UploadOptions::class, $result);
    $this->assertEquals(['jpg', 'png', 'gif'], $result->allowedExtensions);
    $this->assertEquals('public://uploads', $result->destination);
    $this->assertEquals(10 * 1024 * 1024, $result->maxSize);
    $this->assertArrayHasKey('custom_key', $result->context);
    $this->assertEquals('custom_value', $result->context['custom_key']);
  }

  /**
   * Tests that resolveFromField throws for invalid entity type.
   *
   * @covers ::resolveFromField
   */
  public function testResolveFromFieldThrowsForInvalidEntityType(): void {
    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('Entity type not found');

    $this->resolver->resolveFromField('nonexistent_entity', 'bundle', 'field_name');
  }

  /**
   * Tests that resolveFromField throws for non-existent field.
   *
   * @covers ::resolveFromField
   */
  public function testResolveFromFieldThrowsForNonExistentField(): void {
    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('Field not found');

    // 'user' entity type exists, but this field doesn't.
    $this->resolver->resolveFromField('user', 'user', 'nonexistent_field');
  }

  /**
   * Tests that resolveFromMediaType throws for non-existent media type.
   *
   * @covers ::resolveFromMediaType
   */
  public function testResolveFromMediaTypeThrowsForNonExistent(): void {
    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('Media type not found');

    $this->resolver->resolveFromMediaType('nonexistent_media_type');
  }

  /**
   * Tests that resolveFromMediaType returns options for valid media type.
   *
   * @covers ::resolveFromMediaType
   */
  public function testResolveFromMediaTypeReturnsOptions(): void {
    // Create a media type with image source.
    $this->createImageMediaType('test_image');

    $result = $this->resolver->resolveFromMediaType('test_image');

    $this->assertInstanceOf(UploadOptions::class, $result);
    $this->assertNotEmpty($result->allowedExtensions);
    $this->assertContains('png', $result->allowedExtensions);
    $this->assertStringStartsWith('public://', $result->destination);
    $this->assertEquals('test_image', $result->context['media_type']);
  }

  /**
   * Tests that resolveFromViewsArea throws for non-existent view.
   *
   * Note: This test requires the views module which adds complexity.
   * The views-specific functionality is tested via controller tests when
   * the filepond_views submodule is enabled. Here we just verify the
   * resolver handles the case when views module is not available.
   *
   * @covers ::resolveFromViewsArea
   */
  public function testResolveFromViewsAreaThrowsForNonExistentView(): void {
    // When views module is not installed, Views::getView() returns NULL.
    // The resolver should throw NotFoundHttpException.
    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('View not found');

    $this->resolver->resolveFromViewsArea('nonexistent_view', 'default');
  }

  /**
   * Tests UploadOptions toArray method.
   *
   * @covers \Drupal\filepond\UploadOptions::toArray
   */
  public function testUploadOptionsToArray(): void {
    $options = new UploadOptions(
      allowedExtensions: ['jpg', 'png'],
      allowedMimeTypes: ['image/jpeg', 'image/png'],
      destination: 'public://test',
      maxSize: 5242880,
      context: ['key' => 'value'],
    );

    $array = $options->toArray();

    $this->assertIsArray($array);
    $this->assertEquals(['jpg', 'png'], $array['allowed_extensions']);
    $this->assertEquals(['image/jpeg', 'image/png'], $array['allowed_mime_types']);
    $this->assertEquals('public://test', $array['destination']);
    $this->assertEquals(5242880, $array['max_size']);
    $this->assertEquals(['key' => 'value'], $array['context']);
  }

  /**
   * Tests UploadOptions toArray omits max_size when NULL.
   *
   * @covers \Drupal\filepond\UploadOptions::toArray
   */
  public function testUploadOptionsToArrayOmitsNullMaxSize(): void {
    $options = new UploadOptions(
      allowedExtensions: ['jpg'],
      allowedMimeTypes: ['image/jpeg'],
      destination: 'public://test',
      maxSize: NULL,
      context: [],
    );

    $array = $options->toArray();

    $this->assertArrayNotHasKey('max_size', $array);
  }

  /**
   * Tests that MIME types are correctly derived from extensions.
   *
   * @covers ::resolveFromState
   */
  public function testMimeTypesAreDerivedFromExtensions(): void {
    $this->storeFormConfig('mime_test_form', 'field', [
      'extensions' => 'jpg png pdf',
      'upload_location' => 'public://test',
    ]);

    $result = $this->resolver->resolveFromState('mime_test_form', 'field');

    $this->assertContains('image/jpeg', $result->allowedMimeTypes);
    $this->assertContains('image/png', $result->allowedMimeTypes);
    $this->assertContains('application/pdf', $result->allowedMimeTypes);
  }

  /**
   * Tests that max filesize string is converted to bytes.
   *
   * @covers ::resolveFromState
   */
  public function testMaxFilesizeConversion(): void {
    $this->storeFormConfig('size_test_form', 'field', [
      'extensions' => 'jpg',
      'max_filesize' => '5M',
      'upload_location' => 'public://test',
    ]);

    $result = $this->resolver->resolveFromState('size_test_form', 'field');

    $this->assertEquals(5 * 1024 * 1024, $result->maxSize);
  }

  /**
   * Tests that numeric max filesize is handled correctly.
   *
   * @covers ::resolveFromState
   */
  public function testNumericMaxFilesize(): void {
    $this->storeFormConfig('numeric_size_form', 'field', [
      'extensions' => 'jpg',
      'max_filesize' => 1048576,
      'upload_location' => 'public://test',
    ]);

    $result = $this->resolver->resolveFromState('numeric_size_form', 'field');

    $this->assertEquals(1048576, $result->maxSize);
  }

  /**
   * Creates an image media type for testing.
   *
   * @param string $id
   *   The media type ID.
   */
  protected function createImageMediaType(string $id): void {
    $media_type = MediaType::create([
      'id' => $id,
      'label' => ucfirst($id),
      'source' => 'image',
      'source_configuration' => [],
    ]);
    $media_type->save();

    // Create the source field.
    $source = $media_type->getSource();
    $source_field = $source->createSourceField($media_type);
    $source_field->getFieldStorageDefinition()->save();
    $source_field->save();

    // Set source field in media type config.
    $media_type->set('source_configuration', [
      'source_field' => $source_field->getName(),
    ]);
    $media_type->save();
  }

}
