/**
 * @file
 * FilePond form element integration.
 *
 * Initializes FilePond on form elements and handles file ID management.
 * Unlike dropzonejs which stores temp paths, this creates file entities
 * during upload and stores file IDs in the hidden field.
 */

/* global FilePond, FilePondPluginFileValidateType, FilePondPluginFileValidateSize, FilePondPluginImagePreview, FilePondPluginFilePoster */

(function (Drupal, drupalSettings) {
  function ratioToDecimal(ratio) {
    if (typeof ratio === 'number') {
      return ratio;
    }
    if (typeof ratio === 'string' && ratio.includes(':')) {
      const parts = ratio.split(':');
      const width = parseFloat(parts[0]);
      const height = parseFloat(parts[1]);
      if (width > 0 && height > 0) {
        return height / width;
      }
    }
    return 1;
  }

  /**
   * Updates the browse/drop state based on file count vs max files.
   *
   * When the file limit is reached, disables adding new files.
   * When files are removed and we're under the limit, re-enables adding.
   *
   * @param {Object} pond
   *   The FilePond instance.
   * @param {number} maxFiles
   *   Maximum allowed files (0 = unlimited).
   */
  function updateBrowseState(pond, maxFiles) {
    // 0 means unlimited - always allow browse.
    if (maxFiles === 0) {
      return;
    }

    const currentCount = pond.getFiles().length;
    const atLimit = currentCount >= maxFiles;

    // Toggle browse/drop based on whether we're at the limit.
    // Note: allowPaste is always false - we handle paste ourselves for context-awareness.
    pond.setOptions({
      allowBrowse: !atLimit,
      allowDrop: !atLimit,
    });

    // Add/remove CSS class for styling purposes.
    if (atLimit) {
      pond.element.classList.add('filepond--limit-reached');
    } else {
      pond.element.classList.remove('filepond--limit-reached');
    }
  }

  /**
   * Updates the hidden field with current file IDs.
   *
   * Handles both:
   * - Newly uploaded files: use serverId (mapped from transfer ID if chunked)
   * - Existing local files: use source (file ID passed during initialization)
   *
   * @param {HTMLElement} hiddenField
   *   The hidden input element.
   * @param {Object} pond
   *   The FilePond instance.
   */
  function updateHiddenField(hiddenField, pond) {
    const fileIds = pond
      .getFiles()
      .filter(function (file) {
        // Include successfully processed files (newly uploaded).
        if (
          file.status === FilePond.FileStatus.PROCESSING_COMPLETE &&
          file.serverId
        ) {
          return true;
        }
        // Include local files (existing files loaded from server).
        // These have origin === 'local' and their source is the file ID.
        // Status will be IDLE (5) for local files that didn't need processing.
        if (file.origin === FilePond.FileOrigin.LOCAL && file.source) {
          return true;
        }
        return false;
      })
      .map(function (file) {
        // For local files (existing), source is the file ID.
        if (file.origin === FilePond.FileOrigin.LOCAL) {
          return file.source;
        }
        // For newly uploaded files, use serverId.
        const { serverId } = file;
        // For chunked uploads, serverId is the transfer ID (32-char hex).
        // Look up the actual file ID from our global map.
        if (Drupal.filepond.transferToFileId[serverId]) {
          return Drupal.filepond.transferToFileId[serverId];
        }
        // For regular uploads, serverId is already the file ID.
        return serverId;
      });

    hiddenField.value = fileIds.join(';');

    // Trigger change event for form state.
    const event = new Event('change', { bubbles: true });
    hiddenField.dispatchEvent(event);
  }

  /**
   * Behavior for FilePond form elements.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.filepondElement = {
    attach(context) {
      // Get CSRF token first, then initialize all instances.
      Drupal.filepond.fetchCsrfToken().then(function (csrfToken) {
        // Don't use once() here - we need to retry if config isn't ready yet
        // (BigPipe may not have delivered drupalSettings when first called).
        // Instead, use a class to track initialization state.
        const inputs = context.querySelectorAll(
          '.filepond--input:not(.filepond-initialized)',
        );

        inputs.forEach(function (input) {
          const elementId = input.getAttribute('data-filepond-id');
          if (!elementId) {
            return;
          }

          const config =
            drupalSettings.filepond &&
            drupalSettings.filepond.instances &&
            drupalSettings.filepond.instances[elementId];
          if (!config) {
            // Config not in drupalSettings yet - BigPipe may still be streaming.
            // Don't mark as initialized; will retry on next attachBehaviors call.
            return;
          }

          // Mark as initialized to prevent re-processing.
          input.classList.add('filepond-initialized');

          // Find the hidden field that stores file IDs.
          // Use data-filepond-fids attribute for reliable lookup across
          // Drupal versions (DOM structure can vary between D10/D11).
          let hiddenField = document.querySelector(
            `input[data-filepond-fids="${elementId}"]`,
          );

          // Fallback: try inside wrapper (legacy support).
          if (!hiddenField) {
            const wrapper = input.closest('.filepond-wrapper');
            hiddenField =
              wrapper && wrapper.querySelector('input[type="hidden"]');
          }

          if (!hiddenField) {
            // eslint-disable-next-line no-console
            console.warn(
              'FilePond: Could not find hidden field for',
              elementId,
            );
            return;
          }

          // Get URLs from drupalSettings config.
          const { processUrl } = config;
          const { revertUrl } = config;
          const { patchUrl } = config;

          // Validate URLs exist before proceeding.
          if (!processUrl || !revertUrl || !patchUrl) {
            // eslint-disable-next-line no-console
            console.error('FilePond: Missing URLs in config', config);
            return;
          }

          // Register plugins.
          Drupal.filepond.registerPlugins();

          // Build base headers with CSRF token.
          const baseHeaders = { 'X-CSRF-Token': csrfToken };

          const patchUrlWithSlash = `${patchUrl}/`;

          // Build server config object.
          const serverConfig = {
            process: {
              url: processUrl,
              headers: baseHeaders,
            },
            // Custom revert function to handle transfer ID -> file ID mapping.
            // For chunked uploads, serverId is the transfer ID, but we need
            // to send the signed file ID for proper deletion.
            revert: (uniqueFileId, load, error) => {
              // Look up actual file ID from global mapping (chunked uploads).
              // For regular uploads, serverId is already the signed file ID.
              const fileId =
                Drupal.filepond.transferToFileId[uniqueFileId] || uniqueFileId;

              // Skip server call for pre-existing files (plain numeric IDs).
              // These are files already attached to the entity - they don't have
              // a signed token (no `:`) and aren't transfer IDs (not 32-char hex).
              // The server would just return "skipped" anyway, so save the hop.
              // File removal happens when the parent entity form is saved.
              const isSigned = fileId.includes(':');
              const isTransferId = /^[a-f0-9]{32}$/.test(fileId);
              if (!isSigned && !isTransferId) {
                load();
                return;
              }

              const request = new XMLHttpRequest();
              request.open('DELETE', revertUrl);

              // Add headers.
              Object.keys(baseHeaders).forEach((key) => {
                request.setRequestHeader(key, baseHeaders[key]);
              });

              request.onload = () => {
                if (request.status >= 200 && request.status < 300) {
                  load();
                } else {
                  error(request.responseText || 'Revert failed');
                }
              };

              request.onerror = () => {
                error('Network error during revert');
              };

              request.send(fileId);
            },
            patch: {
              url: patchUrlWithSlash,
              headers: baseHeaders,
              // Capture file ID from final PATCH response and map it.
              // Server returns JSON: { transferId: "abc...", fileId: "123" }
              // This avoids race conditions with parallel uploads.
              // Callback receives (xhr, chunkIndex, totalChunks).
              onload: (xhr) => {
                const responseText = xhr.responseText || xhr.response || '';
                // Final chunk returns JSON with transfer ID and file ID.
                // Intermediate chunks return empty (status 204).
                if (xhr.status === 200 && responseText) {
                  try {
                    const data = JSON.parse(responseText);
                    if (data.transferId && data.fileId) {
                      // Map transfer ID to file ID in global mapping.
                      Drupal.filepond.transferToFileId[data.transferId] =
                        data.fileId;
                      // Store media ID in global mapping for other scripts.
                      if (data.mediaId) {
                        Drupal.filepond.transferToMediaId[data.transferId] =
                          data.mediaId;
                      }
                      // Return just the fileId - FilePond expects a plain
                      // string server ID, not JSON. Returning JSON may cause
                      // FilePond to think upload isn't complete and retry
                      // (observed in headless Chrome CI tests).
                      return data.fileId;
                    }
                  } catch (e) {
                    // Not JSON, may be legacy plain text response.
                    // eslint-disable-next-line no-console
                    console.warn('FilePond: Could not parse PATCH response', e);
                  }
                }
                return responseText;
              },
            },
          };

          // Create FilePond instance.
          // Server-side validation uses handler defaults or module config.
          // Max files limit (0 or null = unlimited).
          const maxFilesLimit = parseInt(config.maxFiles, 10) || 0;

          // For single-file fields, use allowMultiple: false so the file browser
          // only allows selecting one file. For multi-drop, we add a custom handler
          // below that shows a message explaining why files were rejected.
          const allowMultiple = maxFilesLimit !== 1;

          // Get existing files from config (for edit forms with existing images).
          // These are passed from PHP via drupalSettings with type: 'local'
          // and metadata.poster for thumbnail display via File Poster plugin.
          const existingFiles = config.files || [];

          // Get custom options (these are in config but not native FilePond options).
          const previewFitMode = config.previewFitMode || 'contain';
          const labelProcessing = config.labelProcessing || 'Processing...';
          const aspectRatio = config.styleItemPanelAspectRatio || '1:1';
          // Check if paste was enabled in config (default true if not specified).
          const pasteEnabled = config.allowPaste !== false;

          // Build FilePond options object.
          // Start with config from PHP, then override with our explicit
          // settings.
          const pondOptions = {
            // Pass through config options from PHP.
            ...config,

            // Core settings (use config value or default).
            credits: config.credits ?? false,
            allowMultiple,
            chunkUploads: config.chunkUploads ?? true,
            chunkForce: config.chunkForce ?? false,
            // Disable built-in paste functionality - we handle it ourselves
            // to support multiple widgets on the same page.
            allowPaste: false,

            // Server configuration - use pre-built config object.
            server: serverConfig,
            imagePreviewMaxHeight: 500,

            // Set aspect ratio for consistent grid item heights.
            // styleItemPanelAspectRatio is a decimal - height/width
            styleItemPanelAspectRatio:
              config.stylePanelLayout === 'integrated'
                ? null
                : ratioToDecimal(aspectRatio),

            // Image Crop plugin - when fit mode is 'cover', crop preview to
            // panel aspect ratio.
            imageCropAspectRatio:
              previewFitMode === 'cover' && config.styleItemPanelAspectRatio
                ? aspectRatio
                : null,
            allowImageCrop: config.allowImageCrop || previewFitMode === 'cover',

            // Initialize with existing files (if any).
            // File Poster plugin displays thumbnails from metadata.poster URL.
            files: existingFiles,
          };

          // Remove properties that shouldn't go to FilePond.
          // Server URLs - we build the server config object from these.
          delete pondOptions.processUrl;
          delete pondOptions.revertUrl;
          delete pondOptions.patchUrl;
          // Custom options used by our code but not native FilePond options.
          delete pondOptions.extensions;
          delete pondOptions.previewFitMode;
          delete pondOptions.labelProcessing;
          delete pondOptions.previewImageStyle;

          const pond = FilePond.create(input, {
            ...pondOptions,
            // Don't use FilePond's native maxFiles - it rejects the ENTIRE batch
            // when the total exceeds the limit. Instead, we use beforeAddFile
            // to accept files up to the limit and reject the rest.
            // The beforeAddFile callback is added via setOptions() below
            // because it needs to reference `pond` which isn't defined yet.
            maxFiles: null,

            // Event handlers.
            onaddfile() {
              pond.element.classList.add('has-files');
              const container = document.querySelector('.filepond-container');
              if (container) {
                container.classList.add('has-files');
              }

              // Check if we've hit the file limit.
              updateBrowseState(pond, maxFilesLimit);

              // Disable submit when files are added (before processing).
              if (Drupal.filepondUpdateSubmitState) {
                Drupal.filepondUpdateSubmitState(pond);
              }
            },

            onprocessfileprogress(file, progress) {
              // When upload reaches 100%, server is still processing (creating entities, etc).
              // Show "Processing..." status to indicate work is happening.
              if (progress === 1) {
                const items = pond.element.querySelectorAll(
                  '.filepond--item[data-filepond-item-state="processing"]',
                );
                items.forEach(function (item) {
                  const status = item.querySelector(
                    '.filepond--file-status-main',
                  );
                  if (status) {
                    status.textContent = labelProcessing;
                  }
                  item.classList.add('is-server-processing');
                });
              }
            },

            onprocessfile(error) {
              if (error) {
                // eslint-disable-next-line no-console
                console.error('FilePond upload error:', error);
              }

              // Remove processing class from completed/error items.
              const completedItems = pond.element.querySelectorAll(
                '.filepond--item[data-filepond-item-state="complete"].is-server-processing, ' +
                  '.filepond--item[data-filepond-item-state="processing-error"].is-server-processing',
              );
              completedItems.forEach(function (item) {
                item.classList.remove('is-server-processing');
              });

              // Update hidden field with file IDs.
              // For chunked uploads, the mapping was already done in patch.onload.
              updateHiddenField(hiddenField, pond);
              if (Drupal.filepondUpdateSubmitState) {
                Drupal.filepondUpdateSubmitState(pond);
              }
            },

            onremovefile() {
              const remainingFiles = pond.getFiles();
              if (!remainingFiles.length) {
                pond.element.classList.remove('has-files');
                const container = document.querySelector('.filepond-container');
                if (container) {
                  container.classList.remove('has-files');
                }
              }

              // Re-enable browse if we're back under the limit.
              updateBrowseState(pond, maxFilesLimit);

              updateHiddenField(hiddenField, pond);
              if (Drupal.filepondUpdateSubmitState) {
                Drupal.filepondUpdateSubmitState(pond);
              }
            },

            onprocessfilerevert() {
              updateHiddenField(hiddenField, pond);
              if (Drupal.filepondUpdateSubmitState) {
                Drupal.filepondUpdateSubmitState(pond);
              }
            },

            onreorderfiles() {
              // Update hidden field to preserve new order.
              updateHiddenField(hiddenField, pond);
              if (Drupal.filepondUpdateSubmitState) {
                Drupal.filepondUpdateSubmitState(pond);
              }
            },
          });

          // Configure max files limit using shared utility.
          // This sets up beforeAddFile callback that accepts files up to limit
          // and rejects excess with a warning message.
          if (maxFilesLimit > 0) {
            Drupal.filepond.setMaxFiles(pond, maxFilesLimit);
          }

          // Handle server errors - show message for rate limiting (429).
          pond.on('error', (error) => {
            // FilePond error object has: type, code (HTTP status), body, headers.
            // Note: body contains HTTP status text, not our custom response body.
            if (error && error.code === 429) {
              Drupal.filepond.showMessage(
                Drupal.t(
                  "You've uploaded a lot of files recently. Please wait a bit before uploading more.",
                ),
                {
                  type: 'error',
                  context: pond.element,
                },
              );
            }
          });

          // Store reference for external access.
          input._filepond = pond;

          // Dispatch custom event for other scripts to hook into.
          // Dispatch on the form element wrapper so siblings can listen.
          const pondWrapper = pond.element.closest(
            '.js-form-type-filepond, .form-type-filepond, .js-form-item',
          );

          // Context-aware paste handling.
          // Track hover on wrapper, global paste listener checks this.
          if (pasteEnabled && pondWrapper) {
            pondWrapper.dataset.filepondPaste = 'true';

            pondWrapper.addEventListener('mouseenter', function () {
              Drupal.filepond.hoveredPond = pond;
            });
            pondWrapper.addEventListener('mouseleave', function () {
              if (Drupal.filepond.hoveredPond === pond) {
                Drupal.filepond.hoveredPond = null;
              }
            });
          }

          // For single-file fields (maxFilesLimit === 1), add a drop handler
          // that shows a message when multiple files are dropped.
          // FilePond silently rejects multi-drops when allowMultiple: false.
          if (maxFilesLimit === 1) {
            pond.element.addEventListener('drop', function (e) {
              const files = e.dataTransfer?.files;
              if (files && files.length > 1) {
                // Show message after a brief delay for better UX.
                setTimeout(function () {
                  Drupal.filepond.showMessage(
                    Drupal.t(
                      'Only 1 file allowed. Please drop one file at a time.',
                    ),
                    {
                      type: 'warning',
                      context: pond.element,
                    },
                  );
                }, 100);
              }
            });
          }

          const initEvent = new CustomEvent('filepond:init', {
            bubbles: true,
            detail: { pond, input },
          });
          (pondWrapper || pond.element).dispatchEvent(initEvent);

          // Apply preview fit mode class for CSS styling.
          // 'contain' = fit entire image, 'cover' = crop to fill.
          pond.element.classList.add(`filepond--fit-${previewFitMode}`);

          // Initialize hidden field with existing file IDs (if any).
          // This ensures the form will submit with correct values even
          // if the user doesn't make any changes.
          if (existingFiles.length > 0) {
            // Use a small delay to let FilePond finish initializing files.
            setTimeout(function () {
              updateHiddenField(hiddenField, pond);
              // Check if existing files already meet the limit.
              updateBrowseState(pond, maxFilesLimit);
            }, 100);
          }
        });
      });
    },
  };
})(Drupal, drupalSettings);
