<?php

declare(strict_types=1);

namespace Drupal\filepond_eb_widget\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\filepond\Controller\UploadController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Handles FilePond upload requests for Entity Browser widget context.
 *
 * Extends the main UploadController and uses the settings resolver for
 * EB-specific option resolution.
 *
 * Routes:
 * - POST /filepond/eb/process/{entity_browser}/{widget_uuid}
 * - PATCH /filepond/eb/patch/{entity_browser}/{widget_uuid}/{transferId}
 * - DELETE /filepond/eb/revert/{entity_browser}/{widget_uuid}
 */
class EbUploadController extends UploadController {

  /**
   * Access check for Entity Browser upload routes.
   *
   * @param string $entity_browser
   *   The entity browser ID.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(string $entity_browser): AccessResultInterface {
    $account = $this->currentUser();

    // Check filepond permission.
    if (!$account->hasPermission('filepond upload files')) {
      return AccessResult::forbidden()->cachePerPermissions();
    }

    // Check entity browser access permission.
    $eb_permission = 'access ' . $entity_browser . ' entity browser pages';
    return AccessResult::allowedIfHasPermission($account, $eb_permission);
  }

  /**
   * Handles EB process requests (POST).
   */
  public function ebProcess(Request $request, string $entity_browser, string $widget_uuid): Response {
    $options = $this->settingsResolver->resolveFromWidget($entity_browser, $widget_uuid);

    // Check media create access if uploading to a media type.
    $media_type = $options->context['bundle'] ?? NULL;
    if ($media_type) {
      $access_error = $this->checkMediaCreateAccess($media_type);
      if ($access_error) {
        return $access_error;
      }
    }

    return $this->handleProcess($request, $options->toArray());
  }

  /**
   * Handles EB PATCH requests for chunked uploads.
   */
  public function ebPatch(Request $request, string $entity_browser, string $widget_uuid, string $transferId): Response {
    $options = $this->settingsResolver->resolveFromWidget($entity_browser, $widget_uuid);

    // Check media create access if uploading to a media type.
    $media_type = $options->context['bundle'] ?? NULL;
    if ($media_type) {
      $access_error = $this->checkMediaCreateAccess($media_type);
      if ($access_error) {
        return $access_error;
      }
    }

    return $this->handlePatch($request, $transferId, $options->toArray());
  }

  /**
   * Handles EB DELETE requests to revert/cancel uploads.
   */
  public function ebRevert(Request $request, string $entity_browser, string $widget_uuid): Response {
    // Validate widget exists. Revert access handled by parent.
    $this->settingsResolver->resolveFromWidget($entity_browser, $widget_uuid);
    return $this->handleRevert($request);
  }

}
