<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond_eb_widget\Kernel;

use Drupal\entity_browser\Entity\EntityBrowser;
use Drupal\filepond\UploadOptions;
use Drupal\filepond\UploadSettingsResolverInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\media\Entity\MediaType;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Tests the FilePond Entity Browser widget integration.
 *
 * @group filepond_eb_widget
 */
class EbWidgetTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'file',
    'field',
    'image',
    'media',
    'views',
    'entity_browser',
    'filepond',
    'filepond_eb_widget',
  ];

  /**
   * The settings resolver service.
   */
  protected UploadSettingsResolverInterface $resolver;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('media');
    $this->installConfig(['filepond', 'media']);

    $this->resolver = $this->container->get('filepond.settings_resolver');
  }

  /**
   * Creates an image media type for testing.
   *
   * @param string $id
   *   The media type ID.
   * @param array $field_settings
   *   Optional field settings overrides.
   */
  protected function createImageMediaType(string $id, array $field_settings = []): void {
    $media_type = MediaType::create([
      'id' => $id,
      'label' => ucfirst($id),
      'source' => 'image',
      'source_configuration' => [],
    ]);
    $media_type->save();

    // Create the source field.
    $source = $media_type->getSource();
    $source_field = $source->createSourceField($media_type);

    // Apply any custom field settings.
    if (!empty($field_settings)) {
      foreach ($field_settings as $key => $value) {
        $source_field->setSetting($key, $value);
      }
    }

    $source_field->getFieldStorageDefinition()->save();
    $source_field->save();

    // Set source field in media type config.
    $media_type->set('source_configuration', [
      'source_field' => $source_field->getName(),
    ]);
    $media_type->save();
  }

  /**
   * Creates an entity browser with a FilePond widget.
   *
   * @param string $eb_id
   *   The entity browser ID.
   * @param string $widget_uuid
   *   The widget UUID.
   * @param array $widget_settings
   *   Widget settings.
   *
   * @return \Drupal\entity_browser\Entity\EntityBrowser
   *   The created entity browser.
   */
  protected function createEntityBrowserWithWidget(string $eb_id, string $widget_uuid, array $widget_settings): EntityBrowser {
    $entity_browser = EntityBrowser::create([
      'name' => $eb_id,
      'label' => 'Test Browser',
      'display' => 'standalone',
      'display_configuration' => [],
      'selection_display' => 'no_display',
      'selection_display_configuration' => [],
      'widget_selector' => 'single',
      'widget_selector_configuration' => [],
      'widgets' => [
        $widget_uuid => array_merge([
          'id' => 'filepond_media',
          'uuid' => $widget_uuid,
          'label' => 'Upload',
          'weight' => 0,
        ], ['settings' => $widget_settings]),
      ],
    ]);
    $entity_browser->save();
    return $entity_browser;
  }

  /**
   * Tests resolveFromWidget throws for non-existent entity browser.
   */
  public function testResolveFromWidgetThrowsForNonExistentBrowser(): void {
    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('Entity browser not found');

    $this->resolver->resolveFromWidget('nonexistent_browser', 'some-uuid');
  }

  /**
   * Tests resolveFromWidget throws for non-existent widget.
   */
  public function testResolveFromWidgetThrowsForNonExistentWidget(): void {
    // Create browser with a widget.
    $this->createImageMediaType('test_image');
    $this->createEntityBrowserWithWidget('test_browser', 'valid-uuid', [
      'media_type' => 'test_image',
    ]);

    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('Widget not found');

    $this->resolver->resolveFromWidget('test_browser', 'wrong-uuid');
  }

  /**
   * Tests resolveFromWidget throws for wrong widget type.
   */
  public function testResolveFromWidgetThrowsForWrongWidgetType(): void {
    // Create browser with a non-filepond widget.
    $entity_browser = EntityBrowser::create([
      'name' => 'wrong_type_browser',
      'label' => 'Test Browser',
      'display' => 'standalone',
      'display_configuration' => [],
      'selection_display' => 'no_display',
      'selection_display_configuration' => [],
      'widget_selector' => 'single',
      'widget_selector_configuration' => [],
      'widgets' => [
        'widget-uuid' => [
          'id' => 'upload',
          'uuid' => 'widget-uuid',
          'label' => 'Upload',
          'weight' => 0,
          'settings' => [],
        ],
      ],
    ]);
    $entity_browser->save();

    $this->expectException(AccessDeniedHttpException::class);
    $this->expectExceptionMessage('Invalid widget type');

    $this->resolver->resolveFromWidget('wrong_type_browser', 'widget-uuid');
  }

  /**
   * Tests resolveFromWidget throws when media type not configured.
   */
  public function testResolveFromWidgetThrowsWhenMediaTypeNotConfigured(): void {
    $this->createEntityBrowserWithWidget('no_media_browser', 'widget-uuid', [
      'media_type' => '',
    ]);

    $this->expectException(AccessDeniedHttpException::class);
    $this->expectExceptionMessage('Media type not configured');

    $this->resolver->resolveFromWidget('no_media_browser', 'widget-uuid');
  }

  /**
   * Tests resolveFromWidget throws when media type not found.
   */
  public function testResolveFromWidgetThrowsWhenMediaTypeNotFound(): void {
    $this->createEntityBrowserWithWidget('bad_media_browser', 'widget-uuid', [
      'media_type' => 'nonexistent_media_type',
    ]);

    $this->expectException(AccessDeniedHttpException::class);
    $this->expectExceptionMessage('Media type not found');

    $this->resolver->resolveFromWidget('bad_media_browser', 'widget-uuid');
  }

  /**
   * Tests resolveFromWidget returns options with inherit_settings enabled.
   */
  public function testResolveFromWidgetWithInheritSettings(): void {
    // Create media type with custom field settings.
    $this->createImageMediaType('inherit_test', [
      'file_extensions' => 'jpg png webp',
      'max_filesize' => '15M',
      'file_directory' => 'inherited-uploads',
    ]);

    $widget_uuid = 'inherit-widget-uuid';
    $this->createEntityBrowserWithWidget('inherit_browser', $widget_uuid, [
      'media_type' => 'inherit_test',
      'inherit_settings' => TRUE,
      // These should be ignored when inherit is TRUE.
      'extensions' => 'gif',
      'max_filesize' => '5M',
    ]);

    $result = $this->resolver->resolveFromWidget('inherit_browser', $widget_uuid);

    $this->assertInstanceOf(UploadOptions::class, $result);
    // Should use field settings, not widget settings.
    $this->assertEquals(['jpg', 'png', 'webp'], $result->allowedExtensions);
    $this->assertEquals(15 * 1024 * 1024, $result->maxSize);
    $this->assertStringContainsString('inherited-uploads', $result->destination);

    // Check context.
    $this->assertEquals('inherit_browser', $result->context['entity_browser']);
    $this->assertEquals($widget_uuid, $result->context['widget_uuid']);
    $this->assertEquals('inherit_test', $result->context['bundle']);
  }

  /**
   * Tests resolveFromWidget returns options with manual settings.
   */
  public function testResolveFromWidgetWithManualSettings(): void {
    // Create media type (field settings will be ignored).
    $this->createImageMediaType('manual_test', [
      'file_extensions' => 'jpg',
      'max_filesize' => '10M',
    ]);

    $widget_uuid = 'manual-widget-uuid';
    $this->createEntityBrowserWithWidget('manual_browser', $widget_uuid, [
      'media_type' => 'manual_test',
      'inherit_settings' => FALSE,
      'extensions' => 'gif png bmp',
      'max_filesize' => '25M',
      'upload_location' => 'public://manual-uploads',
    ]);

    $result = $this->resolver->resolveFromWidget('manual_browser', $widget_uuid);

    $this->assertInstanceOf(UploadOptions::class, $result);
    // Should use widget settings, not field settings.
    $this->assertEquals(['gif', 'png', 'bmp'], $result->allowedExtensions);
    $this->assertEquals(25 * 1024 * 1024, $result->maxSize);
    $this->assertEquals('public://manual-uploads', $result->destination);
  }

  /**
   * Tests MIME types are correctly derived from extensions.
   */
  public function testMimeTypesAreDerivedFromExtensions(): void {
    $this->createImageMediaType('mime_test', [
      'file_extensions' => 'jpg png gif',
    ]);

    $widget_uuid = 'mime-widget-uuid';
    $this->createEntityBrowserWithWidget('mime_browser', $widget_uuid, [
      'media_type' => 'mime_test',
      'inherit_settings' => TRUE,
    ]);

    $result = $this->resolver->resolveFromWidget('mime_browser', $widget_uuid);

    $this->assertContains('image/jpeg', $result->allowedMimeTypes);
    $this->assertContains('image/png', $result->allowedMimeTypes);
    $this->assertContains('image/gif', $result->allowedMimeTypes);
  }

  /**
   * Tests default extensions when none specified.
   */
  public function testDefaultExtensionsWhenNoneSpecified(): void {
    $this->createImageMediaType('default_ext_test');

    $widget_uuid = 'default-ext-uuid';
    $this->createEntityBrowserWithWidget('default_ext_browser', $widget_uuid, [
      'media_type' => 'default_ext_test',
      'inherit_settings' => FALSE,
      // No extensions specified - should use defaults.
    ]);

    $result = $this->resolver->resolveFromWidget('default_ext_browser', $widget_uuid);

    // Should use default extensions.
    $this->assertContains('png', $result->allowedExtensions);
    $this->assertContains('jpg', $result->allowedExtensions);
    $this->assertContains('gif', $result->allowedExtensions);
    $this->assertContains('jpeg', $result->allowedExtensions);
  }

}
