/**
 * @file
 * Cropper.js v2 integration for FilePond uploads.
 *
 * After FilePond upload completes, shows Cropper.js inline for cropping.
 * Crop coordinates are stored in hidden fields and saved on form submit.
 *
 * States:
 * - No image: Show FilePond uploader, hide crop container
 * - Has image (cropping): Show Cropper.js, hide FilePond
 * - Has image (applied): Show preview, hide Cropper.js and FilePond
 */

/* global Cropper */

(function (Drupal, drupalSettings, once) {
  /**
   * FilePond Crop widget controller.
   *
   * @param {HTMLElement} wrapper
   *   The widget wrapper element.
   * @param {object} settings
   *   The widget settings from drupalSettings.
   */
  function FilepondCropWidget(wrapper, settings) {
    this.wrapper = wrapper;
    this.settings = settings;
    this.cropper = null;
    this.pond = null;
    this.currentImageUrl = settings.imageUrl || null;
    this.isApplied = false;
    this.pendingAutoApply = false;

    // Original image dimensions (for scaling crop coordinates).
    this.originalWidth = settings.originalWidth || 0;
    this.originalHeight = settings.originalHeight || 0;

    // Flag to skip bounds check during programmatic changes.
    this.skipBoundsCheck = false;

    // Display scaling info (set in createCropper).
    this.displayScale = 1;
    this.imageOffsetX = 0;
    this.imageOffsetY = 0;
    this.imageBounds = null;

    // Track which image URL the cropper was created for.
    this.cropperImageUrl = null;
    // Flag to prevent concurrent cropper initialization.
    this.isInitializingCropper = false;

    // Cache DOM elements.
    this.cropContainer = wrapper.querySelector(
      '[data-filepond-crop="container"]',
    );
    this.imageWrapper = wrapper.querySelector(
      '[data-filepond-crop="image-wrapper"]',
    );
    this.image = wrapper.querySelector('[data-filepond-crop="image"]');
    this.applyButton = wrapper.querySelector('[data-filepond-crop="apply"]');
    this.removeButton = wrapper.querySelector('[data-filepond-crop="remove"]');
    this.editButton = wrapper.querySelector('[data-filepond-crop="edit"]');
    this.resetButton = wrapper.querySelector('[data-filepond-crop="reset"]');
    this.preview = wrapper.querySelector('[data-filepond-crop="preview"]');

    // Hidden fields for crop coordinates and file ID.
    this.fields = {
      fid: wrapper.querySelector('[data-filepond-crop="fid"]'),
      x: wrapper.querySelector('[data-filepond-crop="x"]'),
      y: wrapper.querySelector('[data-filepond-crop="y"]'),
      width: wrapper.querySelector('[data-filepond-crop="width"]'),
      height: wrapper.querySelector('[data-filepond-crop="height"]'),
      applied: wrapper.querySelector('[data-filepond-crop="applied"]'),
    };

    this.init();
  }

  /**
   * Initialize the widget.
   */
  FilepondCropWidget.prototype.init = function () {
    const self = this;

    // Bind button events with preventDefault to stop form submission.
    if (this.applyButton) {
      this.applyButton.addEventListener('click', function handleApplyClick(e) {
        e.preventDefault();
        e.stopPropagation();
        self.applyCrop();
        return false;
      });
    }
    if (this.removeButton) {
      this.removeButton.addEventListener(
        'click',
        function handleRemoveClick(e) {
          e.preventDefault();
          e.stopPropagation();
          // Confirm before removing to prevent accidental clicks.
          if (
            window.confirm(
              Drupal.t('Are you sure you want to remove this image?'),
            )
          ) {
            self.removeImage();
          }
          return false;
        },
      );
    }
    if (this.editButton) {
      this.editButton.addEventListener('click', function handleEditClick(e) {
        e.preventDefault();
        e.stopPropagation();
        self.editCrop();
        return false;
      });
    }
    if (this.resetButton) {
      this.resetButton.addEventListener('click', function handleResetClick(e) {
        e.preventDefault();
        e.stopPropagation();
        self.resetCropData();
        return false;
      });
    }

    // Listen for FilePond initialization.
    this.wrapper.addEventListener(
      'filepond:init',
      this.onFilepondInit.bind(this),
    );

    // Check if FilePond is already initialized (event may have fired before us).
    const filepondInput = this.wrapper.querySelector('.filepond--input');
    if (filepondInput && filepondInput._filepond) {
      this.attachToPond(filepondInput._filepond);
    }

    // Check initial state.
    if (this.settings.hasAppliedCrop) {
      // Existing image with applied crop.
      if (this.settings.showCropPreview) {
        // Preview mode: show preview (FilePond is empty).
        this.isApplied = true;
        this.showAppliedState();
      } else {
        // Direct mode: show cropper with existing crop data.
        this.showCropperState();
      }
    } else if (this.settings.hasImage && this.currentImageUrl) {
      // Existing image without crop - show cropper.
      this.showCropperState();
    } else {
      // No image - show FilePond state.
      this.showFilePondState();
    }
  };

  /**
   * Handle FilePond initialization.
   *
   * @param {CustomEvent} event
   *   The filepond:init event.
   */
  FilepondCropWidget.prototype.onFilepondInit = function (event) {
    this.attachToPond(event.detail.pond);
  };

  /**
   * Attach to a FilePond instance.
   *
   * @param {object} pond
   *   The FilePond instance.
   */
  FilepondCropWidget.prototype.attachToPond = function (pond) {
    // Don't attach twice.
    if (this.pond) {
      return;
    }

    this.pond = pond;

    // Listen for file upload completion.
    this.pond.on('processfile', this.onFileUploaded.bind(this));

    // Listen for file removal via FilePond (if visible).
    this.pond.on('removefile', this.onFileRemoved.bind(this));
  };

  /**
   * Handle file upload completion.
   *
   * @param {object} error
   *   Error object if upload failed.
   * @param {object} file
   *   The FilePond file item.
   */
  FilepondCropWidget.prototype.onFileUploaded = function (error, file) {
    if (error) {
      return;
    }

    // Get file ID for the hidden field.
    const fileId = this.getFileId(file);
    if (fileId && this.fields.fid) {
      this.fields.fid.value = fileId;
    }

    // Get original dimensions from FilePond file metadata.
    // The image-preview plugin stores this when it reads the image.
    this.extractOriginalDimensions(file);

    // Get the uploaded file URL.
    const fileUrl = this.getFileUrl(file);

    if (!fileUrl) {
      return;
    }

    this.currentImageUrl = fileUrl;
    this.isApplied = false;

    // Clear any existing crop data (but keep fid which we just set).
    this.settings.existingCrop = null;
    this.clearCropFields();

    // Auto-apply mode: initialize cropper silently and apply immediately.
    // Manual mode: show cropper for user to adjust.
    if (this.settings.showDefaultCrop) {
      this.pendingAutoApply = true;
    }

    // Show cropper for adjustment or auto-apply.
    this.showCropperState();
  };

  /**
   * Extract original image dimensions from FilePond file.
   *
   * @param {object} file
   *   The FilePond file item.
   */
  FilepondCropWidget.prototype.extractOriginalDimensions = function (file) {
    // Method 1: FilePond image-preview plugin stores size in metadata.
    // Use getMetadata('size') for specific key access.
    if (typeof file.getMetadata === 'function') {
      const size = file.getMetadata('size');
      if (size && size.width && size.height) {
        this.originalWidth = size.width;
        this.originalHeight = size.height;
        return;
      }
    }

    // Method 2: Check full metadata object.
    const metadata = file.getMetadata ? file.getMetadata() : file.metadata;
    if (metadata) {
      if (metadata.size && metadata.size.width && metadata.size.height) {
        this.originalWidth = metadata.size.width;
        this.originalHeight = metadata.size.height;
        return;
      }
      if (metadata.width && metadata.height) {
        this.originalWidth = metadata.width;
        this.originalHeight = metadata.height;
        return;
      }
    }

    // Method 3: Synchronous read from File object using createImageBitmap.
    // This is faster than FileReader and runs synchronously in modern browsers.
    if (file.file && file.file.type && file.file.type.startsWith('image/')) {
      const self = this;
      createImageBitmap(file.file)
        .then(function (bitmap) {
          self.originalWidth = bitmap.width;
          self.originalHeight = bitmap.height;
          bitmap.close();
        })
        .catch(function () {
          // Fallback to FileReader for older browsers.
          const reader = new FileReader();
          reader.onload = function (e) {
            const img = new Image();
            img.onload = function () {
              self.originalWidth = img.naturalWidth;
              self.originalHeight = img.naturalHeight;
            };
            img.src = e.target.result;
          };
          reader.readAsDataURL(file.file);
        });
    }
  };

  /**
   * Get file entity ID from a FilePond file item.
   *
   * @param {object} file
   *   The FilePond file item.
   *
   * @return {string|null}
   *   The file entity ID or null.
   */
  FilepondCropWidget.prototype.getFileId = function (file) {
    const { serverId } = file;
    if (!serverId) {
      return null;
    }

    // Parse file ID from serverId (may have signature).
    let fileId = serverId;
    if (serverId.includes(':')) {
      // Signed file ID format: "123:signature"
      [fileId] = serverId.split(':');
    }

    // Check for chunked upload mapping.
    if (Drupal.filepond && Drupal.filepond.transferToFileId) {
      const mappedId = Drupal.filepond.transferToFileId[serverId];
      if (mappedId) {
        [fileId] = mappedId.split(':');
      }
    }

    return fileId;
  };

  /**
   * Handle file removal from FilePond.
   */
  FilepondCropWidget.prototype.onFileRemoved = function () {
    if (!this.pond) {
      return;
    }

    // If we have an image URL but FilePond is empty, we're editing an existing
    // image that wasn't loaded into FilePond. Don't reset.
    if (this.currentImageUrl && this.pond.getFiles().length === 0) {
      return;
    }

    // If FilePond still has files, this was a partial removal.
    if (this.pond.getFiles().length > 0) {
      return;
    }

    // FilePond had files and now has none - user removed the uploaded file.
    this.currentImageUrl = null;
    this.isApplied = false;
    this.destroyCropper();
    this.clearCropFields();
    this.showFilePondState();
  };

  /**
   * Get the URL for an uploaded file.
   *
   * @param {object} file
   *   The FilePond file item.
   *
   * @return {string|null}
   *   The file URL or null if not available.
   */
  FilepondCropWidget.prototype.getFileUrl = function (file) {
    // For local files (existing), use metadata.poster or source URL.
    if (file.origin === 2) {
      // FilePond.FileOrigin.LOCAL = 2
      return file.source || null;
    }

    // Get the preview image that FilePond already rendered.
    // This avoids reloading the image - it's already on screen!
    return this.getFilePondPreviewUrl();
  };

  /**
   * Get the preview image URL from FilePond's rendered preview.
   *
   * FilePond's image-preview plugin renders a canvas element.
   * We can grab it and convert to data URL - no server round-trip needed.
   *
   * @return {string|null}
   *   Data URL of the preview canvas, or null if not found.
   */
  FilepondCropWidget.prototype.getFilePondPreviewUrl = function () {
    // Find the FilePond preview canvas.
    const previewWrapper = this.wrapper.querySelector(
      '.filepond--image-preview-wrapper',
    );
    if (!previewWrapper) {
      return null;
    }

    // FilePond renders the preview in a canvas element.
    const canvas = previewWrapper.querySelector('canvas');
    if (!canvas) {
      return null;
    }

    // Convert canvas to data URL.
    try {
      return canvas.toDataURL('image/jpeg', 0.92);
    } catch (e) {
      // Canvas might be tainted or too large.
      return null;
    }
  };

  /**
   * Show the FilePond uploader state.
   *
   * Visibility is controlled by CSS based on wrapper state class.
   */
  FilepondCropWidget.prototype.showFilePondState = function () {
    // Update widget class - CSS handles visibility.
    this.wrapper.classList.remove(
      'filepond-crop--has-image',
      'filepond-crop--applied',
      'filepond-crop--cropping',
    );
    this.wrapper.classList.add('filepond-crop--empty');
  };

  /**
   * Show the cropper state.
   *
   * Visibility is controlled by CSS based on wrapper state class.
   */
  FilepondCropWidget.prototype.showCropperState = function () {
    // Update widget class - CSS handles visibility.
    this.wrapper.classList.remove(
      'filepond-crop--empty',
      'filepond-crop--applied',
    );
    this.wrapper.classList.add(
      'filepond-crop--has-image',
      'filepond-crop--cropping',
    );

    // Determine if we can reuse existing cropper or need a new one.
    const cropperCanvas = this.wrapper.querySelector('cropper-canvas');
    const sameImage = this.cropperImageUrl === this.currentImageUrl;

    if (cropperCanvas && this.cropper && sameImage) {
      // Same image and cropper exists - just show it.
      cropperCanvas.style.display = '';
    } else if (this.image && this.currentImageUrl) {
      // Different image or no cropper - destroy old and create new.
      if (cropperCanvas || this.cropper) {
        this.destroyCropper();
      }
      // Only set crossOrigin for server URLs (not blob or data URLs).
      if (
        !this.currentImageUrl.startsWith('blob:') &&
        !this.currentImageUrl.startsWith('data:')
      ) {
        this.image.crossOrigin = 'anonymous';
      }
      this.image.src = this.currentImageUrl;
      this.initCropper();
    }
  };

  /**
   * Show the applied crop state with preview.
   *
   * Visibility is controlled by CSS based on wrapper state class.
   */
  FilepondCropWidget.prototype.showAppliedState = function () {
    // Hide cropper canvas (don't destroy - can be reused when editing same image).
    const cropperCanvas = this.wrapper.querySelector('cropper-canvas');
    if (cropperCanvas) {
      cropperCanvas.style.display = 'none';
    }

    // Update widget class - CSS handles visibility.
    this.wrapper.classList.remove(
      'filepond-crop--empty',
      'filepond-crop--cropping',
    );
    this.wrapper.classList.add(
      'filepond-crop--has-image',
      'filepond-crop--applied',
    );
  };

  /**
   * Initialize Cropper.js on the image.
   */
  FilepondCropWidget.prototype.initCropper = function () {
    // Destroy existing cropper.
    this.destroyCropper();

    if (!this.image || !this.currentImageUrl) {
      return;
    }

    const self = this;

    // Wait for image to load before initializing cropper.
    if (this.image.complete && this.image.naturalWidth > 0) {
      this.createCropper();
    } else {
      this.image.onload = function handleImageLoad() {
        self.createCropper();
      };
    }
  };

  /**
   * Create the Cropper.js v2 instance.
   */
  FilepondCropWidget.prototype.createCropper = function () {
    // Skip if cropper already exists (JS instance).
    if (this.cropper) {
      return;
    }

    // Prevent concurrent initialization (onload race condition).
    if (this.isInitializingCropper) {
      return;
    }
    this.isInitializingCropper = true;

    // Also check for orphaned cropper-canvas in DOM and remove it.
    // This can happen if the JS instance was destroyed but DOM wasn't cleaned.
    const existingCanvas = this.wrapper.querySelector('cropper-canvas');
    if (existingCanvas) {
      existingCanvas.remove();
    }

    const self = this;

    // Track which image URL this cropper is for.
    this.cropperImageUrl = this.currentImageUrl;

    // Create Cropper.js v2 instance with default template.
    // Note: v2 UMD build exports constructor at Cropper.default
    // eslint-disable-next-line new-cap
    this.cropper = new Cropper.default(this.image);

    // Get the cropper elements.
    const canvas = this.cropper.getCropperCanvas();
    const selection = this.cropper.getCropperSelection();
    const cropperImage = this.cropper.getCropperImage();

    if (selection) {
      // Set aspect ratio if configured.
      if (this.settings.aspectRatio) {
        selection.aspectRatio = this.settings.aspectRatio;
      }

      // Set initial coverage to full image.
      selection.initialCoverage = 1;

      // Apply circular styling if enabled.
      // Uses $addStyles to inject CSS into shadow DOM.
      if (this.settings.circularCrop) {
        selection.$addStyles(`
          :host {
            border-radius: 50%;
            overflow: hidden;
          }
        `);
        // Also style the grid element inside.
        const grid = selection.querySelector('cropper-grid');
        if (grid && grid.$addStyles) {
          grid.$addStyles(`
            :host {
              border-radius: 50%;
            }
          `);
        }
        // Style the shade element (dark overlay outside selection).
        const shade = canvas.querySelector('cropper-shade');
        if (shade && shade.$addStyles) {
          shade.$addStyles(`
            :host {
              border-radius: 50%;
            }
          `);
        }
      }

      // Snap selection to bounds and enforce minimum width on drag end.
      // We don't block during drag (causes stuttering) - instead snap on release.
      canvas.addEventListener('actionend', function handleActionEnd() {
        if (!self.imageBounds) {
          return;
        }

        const b = self.imageBounds;
        let { x, y, width, height } = selection;
        let needsSnap = false;

        // Enforce minimum width (in original image pixels).
        if (
          self.settings.minCropSize > 0 &&
          self.originalWidth > 0 &&
          self.displayScale > 0
        ) {
          const naturalWidth = self.image ? self.image.naturalWidth : 0;
          const displayedWidth = naturalWidth * self.displayScale;
          const scaleToOriginal =
            displayedWidth > 0 ? self.originalWidth / displayedWidth : 1;
          const origWidth = width * scaleToOriginal;

          if (origWidth < self.settings.minCropSize) {
            // Calculate minimum display width from minimum original width.
            const minDisplayWidth = self.settings.minCropSize / scaleToOriginal;
            width = minDisplayWidth;

            // Adjust height to maintain aspect ratio if set.
            if (self.settings.aspectRatio) {
              height = width / self.settings.aspectRatio;
            }
            needsSnap = true;
          }
        }

        // Constrain size to not exceed image bounds.
        if (width > b.width) {
          width = b.width;
          if (self.settings.aspectRatio) {
            height = width / self.settings.aspectRatio;
          }
          needsSnap = true;
        }
        if (height > b.height) {
          height = b.height;
          if (self.settings.aspectRatio) {
            width = height * self.settings.aspectRatio;
          }
          needsSnap = true;
        }

        // Snap position to stay within image bounds.
        if (x < b.x) {
          x = b.x;
          needsSnap = true;
        }
        if (y < b.y) {
          y = b.y;
          needsSnap = true;
        }
        if (x + width > b.x + b.width) {
          x = b.x + b.width - width;
          needsSnap = true;
        }
        if (y + height > b.y + b.height) {
          y = b.y + b.height - height;
          needsSnap = true;
        }

        // Apply snapped position if needed.
        if (needsSnap) {
          self.skipBoundsCheck = true;
          selection.$change(x, y, width, height);
          self.skipBoundsCheck = false;
        }

        // Update hidden fields with final position.
        self.updateCropFields({
          x: selection.x,
          y: selection.y,
          width: selection.width,
          height: selection.height,
        });
      });

      // Wait for the cropper image to be fully loaded before configuring.
      if (cropperImage && typeof cropperImage.$ready === 'function') {
        cropperImage.$ready().then(function () {
          // Get natural image dimensions from source element.
          // eslint-disable-next-line prefer-destructuring
          const naturalWidth = self.image.naturalWidth;
          // eslint-disable-next-line prefer-destructuring
          const naturalHeight = self.image.naturalHeight;

          if (!naturalWidth || !naturalHeight) {
            return;
          }

          // Calculate display size based on container width and max height.
          const containerWidth = self.imageWrapper.offsetWidth || 600;
          const maxHeight = 500;
          const imageAspect = naturalWidth / naturalHeight;

          let displayWidth = containerWidth;
          let displayHeight = containerWidth / imageAspect;

          // Constrain to max height.
          if (displayHeight > maxHeight) {
            displayHeight = maxHeight;
            displayWidth = maxHeight * imageAspect;
          }

          // Size canvas to calculated display size.
          canvas.style.width = `${displayWidth}px`;
          canvas.style.height = `${displayHeight}px`;

          // Wait for canvas resize to apply, then scale image.
          setTimeout(function () {
            // Use Cropper.js $center('contain') to scale and center the image.
            cropperImage.$center('contain');

            // Disable image manipulation after centering.
            cropperImage.translatable = false;
            cropperImage.scalable = false;
            cropperImage.rotatable = false;

            // Calculate the scale that $center('contain') applied.
            const scale = Math.min(
              displayWidth / naturalWidth,
              displayHeight / naturalHeight,
            );

            // Calculate the resulting scaled dimensions.
            const scaledWidth = naturalWidth * scale;
            const scaledHeight = naturalHeight * scale;

            // Calculate offsets (image is centered in canvas).
            const offsetX = (displayWidth - scaledWidth) / 2;
            const offsetY = (displayHeight - scaledHeight) / 2;

            // Store scale and offset for coordinate conversion.
            self.displayScale = scale;
            self.imageOffsetX = offsetX;
            self.imageOffsetY = offsetY;

            // Use scaled dimensions for selection calculations.
            const imageWidth = scaledWidth;
            const imageHeight = scaledHeight;

            // Selection coordinates relative to image position in canvas.
            const imageX = offsetX;
            const imageY = offsetY;

            // Store image bounds for selection constraint checking.
            self.imageBounds = {
              x: imageX,
              y: imageY,
              width: imageWidth,
              height: imageHeight,
            };

            // If we have existing crop data, apply it.
            // Otherwise, use resetCropData to set initial default selection.
            if (self.settings.existingCrop && !self.isApplied) {
              self.setCropData(self.settings.existingCrop);
            } else {
              // Set initial selection using the same logic as reset.
              self.resetCropData();
            }

            // Add ready class to trigger fade-in.
            if (self.imageWrapper) {
              self.imageWrapper.classList.add('cropper-ready');
            }

            // Make handles larger for easier grabbing.
            // Must run after cropper is ready so handles exist in DOM.
            // Cropper.js v2 handles have default width/height of 0.
            const handles = selection.querySelectorAll('cropper-handle');
            handles.forEach(function (handle) {
              if (handle.$addStyles) {
                const action = handle.getAttribute('action') || '';
                // Corner handles (larger hit area).
                if (action.includes('-resize') && action.length > 8) {
                  handle.$addStyles(`
                    :host {
                      width: 24px;
                      height: 24px;
                    }
                  `);
                } else if (action.includes('-resize')) {
                  // Edge handles (wider in one dimension).
                  if (action === 'n-resize' || action === 's-resize') {
                    handle.$addStyles(`
                      :host {
                        width: 48px;
                        height: 16px;
                      }
                    `);
                  } else {
                    handle.$addStyles(`
                      :host {
                        width: 16px;
                        height: 48px;
                      }
                    `);
                  }
                }
              }
            });

            // Mark initialization complete.
            self.isInitializingCropper = false;

            // Handle pending auto-apply.
            if (self.pendingAutoApply) {
              self.pendingAutoApply = false;
              self.applyCrop();
            }
          }, 50);
        });
      }
    }
  };

  /**
   * Destroy the Cropper.js instance.
   */
  FilepondCropWidget.prototype.destroyCropper = function () {
    if (this.cropper) {
      this.cropper.destroy();
      this.cropper = null;
    }

    // Manually remove all cropper-canvas elements from the widget.
    // Cropper.js v2's destroy() doesn't always clean up the web components.
    // Check both imageWrapper and the main wrapper since Cropper.js may
    // insert elements in different locations.
    const canvases = this.wrapper.querySelectorAll('cropper-canvas');
    canvases.forEach(function (canvas) {
      canvas.remove();
    });

    // Reset tracking flags.
    this.cropperImageUrl = null;
    this.isInitializingCropper = false;
    this.imageBounds = null;

    // Remove ready class so next image starts hidden.
    if (this.imageWrapper) {
      this.imageWrapper.classList.remove('cropper-ready');
    }
  };

  /**
   * Get the scale factor between original and displayed image.
   *
   * @return {number}
   *   Scale factor (original / natural). Returns 1 if no scaling needed.
   */
  FilepondCropWidget.prototype.getScaleFactor = function () {
    if (!this.image || !this.originalWidth) {
      return 1;
    }

    const naturalWidth = this.image.naturalWidth || this.originalWidth;

    if (naturalWidth <= 0) {
      return 1;
    }

    return this.originalWidth / naturalWidth;
  };

  /**
   * Set crop data on the cropper.
   *
   * @param {object} data
   *   Crop coordinates: x, y, width, height (in original file coordinates).
   */
  FilepondCropWidget.prototype.setCropData = function (data) {
    if (!this.cropper) {
      return;
    }

    const selection = this.cropper.getCropperSelection();
    if (!selection) {
      return;
    }

    // Convert from original file coordinates to canvas coordinates.
    // This is the inverse of updateCropFields.
    const naturalWidth = this.image ? this.image.naturalWidth : 0;
    const naturalHeight = this.image ? this.image.naturalHeight : 0;

    const targetWidth = this.originalWidth || naturalWidth;
    const targetHeight = this.originalHeight || naturalHeight;

    const displayedWidth = naturalWidth * this.displayScale;
    const displayedHeight = naturalHeight * this.displayScale;

    // Scale factor: displayed size / original file size (inverse of updateCropFields).
    const scaleX = targetWidth > 0 ? displayedWidth / targetWidth : 1;
    const scaleY = targetHeight > 0 ? displayedHeight / targetHeight : 1;

    // Convert from original file coordinates to canvas coordinates.
    // 1. Multiply by scale to get displayed size.
    // 2. Add offset to get canvas position.
    const canvasX = (parseFloat(data.x) || 0) * scaleX + this.imageOffsetX;
    const canvasY = (parseFloat(data.y) || 0) * scaleY + this.imageOffsetY;
    const canvasWidth = (parseFloat(data.width) || 100) * scaleX;
    const canvasHeight = (parseFloat(data.height) || 100) * scaleY;

    // Use $change to position the selection box.
    // Skip bounds check for programmatic changes.
    this.skipBoundsCheck = true;
    selection.$change(canvasX, canvasY, canvasWidth, canvasHeight);
    this.skipBoundsCheck = false;
  };

  /**
   * Update hidden fields with crop coordinates.
   *
   * @param {object} data
   *   Selection data with x, y, width, height (in canvas/display coordinates).
   */
  FilepondCropWidget.prototype.updateCropFields = function (data) {
    // Selection coordinates are in canvas space (display pixels).
    // We need to convert to ORIGINAL FILE coordinates.
    //
    // The image shown might be:
    // 1. A FilePond preview canvas (e.g., 400x300)
    // 2. Scaled to fit our display canvas (e.g., 600x400)
    //
    // But the crop needs to be relative to the original file (e.g., 3000x2000).
    //
    // Calculation:
    // - displayScale = displaySize / loadedImageNaturalSize
    // - originalScale = originalFileSize / loadedImageNaturalSize
    // - To convert: canvasCoords / displayScale * originalScale
    //   = canvasCoords * (originalFileSize / displaySize)

    const naturalWidth = this.image ? this.image.naturalWidth : 0;
    const naturalHeight = this.image ? this.image.naturalHeight : 0;

    // If we have original dimensions from server, use those.
    // Otherwise fall back to the loaded image's natural dimensions.
    const targetWidth = this.originalWidth || naturalWidth;
    const targetHeight = this.originalHeight || naturalHeight;

    // Get the actual displayed dimensions (after $center('contain')).
    const displayedWidth = naturalWidth * this.displayScale;
    const displayedHeight = naturalHeight * this.displayScale;

    // Scale factor: original file size / displayed size.
    const scaleX = displayedWidth > 0 ? targetWidth / displayedWidth : 1;
    const scaleY = displayedHeight > 0 ? targetHeight / displayedHeight : 1;

    // Convert from canvas coordinates to original file coordinates.
    // 1. Subtract offset to get position relative to the displayed image.
    // 2. Multiply by scale to get original file coordinates.
    const origX = (data.x - this.imageOffsetX) * scaleX;
    const origY = (data.y - this.imageOffsetY) * scaleY;
    const origWidth = data.width * scaleX;
    const origHeight = data.height * scaleY;

    if (this.fields.x) {
      this.fields.x.value = Math.round(origX);
    }
    if (this.fields.y) {
      this.fields.y.value = Math.round(origY);
    }
    if (this.fields.width) {
      this.fields.width.value = Math.round(origWidth);
    }
    if (this.fields.height) {
      this.fields.height.value = Math.round(origHeight);
    }

    // In direct mode (no preview), mark as applied so form save works.
    if (!this.settings.showCropPreview && this.fields.applied) {
      this.fields.applied.value = '1';
    }
  };

  /**
   * Clear crop coordinate fields.
   */
  FilepondCropWidget.prototype.clearCropFields = function () {
    if (this.fields.x) {
      this.fields.x.value = '';
    }
    if (this.fields.y) {
      this.fields.y.value = '';
    }
    if (this.fields.width) {
      this.fields.width.value = '';
    }
    if (this.fields.height) {
      this.fields.height.value = '';
    }
    if (this.fields.applied) {
      this.fields.applied.value = '0';
    }
  };

  /**
   * Reset crop data to default (centered selection).
   *
   * Clears stored crop data and resets the cropper selection to cover
   * the full image (respecting aspect ratio), centered.
   */
  FilepondCropWidget.prototype.resetCropData = function () {
    // Clear stored crop data.
    this.clearCropFields();
    this.settings.existingCrop = null;
    this.isApplied = false;

    // Reset the cropper selection if cropper exists.
    if (!this.cropper) {
      // eslint-disable-next-line no-console
      console.warn('FilePond Crop: resetCropData - no cropper');
      return;
    }
    if (!this.imageBounds) {
      // eslint-disable-next-line no-console
      console.warn('FilePond Crop: resetCropData - no imageBounds');
      return;
    }

    const selection = this.cropper.getCropperSelection();
    if (!selection) {
      // eslint-disable-next-line no-console
      console.warn('FilePond Crop: resetCropData - no selection');
      return;
    }

    const b = this.imageBounds;

    // Calculate default selection (full image with aspect ratio).
    let selWidth = b.width;
    let selHeight = b.height;

    if (this.settings.aspectRatio) {
      const { aspectRatio } = this.settings;
      if (b.width / b.height > aspectRatio) {
        // Image is wider than aspect ratio - constrain width.
        selWidth = b.height * aspectRatio;
      } else {
        // Image is taller than aspect ratio - constrain height.
        selHeight = b.width / aspectRatio;
      }
    }

    // Center the selection on the image.
    const selX = b.x + (b.width - selWidth) / 2;
    const selY = b.y + (b.height - selHeight) / 2;

    // Apply the default selection.
    // Set flag to skip bounds check (the $change triggers the change event).
    this.skipBoundsCheck = true;
    selection.$change(selX, selY, selWidth, selHeight);
    this.skipBoundsCheck = false;

    // Update hidden fields with the new selection.
    this.updateCropFields({
      x: selX,
      y: selY,
      width: selWidth,
      height: selHeight,
    });
  };

  /**
   * Apply the current crop selection.
   */
  FilepondCropWidget.prototype.applyCrop = function () {
    if (!this.cropper) {
      // eslint-disable-next-line no-console
      console.warn('FilePond Crop: No cropper instance');
      return;
    }

    const selection = this.cropper.getCropperSelection();
    if (!selection) {
      return;
    }

    // Get and store final crop data.
    const data = {
      x: selection.x,
      y: selection.y,
      width: selection.width,
      height: selection.height,
    };
    this.updateCropFields(data);

    // Generate a client-side preview (only in preview mode).
    if (this.settings.showCropPreview) {
      this.updatePreviewFromCropper();
    }

    // Mark as applied.
    if (this.fields.applied) {
      this.fields.applied.value = '1';
    }
    this.isApplied = true;

    // Switch to applied state only in preview mode.
    // In direct mode, stay in cropping state.
    if (this.settings.showCropPreview) {
      this.showAppliedState();
    }
  };

  /**
   * Update the preview image using Cropper.js $toCanvas.
   */
  FilepondCropWidget.prototype.updatePreviewFromCropper = function () {
    if (!this.cropper) {
      return;
    }

    const self = this;
    const selection = this.cropper.getCropperSelection();
    if (!selection) {
      return;
    }

    // Get cropped canvas - limit size for performance.
    selection
      .$toCanvas({
        width: 400,
        height: 400,
      })
      .then(function (canvas) {
        if (!canvas) {
          return;
        }

        // Apply circular mask if enabled.
        let outputCanvas = canvas;
        if (self.settings.circularCrop) {
          outputCanvas = self.applyCircularMask(canvas);
        }

        // Convert to data URL and update preview image.
        // Use PNG for circular to preserve transparency.
        const format = self.settings.circularCrop ? 'image/png' : 'image/jpeg';
        const quality = self.settings.circularCrop ? undefined : 0.9;
        const dataUrl = outputCanvas.toDataURL(format, quality);
        const previewImage = self.preview?.querySelector(
          '[data-filepond-crop="preview-image"]',
        );

        if (previewImage) {
          previewImage.src = dataUrl;
        } else if (self.preview) {
          // Create preview image if it doesn't exist.
          const img = document.createElement('img');
          img.src = dataUrl;
          img.className = 'filepond-crop-preview-image';
          img.setAttribute('data-filepond-crop', 'preview-image');
          self.preview.insertBefore(img, self.preview.firstChild);
        }
      })
      .catch(function (err) {
        // eslint-disable-next-line no-console
        console.error('FilePond Crop: Failed to generate preview', err);
      });
  };

  /**
   * Apply circular mask to a canvas.
   *
   * @param {HTMLCanvasElement} sourceCanvas
   *   The source canvas to mask.
   *
   * @return {HTMLCanvasElement}
   *   A new canvas with circular mask applied.
   */
  FilepondCropWidget.prototype.applyCircularMask = function (sourceCanvas) {
    const canvas = document.createElement('canvas');
    const context = canvas.getContext('2d');
    const { width } = sourceCanvas;
    const { height } = sourceCanvas;

    canvas.width = width;
    canvas.height = height;

    // Draw the source image.
    context.drawImage(sourceCanvas, 0, 0, width, height);

    // Apply circular mask using compositing.
    context.globalCompositeOperation = 'destination-in';
    context.beginPath();
    context.arc(
      width / 2,
      height / 2,
      Math.min(width, height) / 2,
      0,
      Math.PI * 2,
    );
    context.fill();

    return canvas;
  };

  /**
   * Edit the current crop.
   */
  FilepondCropWidget.prototype.editCrop = function () {
    this.isApplied = false;

    // Override settings.existingCrop with current field values.
    this.settings.existingCrop = {
      x: parseFloat(this.fields.x?.value) || 0,
      y: parseFloat(this.fields.y?.value) || 0,
      width: parseFloat(this.fields.width?.value) || 100,
      height: parseFloat(this.fields.height?.value) || 100,
    };

    // Switch to cropper state.
    this.showCropperState();
  };

  /**
   * Remove the current image.
   */
  FilepondCropWidget.prototype.removeImage = function () {
    // Destroy cropper and reset state.
    this.destroyCropper();
    this.currentImageUrl = null;
    this.isApplied = false;
    this.pendingAutoApply = false;

    // Clear existing crop data so it's not applied to new uploads.
    this.settings.existingCrop = null;

    // Clear crop fields and file ID.
    this.clearCropFields();
    if (this.fields.fid) {
      this.fields.fid.value = '';
    }

    // Clear image element src.
    if (this.image) {
      this.image.src = '';
    }

    // Clear preview image if present.
    const previewImage = this.preview?.querySelector(
      '[data-filepond-crop="preview-image"]',
    );
    if (previewImage) {
      previewImage.remove();
    }

    // Remove file from FilePond if present.
    if (this.pond) {
      const files = this.pond.getFiles();
      files.forEach(
        function (file) {
          this.pond.removeFile(file.id);
        }.bind(this),
      );
    }

    // Reset FilePond's inline height (it persists after file removal).
    const filepondRoot = this.wrapper.querySelector('.filepond--root');
    if (filepondRoot) {
      filepondRoot.style.height = '';
    }

    // Show FilePond state.
    this.showFilePondState();
  };

  /**
   * Behavior for FilePond Crop widgets.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.filepondCrop = {
    attach(context, settings) {
      const cropSettings = settings.filepondCrop || {};

      Object.keys(cropSettings).forEach(function (elementId) {
        const widgetSettings = cropSettings[elementId];
        const wrapper = context.querySelector(`#${elementId}`);

        if (!wrapper) {
          return;
        }

        // Use once() to prevent re-initialization.
        once('filepond-crop', wrapper).forEach(function (element) {
          // Create widget controller.
          element._filepondCrop = new FilepondCropWidget(
            element,
            widgetSettings,
          );
        });
      });
    },
  };
})(Drupal, drupalSettings, once);
