<?php

declare(strict_types=1);

namespace Drupal\filepond_crop;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\crop\CropInterface;
use Drupal\crop\Entity\Crop;

/**
 * Manages crop entity operations for FilePond uploads.
 */
class CropManager {

  /**
   * The crop entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $cropStorage;

  /**
   * The file entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $fileStorage;

  /**
   * The crop type entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $cropTypeStorage;

  /**
   * The image style entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $imageStyleStorage;

  /**
   * Constructs a CropManager.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager) {
    $this->cropStorage = $entityTypeManager->getStorage('crop');
    $this->fileStorage = $entityTypeManager->getStorage('file');
    $this->cropTypeStorage = $entityTypeManager->getStorage('crop_type');
    $this->imageStyleStorage = $entityTypeManager->getStorage('image_style');
  }

  /**
   * Saves or updates a crop for a file.
   *
   * @param int|string $file_id
   *   The file entity ID.
   * @param string $crop_type_id
   *   The crop type machine name.
   * @param array $coordinates
   *   Cropper.js coordinates (top-left based): x, y, width, height.
   *
   * @return \Drupal\crop\CropInterface
   *   The saved crop entity.
   *
   * @throws \InvalidArgumentException
   *   If the file or crop type is not found.
   */
  public function saveCrop(int|string $file_id, string $crop_type_id, array $coordinates): CropInterface {
    /** @var \Drupal\file\FileInterface|null $file */
    $file = $this->fileStorage->load($file_id);
    if (!$file) {
      throw new \InvalidArgumentException("File not found: $file_id");
    }

    /** @var \Drupal\crop\Entity\CropType|null $crop_type */
    $crop_type = $this->cropTypeStorage->load($crop_type_id);
    if (!$crop_type) {
      throw new \InvalidArgumentException("Crop type not found: $crop_type_id");
    }

    $uri = $file->getFileUri();

    // Convert Cropper.js coordinates (top-left) to Crop module (center).
    $center_coords = $this->convertToCenter($coordinates);

    // Check for existing crop.
    $crop = Crop::findCrop($uri, $crop_type_id);

    if ($crop) {
      // Update existing crop.
      $crop->set('x', $center_coords['x']);
      $crop->set('y', $center_coords['y']);
      $crop->set('width', $center_coords['width']);
      $crop->set('height', $center_coords['height']);
    }
    else {
      // Create new crop.
      $crop = $this->cropStorage->create([
        'type' => $crop_type_id,
        'entity_id' => $file_id,
        'entity_type' => 'file',
        'uri' => $uri,
        'x' => $center_coords['x'],
        'y' => $center_coords['y'],
        'width' => $center_coords['width'],
        'height' => $center_coords['height'],
      ]);
    }

    $crop->save();

    // Flush cached image style derivatives that use this crop type.
    $this->flushImageStyles($crop_type_id, $uri);

    return $crop;
  }

  /**
   * Finds an existing crop for a file.
   *
   * @param string $file_uri
   *   The file URI.
   * @param string $crop_type_id
   *   The crop type machine name.
   *
   * @return \Drupal\crop\CropInterface|null
   *   The crop entity, or NULL if not found.
   */
  public function findCrop(string $file_uri, string $crop_type_id): ?CropInterface {
    return Crop::findCrop($file_uri, $crop_type_id);
  }

  /**
   * Deletes a crop for a file.
   *
   * @param string $file_uri
   *   The file URI.
   * @param string $crop_type_id
   *   The crop type machine name.
   */
  public function deleteCrop(string $file_uri, string $crop_type_id): void {
    $crops = $this->cropStorage->loadByProperties([
      'type' => $crop_type_id,
      'uri' => $file_uri,
    ]);

    if (!empty($crops)) {
      $this->cropStorage->delete($crops);
    }
  }

  /**
   * Converts Cropper.js coordinates to Crop module format.
   *
   * Cropper.js uses top-left corner coordinates.
   * Crop module stores center coordinates.
   *
   * @param array $cropper_coords
   *   Keys: x, y, width, height (top-left based).
   *
   * @return array
   *   Keys: x, y, width, height (center-based for Crop entity).
   */
  public function convertToCenter(array $cropper_coords): array {
    $x = (float) ($cropper_coords['x'] ?? 0);
    $y = (float) ($cropper_coords['y'] ?? 0);
    $width = (float) ($cropper_coords['width'] ?? 0);
    $height = (float) ($cropper_coords['height'] ?? 0);

    return [
      'x' => (int) round($x + ($width / 2)),
      'y' => (int) round($y + ($height / 2)),
      'width' => (int) round($width),
      'height' => (int) round($height),
    ];
  }

  /**
   * Converts Crop module coordinates to Cropper.js format.
   *
   * @param \Drupal\crop\CropInterface $crop
   *   The crop entity.
   *
   * @return array
   *   Keys: x, y, width, height (top-left based for Cropper.js).
   */
  public function convertToTopLeft(CropInterface $crop): array {
    // Crop::anchor() already calculates top-left from center.
    $anchor = $crop->anchor();
    $size = $crop->size();

    return [
      'x' => $anchor['x'],
      'y' => $anchor['y'],
      'width' => $size['width'],
      'height' => $size['height'],
    ];
  }

  /**
   * Gets the aspect ratio for a crop type.
   *
   * @param string $crop_type_id
   *   The crop type machine name.
   *
   * @return float|null
   *   The aspect ratio as a float, or NULL if not set or invalid.
   */
  public function getAspectRatio(string $crop_type_id): ?float {
    /** @var \Drupal\crop\Entity\CropType|null $crop_type */
    $crop_type = $this->cropTypeStorage->load($crop_type_id);
    if (!$crop_type) {
      return NULL;
    }

    $ratio_string = $crop_type->getAspectRatio();
    if (empty($ratio_string)) {
      return NULL;
    }

    return $this->parseAspectRatio($ratio_string);
  }

  /**
   * Parses an aspect ratio string to a float.
   *
   * @param string $ratio_string
   *   The ratio string (e.g., "16:9", "1:1", "500:180").
   *
   * @return float|null
   *   The ratio as width/height, or NULL if invalid.
   */
  public function parseAspectRatio(string $ratio_string): ?float {
    if (empty($ratio_string) || !str_contains($ratio_string, ':')) {
      return NULL;
    }

    $parts = explode(':', $ratio_string);
    if (count($parts) !== 2) {
      return NULL;
    }

    $width = (float) $parts[0];
    $height = (float) $parts[1];

    if ($height <= 0) {
      return NULL;
    }

    return $width / $height;
  }

  /**
   * Gets all available crop types as options.
   *
   * @return array
   *   An array of crop type labels keyed by machine name.
   */
  public function getCropTypeOptions(): array {
    $options = [];

    /** @var \Drupal\crop\Entity\CropType[] $crop_types */
    $crop_types = $this->cropTypeStorage->loadMultiple();
    foreach ($crop_types as $crop_type) {
      $options[$crop_type->id()] = $crop_type->label();
    }

    return $options;
  }

  /**
   * Gets image style options excluding those that use crop effects.
   *
   * Useful for the cropper UI image style setting, where using a crop-based
   * style would pre-crop the image before showing it in the cropper.
   *
   * @return array
   *   An array of image style labels keyed by machine name.
   */
  public function getImageStyleOptionsWithoutCrop(): array {
    $options = [];

    /** @var \Drupal\image\ImageStyleInterface[] $image_styles */
    $image_styles = $this->imageStyleStorage->loadMultiple();

    foreach ($image_styles as $image_style) {
      $uses_crop = FALSE;
      foreach ($image_style->getEffects() as $effect) {
        $config = $effect->getConfiguration();
        // Check if this effect uses a crop type (e.g., crop_crop effect).
        if (!empty($config['data']['crop_type'])) {
          $uses_crop = TRUE;
          break;
        }
      }

      if (!$uses_crop) {
        $options[$image_style->id()] = $image_style->label();
      }
    }

    return $options;
  }

  /**
   * Gets all image styles that use a specific crop type.
   *
   * @param string $crop_type_id
   *   The crop type machine name.
   *
   * @return \Drupal\image\ImageStyleInterface[]
   *   Array of image styles that use this crop type.
   */
  public function getImageStylesByCrop(string $crop_type_id): array {
    $styles = [];

    /** @var \Drupal\image\ImageStyleInterface[] $image_styles */
    $image_styles = $this->imageStyleStorage->loadMultiple();

    foreach ($image_styles as $image_style) {
      foreach ($image_style->getEffects() as $effect) {
        $config = $effect->getConfiguration();
        // Check if this effect uses a crop type (e.g., crop_crop effect).
        if (!empty($config['data']['crop_type']) && $config['data']['crop_type'] === $crop_type_id) {
          $styles[] = $image_style;
          // Only add each style once.
          break;
        }
      }
    }

    return $styles;
  }

  /**
   * Flushes cached image style derivatives for a file.
   *
   * @param string $crop_type_id
   *   The crop type machine name.
   * @param string $file_uri
   *   The file URI.
   */
  public function flushImageStyles(string $crop_type_id, string $file_uri): void {
    $image_styles = $this->getImageStylesByCrop($crop_type_id);

    foreach ($image_styles as $image_style) {
      // Flush the cached derivative for this specific file.
      $image_style->flush($file_uri);
    }
  }

}
