<?php

declare(strict_types=1);

namespace Drupal\filepond_crop\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crop\Entity\Crop;
use Drupal\filepond\ImageDimensionHelper;
use Drupal\filepond\Plugin\Field\FieldWidget\FilePondImageWidget;
use Drupal\filepond_crop\CropManager;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * FilePond image widget with Cropper.js integration.
 *
 * Uses pure JavaScript for the crop UI - no AJAX form rebuilds.
 * After FilePond upload, Cropper.js appears inline. Crop coordinates
 * are stored in hidden fields and saved as Crop entities on form submit.
 *
 * Single-cardinality image fields only.
 */
#[FieldWidget(
  id: 'filepond_image_crop',
  label: new TranslatableMarkup('FilePond Image with Crop'),
  field_types: ['image'],
)]
class FilePondCropWidget extends FilePondImageWidget {

  /**
   * The crop manager service.
   *
   * @var \Drupal\filepond_crop\CropManager
   */
  protected CropManager $cropManager;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->cropManager = $container->get('filepond_crop.crop_manager');
    $instance->logger = $container->get('logger.factory')->get('filepond_crop');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'crop_type' => '',
      'show_default_crop' => TRUE,
      'show_crop_preview' => TRUE,
      'crop_preview_image_style' => '',
      'cropper_image_style' => '',
      'circular_crop' => FALSE,
      'min_crop_size' => 0,
      'show_reset_button' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $element = parent::settingsForm($form, $form_state);

    // Hide preview_image_style - crop widget uses integrated layout without
    // FilePond image previews (just shows filename).
    $element['preview_image_style'] = [
      '#type' => 'value',
      '#value' => '',
    ];

    // Get available crop types.
    $crop_type_options = $this->cropManager->getCropTypeOptions();

    if (empty($crop_type_options)) {
      $element['crop_warning'] = [
        '#markup' => $this->t('<strong>Warning:</strong> No crop types found. Please create a crop type first.'),
      ];
    }

    $element['crop_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Crop type'),
      '#options' => $crop_type_options,
      '#default_value' => $this->getSetting('crop_type'),
      '#required' => TRUE,
      '#description' => $this->t('The crop type determines the aspect ratio.'),
    ];

    $element['circular_crop'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Circular crop'),
      '#default_value' => $this->getSetting('circular_crop'),
      '#description' => $this->t('Display crop selection as a circle. Useful for avatars. Forces 1:1 aspect ratio.'),
    ];

    $element['cropper_image_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Cropper image style'),
      '#options' => $this->cropManager->getImageStyleOptionsWithoutCrop(),
      '#empty_option' => $this->t('- Original image -'),
      '#default_value' => $this->getSetting('cropper_image_style'),
      '#description' => $this->t('Image style for the cropper UI. Styles using crop effects are excluded.'),
    ];

    $element['show_crop_preview'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show crop preview'),
      '#default_value' => $this->getSetting('show_crop_preview'),
      '#description' => $this->t('Show Apply Crop button and preview. If unchecked, the cropper is always visible and crop is saved with the form.'),
    ];

    $element['show_default_crop'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show default crop on upload'),
      '#default_value' => $this->getSetting('show_default_crop'),
      '#description' => $this->t('Automatically apply a default crop covering the full image.'),
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $this->fieldDefinition->getName() . '][settings_edit_form][settings][show_crop_preview]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $element['crop_preview_image_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Crop preview image style'),
      '#options' => image_style_options(FALSE),
      '#empty_option' => $this->t('- None -'),
      '#default_value' => $this->getSetting('crop_preview_image_style'),
      '#description' => $this->t('Image style to show after crop is applied. Should use the same crop type.'),
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $this->fieldDefinition->getName() . '][settings_edit_form][settings][show_crop_preview]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $element['min_crop_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum crop width'),
      '#default_value' => $this->getSetting('min_crop_size'),
      '#min' => 0,
      '#field_suffix' => 'px',
      '#description' => $this->t('Minimum width of the cropped output in pixels. Set to 0 for no minimum.'),
    ];

    $element['show_reset_button'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show reset button'),
      '#default_value' => $this->getSetting('show_reset_button'),
      '#description' => $this->t('Show a button to reset the crop selection to the default centered position.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = parent::settingsSummary();

    // Remove "Preview:" line from parent since crop widget doesn't use
    // FilePond previews.
    $summary = array_filter($summary, function ($item) {
      $string = $item instanceof TranslatableMarkup ? $item->getUntranslatedString() : (string) $item;
      return strpos($string, 'Preview:') !== 0;
    });

    $crop_type = $this->getSetting('crop_type');
    if ($crop_type) {
      $crop_types = $this->cropManager->getCropTypeOptions();
      $summary[] = $this->t('Crop type: @type', [
        '@type' => $crop_types[$crop_type] ?? $crop_type,
      ]);
    }
    else {
      $summary[] = $this->t('Crop type: Not configured');
    }

    $cropper_style = $this->getSetting('cropper_image_style');
    if ($cropper_style) {
      $styles = image_style_options(FALSE);
      $summary[] = $this->t('Cropper: @style', [
        '@style' => $styles[$cropper_style] ?? $cropper_style,
      ]);
    }

    if ($this->getSetting('show_default_crop')) {
      $summary[] = $this->t('Auto-apply crop');
    }

    if (!$this->getSetting('show_crop_preview')) {
      $summary[] = $this->t('Direct mode (no preview)');
    }

    if ($this->getSetting('circular_crop')) {
      $summary[] = $this->t('Circular crop');
    }

    $min_size = $this->getSetting('min_crop_size');
    if ($min_size > 0) {
      $summary[] = $this->t('Min width: @size', ['@size' => $min_size . 'px']);
    }

    if ($this->getSetting('show_reset_button')) {
      $summary[] = $this->t('Reset button');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(
    FieldItemListInterface $items,
    $delta,
    array $element,
    array &$form,
    FormStateInterface $form_state,
  ): array {
    // Get parent element (FilePond uploader).
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    // If parent returned empty (delta > 0 or no access), return as-is.
    if (empty($element) || !isset($element['#type'])) {
      return $element;
    }

    $crop_type_id = $this->getSetting('crop_type');
    if (empty($crop_type_id)) {
      return $element;
    }

    $field_name = $this->fieldDefinition->getName();

    // Get existing file if any.
    $existing_file = NULL;
    $existing_crop_data = NULL;
    $has_applied_crop = FALSE;
    $first_item = $items->first();

    if ($first_item && !empty($first_item->target_id)) {
      /** @var \Drupal\file\FileInterface|null $existing_file */
      $existing_file = $this->entityTypeManager->getStorage('file')->load($first_item->target_id);

      if ($existing_file) {
        // Check for existing crop.
        $crop = Crop::findCrop($existing_file->getFileUri(), $crop_type_id);
        if ($crop) {
          $existing_crop_data = $this->cropManager->convertToTopLeft($crop);
          $existing_crop_data['applied'] = TRUE;
          $has_applied_crop = TRUE;
        }
      }
    }

    // When there's an existing file with a crop, don't load it into FilePond.
    // We'll show a static preview instead and FilePond stays empty/hidden.
    if ($has_applied_crop) {
      $element['#default_value'] = [];
    }

    // Use integrated layout for FilePond - no item panels since we show
    // the cropper separately. Also disable image preview.
    $element['#config']['stylePanelLayout'] = 'integrated';
    $element['#config']['allowImagePreview'] = TRUE;

    // Generate unique element ID for JS targeting.
    $element_id = 'filepond-crop-' . $field_name;

    // Determine initial state class.
    $state_class = 'filepond-crop--empty';
    if ($has_applied_crop) {
      $state_class = 'filepond-crop--applied';
    }

    // Convert the element to a fieldset wrapper (like entity_browser does).
    // This keeps form value paths intact while adding fieldset wrapper.
    // Store the original filepond config before changing type.
    $filepond_config = [
      '#type' => $element['#type'],
      '#default_value' => $element['#default_value'] ?? [],
      '#config' => $element['#config'] ?? [],
      '#extensions' => $element['#extensions'] ?? NULL,
      '#max_filesize' => $element['#max_filesize'] ?? NULL,
      '#max_files' => $element['#max_files'] ?? NULL,
      '#upload_location' => $element['#upload_location'] ?? NULL,
      '#context' => $element['#context'] ?? [],
      '#columns' => $element['#columns'] ?? 5,
      '#max_width' => $element['#max_width'] ?? NULL,
    ];

    // Change element to fieldset and add our ID/classes.
    $element['#type'] = 'fieldset';
    $element['#title'] = $this->fieldDefinition->getLabel();
    $element['#id'] = $element_id;
    $element['#attributes']['id'] = $element_id;
    $element['#attributes']['class'][] = 'filepond-crop-widget';
    $element['#attributes']['class'][] = $state_class;

    // Add circular crop class if enabled.
    if ($this->getSetting('circular_crop')) {
      $element['#attributes']['class'][] = 'filepond-crop--circular';
    }

    // Add the filepond uploader as a child element.
    $element['uploader'] = $filepond_config;

    // Add crop container.
    $element['crop_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['filepond-crop-container'],
        'data-filepond-crop' => 'container',
      ],
      '#weight' => 100,
    ];

    // Image element for Cropper.js (used when editing crop).
    // Visibility controlled by CSS state classes on wrapper.
    $element['crop_container']['image_wrapper'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['filepond-crop-image-wrapper'],
        'data-filepond-crop' => 'image-wrapper',
      ],
    ];

    // Cropper.js will attach to this image.
    $element['crop_container']['image_wrapper']['image'] = [
      '#type' => 'html_tag',
      '#tag' => 'img',
      '#attributes' => [
        'class' => ['filepond-crop-image'],
        'data-filepond-crop' => 'image',
        'src' => $existing_file ? $existing_file->createFileUrl() : '',
      ],
    ];

    $show_crop_preview = $this->getSetting('show_crop_preview');

    // Preview container only shown in preview mode.
    if ($show_crop_preview) {
      // Preview container - visibility controlled by CSS state classes.
      $element['crop_container']['preview'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['filepond-crop-preview'],
          'data-filepond-crop' => 'preview',
        ],
      ];

      // Add static preview image using image style if available.
      $preview_image_style = $this->getSetting('crop_preview_image_style');

      if ($existing_file && $has_applied_crop) {
        if ($preview_image_style) {
          $element['crop_container']['preview']['image'] = [
            '#theme' => 'image_style',
            '#style_name' => $preview_image_style,
            '#uri' => $existing_file->getFileUri(),
            '#attributes' => [
              'class' => ['filepond-crop-preview-image'],
              'data-filepond-crop' => 'preview-image',
            ],
          ];
        }
        else {
          // Fallback to original image if no style configured.
          $element['crop_container']['preview']['image'] = [
            '#type' => 'html_tag',
            '#tag' => 'img',
            '#attributes' => [
              'src' => $existing_file->createFileUrl(),
              'class' => ['filepond-crop-preview-image'],
              'data-filepond-crop' => 'preview-image',
            ],
          ];
        }
      }
    }

    // Action buttons.
    $element['crop_container']['actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['filepond-crop-actions']],
    ];

    // Apply button only shown in preview mode.
    if ($show_crop_preview) {
      $element['crop_container']['actions']['apply'] = [
        '#type' => 'html_tag',
        '#tag' => 'button',
        '#value' => $this->t('Apply Crop'),
        '#attributes' => [
          'class' => ['button', 'button--small', 'filepond-crop-apply'],
          'data-filepond-crop' => 'apply',
          'type' => 'button',
        ],
      ];
      $element['crop_container']['actions']['edit'] = [
        '#type' => 'html_tag',
        '#tag' => 'button',
        '#value' => $this->t('Edit Crop'),
        '#attributes' => [
          'class' => ['button', 'button--small', 'filepond-crop-edit'],
          'data-filepond-crop' => 'edit',
          'type' => 'button',
        ],
      ];
    }

    $element['crop_container']['actions']['remove'] = [
      '#type' => 'html_tag',
      '#tag' => 'button',
      '#value' => $this->t('Remove Image'),
      '#attributes' => [
        'class' => [
          'button',
          'button--danger',
          'button--small',
          'filepond-crop-remove',
        ],
        'data-filepond-crop' => 'remove',
        'type' => 'button',
      ],
    ];

    // Reset button (optional, shown only in cropping state).
    if ($this->getSetting('show_reset_button')) {
      $element['crop_container']['actions']['reset'] = [
        '#type' => 'html_tag',
        '#tag' => 'button',
        '#value' => $this->t('Reset Crop'),
        '#attributes' => [
          'class' => [
            'button',
            'button--small',
            'button-cancel',
            'filepond-crop-reset',
          ],
          'data-filepond-crop' => 'reset',
          'type' => 'button',
        ],
      ];
    }

    // Hidden fields for crop coordinates and file ID.
    $element['crop'] = [
      '#type' => 'container',
      '#tree' => TRUE,
    ];

    // Store file ID so we know which file to keep on submit.
    $element['crop']['fid'] = [
      '#type' => 'hidden',
      '#default_value' => $existing_file ? $existing_file->id() : '',
      '#attributes' => ['data-filepond-crop' => 'fid'],
    ];

    $element['crop']['x'] = [
      '#type' => 'hidden',
      '#default_value' => $existing_crop_data['x'] ?? '',
      '#attributes' => ['data-filepond-crop' => 'x'],
    ];

    $element['crop']['y'] = [
      '#type' => 'hidden',
      '#default_value' => $existing_crop_data['y'] ?? '',
      '#attributes' => ['data-filepond-crop' => 'y'],
    ];

    $element['crop']['width'] = [
      '#type' => 'hidden',
      '#default_value' => $existing_crop_data['width'] ?? '',
      '#attributes' => ['data-filepond-crop' => 'width'],
    ];

    $element['crop']['height'] = [
      '#type' => 'hidden',
      '#default_value' => $existing_crop_data['height'] ?? '',
      '#attributes' => ['data-filepond-crop' => 'height'],
    ];

    $element['crop']['applied'] = [
      '#type' => 'hidden',
      '#default_value' => $has_applied_crop ? '1' : '0',
      '#attributes' => ['data-filepond-crop' => 'applied'],
    ];

    // Get aspect ratio for JS.
    $aspect_ratio = $this->cropManager->getAspectRatio($crop_type_id);

    // Get cropper image style and original dimensions.
    $cropper_image_style = $this->getSetting('cropper_image_style');
    $original_width = 0;
    $original_height = 0;
    $cropper_image_url = NULL;

    if ($existing_file) {
      // Get original image dimensions via helper (checks cache, tempstore,
      // then FastImage for S3 - only reads header bytes).
      $dimensions = ImageDimensionHelper::getDimensions($existing_file);
      if (!empty($dimensions)) {
        $original_width = $dimensions['width'];
        $original_height = $dimensions['height'];
      }

      // Build cropper image URL (with optional image style).
      if ($cropper_image_style) {
        /** @var \Drupal\image\ImageStyleInterface|null $style */
        $style = $this->entityTypeManager->getStorage('image_style')->load($cropper_image_style);
        if ($style) {
          $uri = $existing_file->getFileUri();
          // Pre-generate the derivative so it's ready immediately.
          $derivative_uri = $style->buildUri($uri);
          if (!file_exists($derivative_uri)) {
            $style->createDerivative($uri, $derivative_uri);
          }
          $cropper_image_url = $style->buildUrl($uri);
        }
      }
      if (!$cropper_image_url) {
        $cropper_image_url = $existing_file->createFileUrl();
      }
    }

    // Circular crop forces 1:1 aspect ratio.
    $circular_crop = (bool) $this->getSetting('circular_crop');
    if ($circular_crop) {
      $aspect_ratio = 1;
    }

    // Attach library and settings.
    $element['#attached']['library'][] = 'filepond_crop/filepond_crop';
    $element['#attached']['drupalSettings']['filepondCrop'][$element_id] = [
      'elementId' => $element_id,
      'hasImage' => !empty($existing_file),
      'hasAppliedCrop' => $has_applied_crop,
      'fileId' => $existing_file ? $existing_file->id() : NULL,
      'imageUrl' => $cropper_image_url,
      'originalWidth' => $original_width,
      'originalHeight' => $original_height,
      'existingCrop' => $existing_crop_data,
      'aspectRatio' => $aspect_ratio,
      'cropType' => $crop_type_id,
      'cropperImageStyle' => $cropper_image_style ?: NULL,
      'showDefaultCrop' => (bool) $this->getSetting('show_default_crop'),
      'showCropPreview' => (bool) $this->getSetting('show_crop_preview'),
      'cropPreviewImageStyle' => $this->getSetting('crop_preview_image_style') ?: NULL,
      'circularCrop' => $circular_crop,
      'minCropSize' => (int) $this->getSetting('min_crop_size'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(
    array $values,
    array $form,
    FormStateInterface $form_state,
  ): array {
    // The filepond uploader is now nested under 'uploader' key due to fieldset.
    // Move its values up so parent can process them.
    if (isset($values[0]['uploader'])) {
      $values[0] = array_merge($values[0], $values[0]['uploader']);
    }

    // Let parent handle file processing.
    $result = parent::massageFormValues($values, $form, $form_state);

    // Get crop data from form values.
    $crop_data = $values[0]['crop'] ?? [];
    $crop_type_id = $this->getSetting('crop_type');

    if (empty($crop_type_id)) {
      return $result;
    }

    // Get file ID from parent result, or from hidden field if parent is empty.
    // The hidden field is used when editing existing files with applied crops,
    // where we don't load the file into FilePond.
    $file_id = $result[0]['target_id'] ?? NULL;
    if (!$file_id && !empty($crop_data['fid'])) {
      $file_id = $crop_data['fid'];
      // Rebuild result with the file ID from hidden field.
      $result = [['target_id' => $file_id]];
    }

    if (!$file_id) {
      return $result;
    }

    // Check if crop coordinates are set (can be from cropper or auto-default).
    $has_crop_data = !empty($crop_data['width']) && !empty($crop_data['height']);

    if ($has_crop_data) {
      try {
        $this->cropManager->saveCrop($file_id, $crop_type_id, [
          'x' => (float) ($crop_data['x'] ?? 0),
          'y' => (float) ($crop_data['y'] ?? 0),
          'width' => (float) $crop_data['width'],
          'height' => (float) $crop_data['height'],
        ]);
      }
      catch (\Exception $e) {
        // Log but don't block form submission.
        $this->logger->error('Failed to save crop: @message', [
          '@message' => $e->getMessage(),
        ]);
      }
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition): bool {
    // Only for single-cardinality image fields.
    if ($field_definition->getType() !== 'image') {
      return FALSE;
    }

    $cardinality = $field_definition->getFieldStorageDefinition()->getCardinality();
    return $cardinality === 1;
  }

}
