<?php

declare(strict_types=1);

namespace Drupal\filepond_views\Plugin\views\area;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\filepond\FilePondConfigFormTrait;
use Drupal\views\Attribute\ViewsArea;
use Drupal\views\Plugin\views\area\AreaPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Views area handler to display a FilePond uploader.
 *
 * Creates media entities on upload and refreshes the view to display them.
 * Uses a lazy builder for the actual uploader so the view can be cached.
 *
 * @ingroup views_area_handlers
 */
#[ViewsArea("filepond_upload")]
class FilePondUploadArea extends AreaPluginBase {

  use FilePondConfigFormTrait;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions(): array {
    $options = parent::defineOptions();

    // Use defaults from the trait.
    foreach ($this->getFilePondDefaults() as $key => $value) {
      $options[$key] = ['default' => $value];
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {

    parent::buildOptionsForm($form, $form_state);

    // Use shared form builder from trait.
    // Extensions always come from media type, so hide that field.
    $form = $this->buildFilePondConfigForm($form, $this->options, [
      'media_type_options' => filepond_get_file_based_media_types(),
      'show_auto_select' => $this->view->getDisplay()->getPluginId() === 'entity_browser',
      'show_extensions' => FALSE,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::submitOptionsForm($form, $form_state);
    // Process values (e.g., append 'M' to max_filesize).
    $options = $form_state->getValue('options');
    $options = $this->processFilePondConfigValues($options);
    $form_state->setValue('options', $options);
  }

  /**
   * Determines the area type (header/footer/empty) for this handler.
   *
   * @return string
   *   The area type.
   */
  protected function getAreaType(): string {
    $handler_id = $this->options['id'] ?? NULL;
    if (!$handler_id) {
      return 'header';
    }

    // Check which area contains this handler by checking display options.
    foreach (['header', 'footer', 'empty'] as $area_type) {
      $handlers = $this->view->display_handler->getOption($area_type) ?? [];
      if (isset($handlers[$handler_id])) {
        return $area_type;
      }
    }

    // Default to header if not found (shouldn't happen).
    return 'header';
  }

  /**
   * {@inheritdoc}
   */
  public function render($empty = FALSE): array {
    // Respect the "empty" setting.
    if ($empty && empty($this->options['empty'])) {
      return [];
    }

    // Get view identifiers for the lazy builder.
    $view_id = $this->view->storage->id();
    $display_id = $this->view->current_display;
    $view_dom_id = $this->view->dom_id;
    $area_type = $this->getAreaType();
    $handler_id = $this->options['id'] ?? 'filepond_upload';

    // Use lazy builder so the view itself can be cached.
    // The uploader has user-specific and request-specific data
    // (permissions, Entity Browser cardinality) that must render dynamically.
    return [
      '#lazy_builder' => [
        'filepond_views.lazy_builder:renderUploader',
        [
          $view_id,
          $display_id,
          $view_dom_id,
          $area_type,
          $handler_id,
        ],
      ],
      '#create_placeholder' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    // The uploader is never empty - it always shows.
    return FALSE;
  }

}
