<?php

declare(strict_types=1);

namespace Drupal\filepond\Element;

use Drupal\Component\Utility\Bytes;
use Drupal\Component\Utility\Environment;
use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\FormElementBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\image\Entity\ImageStyle;

/**
 * Provides a FilePond form element.
 *
 * Unlike dropzonejs which stores temp file paths and creates file entities
 * on form submit, this element creates file entities during upload and
 * stores file IDs. This simplifies form handling as you just get file IDs.
 *
 * Properties:
 * - #max_filesize (string): Max file size (e.g., '10M', '500K'). Defaults to
 *   PHP upload_max_filesize.
 * - #extensions (string): Allowed extensions space-separated. Defaults to
 *   'jpg jpeg gif png'.
 * - #max_files (int): Max number of files. 0 or NULL for unlimited.
 * - #upload_location (string): Destination URI for files. Supports tokens.
 *   Defaults to config or 'public://filepond-uploads'.
 * - #upload_prompt (string): Message shown in the drop area. Use the
 *   filepond--label-action class on a span for clickable text styling.
 * - #chunk_size (int): Chunk size in bytes for chunked uploads. Defaults to
 *   5MB.
 * - #allow_reorder (bool): Allow drag-to-reorder files. Only works in single
 *   column mode. Defaults to FALSE.
 * - #preview_image_style (string): Image style for displaying existing files.
 *   Uses FilePond's file-poster plugin to show thumbnails. NULL for direct URL.
 *
 * Returns an array with 'fids' key containing array of file entity IDs.
 *
 * @FormElement("filepond")
 */
class FilePond extends FormElementBase {

  /**
   * Default chunk size (5MB).
   */
  public const DEFAULT_CHUNK_SIZE = 5242880;

  /**
   * {@inheritdoc}
   *
   * Root properties (familiar Drupal file element properties):
   * - #default_value (array): Array of existing file IDs.
   * - #required (bool): Whether the field is required.
   * - #extensions (string): Space-separated allowed extensions.
   *   Default: 'jpg jpeg gif png'.
   * - #max_filesize (string): Max file size like '20M'.
   *   Default: PHP upload_max_filesize.
   * - #max_files (int): Max files allowed. 0 = unlimited.
   *   Default: from module config.
   * - #upload_location (string): Destination URI. Supports tokens.
   *
   * These root properties are used for server-side validation and file storage.
   * They mirror Drupal's managed_file element properties. 'acceptedFileTypes'
   * sent to FilePond JS is derived from #extensions and cannot be set directly,
   * so server and client validation always match.
   *
   * FilePond UI options (#config):
   * - #config (array): FilePond-specific UI options. See processFilePond().
   *
   * If someone sets a root property in #config (e.g., #config['extensions']),
   * it will be ignored - root properties always take precedence.
   */
  public function getInfo(): array {
    $class = static::class;
    return [
      '#input' => TRUE,
      '#process' => [[$class, 'processFilePond']],
      '#pre_render' => [[$class, 'preRenderFilePond']],
      '#theme' => 'filepond',
      '#theme_wrappers' => ['form_element'],
      '#tree' => TRUE,
      '#default_value' => [],
      // Server-side properties (familiar Drupal patterns).
      '#extensions' => NULL,
      '#max_filesize' => NULL,
      '#max_files' => NULL,
      '#upload_location' => NULL,
      // Custom context for event subscribers.
      '#context' => [],
      // FilePond UI options.
      '#config' => [],
      // CSS grid sizing (optional).
      '#columns' => 5,
      '#max_width' => NULL,
    ];
  }

  /**
   * Processes a FilePond form element.
   *
   * Sets defaults for root properties (server-side) and #config (UI options).
   *
   * Root properties (server-side, familiar Drupal patterns):
   * - #extensions: Space-separated allowed extensions.
   * - #max_filesize: Max file size like '20M'.
   * - #max_files: Max files allowed. 0 = unlimited.
   * - #upload_location: Destination URI. Supports tokens.
   *
   * #config options (FilePond UI):
   * - labelIdle (string): Drop area prompt text.
   * - allowReorder (bool): Enable drag-to-reorder.
   * - maxParallelUploads (int): Simultaneous uploads.
   * - styleItemPanelAspectRatio (float): Panel aspect ratio.
   * - previewFitMode (string): 'contain' or 'cover'.
   * - previewImageStyle (string|null): Image style for thumbnails.
   * - labelProcessing (string): Text shown during server processing.
   * - chunkSize (int): Chunk size in bytes.
   * - imagePreviewMaxFileSize (string): Max size for image preview generation.
   *
   * Note: 'acceptedFileTypes' is derived from #extensions and cannot be
   * overridden to ensure server and client validation always match.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $complete_form
   *   The complete form.
   *
   * @return array
   *   The processed element.
   */
  public static function processFilePond(
    array &$element,
    FormStateInterface $form_state,
    array &$complete_form,
  ): array {
    // Check permission - show message instead of upload form if no access.
    if (!\Drupal::currentUser()->hasPermission('filepond upload files')) {
      $element['no_access_message'] = [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [new TranslatableMarkup("You don't have permission to upload files.")],
        ],
      ];
      // Mark as no access so pre_render skips JS attachment.
      $element['#filepond_no_access'] = TRUE;
      return $element;
    }

    // Get module config defaults.
    $moduleConfig = \Drupal::config('filepond.settings');
    $defaults = $moduleConfig->get('defaults') ?? [];

    // Set defaults for root properties (server-side).
    if (empty($element['#extensions'])) {
      $element['#extensions'] = $defaults['extensions'] ?? FilePondUploader::DEFAULT_EXTENSIONS;
    }
    if (empty($element['#max_filesize'])) {
      $element['#max_filesize'] = $defaults['max_filesize'] ?? Environment::getUploadMaxSize();
    }
    // Note: Don't use empty() - 0 means unlimited, which is valid.
    if (!isset($element['#max_files'])) {
      $element['#max_files'] = $defaults['max_files'] ?? NULL;
    }
    if (empty($element['#upload_location'])) {
      $element['#upload_location'] = $defaults['upload_location'] ?? 'public://filepond-uploads';
    }
    if (!isset($element['#columns'])) {
      $element['#columns'] = $defaults['columns'] ?? 5;
    }
    if (empty($element['#max_width'])) {
      $element['#max_width'] = $defaults['max_width'] ?? NULL;
    }

    // NOTE: Token replacement for #upload_location is done at upload time
    // in the controller, not here. This is important because State API stores
    // config once per form element (not per user), so tokens like
    // [current-user:uid] must be replaced when the upload actually happens.
    // Ensure #config is an array.
    if (!isset($element['#config']) || !is_array($element['#config'])) {
      $element['#config'] = [];
    }

    // Remove any server-side properties from #config if mistakenly set there.
    // Root properties always take precedence.
    unset(
      $element['#config']['extensions'],
      $element['#config']['maxFilesize'],
      $element['#config']['maxFiles'],
      $element['#config']['uploadLocation'],
      $element['#config']['acceptedFileTypes']
    );

    // Set defaults for #config (FilePond UI options).
    $element['#config'] += [
      'labelIdle' => $defaults['upload_prompt'] ?? (string) new TranslatableMarkup('Drag & drop files or <span class="filepond--label-action">Browse</span>'),
      'allowReorder' => FALSE,
      'maxParallelUploads' => $defaults['max_parallel_uploads'] ?? 3,
      'styleItemPanelAspectRatio' => $defaults['item_panel_aspect_ratio'] ?? '1:1',
      'previewFitMode' => $defaults['preview_fit_mode'] ?? 'contain',
      'previewImageStyle' => NULL,
      'chunkSize' => ($defaults['chunk_size'] ?? 5) * 1024 * 1024,
      'chunkUploads' => TRUE,
      'imagePreviewMaxFileSize' => '7MB',
      'dropValidation' => TRUE,
      'labelInvalidField' => (string) new TranslatableMarkup('Invalid file'),
      'labelProcessing' => (string) new TranslatableMarkup('Processing...'),
      'labelFileProcessingError' => (string) new TranslatableMarkup('Upload error'),
    ];

    // Check if URLs are already provided (field widget, Entity Browser, or
    // custom forms with their own routes like portfolio_uploader).
    // When URLs are in #config, skip state storage - routes derive from field
    // or the custom form handles its own upload endpoints.
    if (empty($element['#config']['processUrl'])) {
      // Get form ID from the form object.
      $form_object = $form_state->getFormObject();
      $form_id = $form_object->getFormId();

      // Use element name as identifier (from #name or first parent).
      $element_name = $element['#name'] ?? ($element['#parents'][0] ?? 'filepond');

      // Build context: merge custom context with auto-generated element info.
      $context = array_merge([
        'element_name' => $element_name,
        'element_parents' => $element['#parents'] ?? [],
        'field_name' => $element['#field_name'] ?? NULL,
      ], $element['#context'] ?? []);

      // Store element config in SharedTempStore via the settings resolver.
      // Keyed by form_id:element_name:hash. The hash ensures config changes
      // (including form_alter modifications) are respected.
      $form_config = [
        'extensions' => $element['#extensions'],
        'max_filesize' => $element['#max_filesize'],
        'upload_location' => $element['#upload_location'],
        'chunk_size' => $element['#config']['chunkSize'],
        // Context for event subscribers to identify the upload source.
        'context' => $context,
      ];
      $config_hash = \Drupal::service('filepond.settings_resolver')->storeFormConfig($form_id, $element_name, $form_config);

      // Generate upload URLs with form_id, element_name, and config_hash.
      $upload_url = Url::fromRoute('filepond.form_process', [
        'form_id' => $form_id,
        'element_name' => $element_name,
        'config_hash' => $config_hash,
      ])->toString();
      $revert_url = Url::fromRoute('filepond.form_revert', [
        'form_id' => $form_id,
        'element_name' => $element_name,
        'config_hash' => $config_hash,
      ])->toString();
      $patch_url = Url::fromRoute('filepond.form_patch', [
        'form_id' => $form_id,
        'element_name' => $element_name,
        'config_hash' => $config_hash,
        'transferId' => '__TRANSFER_ID__',
      ])->toString();

      // JS will append the actual transferId to this base URL.
      $patch_base_url = str_replace('/__TRANSFER_ID__', '', $patch_url);

      // Store URLs in #config for JS.
      $element['#config']['processUrl'] = $upload_url;
      $element['#config']['revertUrl'] = $revert_url;
      $element['#config']['patchUrl'] = $patch_base_url;
    }

    // Generate a unique key for JS to link input, hidden field, and settings.
    // Use #parents for consistency - always set during processing.
    // Don't rely on #id as it may not be set until render phase.
    $filepond_key = Html::getUniqueId(
      'filepond-' . implode('-', $element['#parents'] ?? ['element'])
    );

    // Build child element parents - extend current parents with child key.
    // Explicitly set #parents to ensure correct naming in Drupal 11+ where
    // child elements added during #process may not automatically inherit.
    $base_parents = $element['#parents'] ?? [];

    // Hidden field stores semicolon-separated file IDs.
    // Add data-filepond-fids attribute so JS can reliably find it.
    $element['fids'] = [
      '#type' => 'hidden',
      '#default_value' => '',
      '#parents' => array_merge($base_parents, ['fids']),
      '#attributes' => [
        'data-filepond-fids' => $filepond_key,
      ],
    ];

    // File input element - FilePond replaces this with its UI.
    // Don't add 'multiple' - FilePond inherits it and ignores JS options.
    $element['input'] = [
      '#type' => 'html_tag',
      '#tag' => 'input',
      '#parents' => array_merge($base_parents, ['input']),
      '#attributes' => [
        'type' => 'file',
        'class' => ['filepond--input'],
        'data-filepond-id' => $filepond_key,
      ],
    ];

    // Store the key for preRenderFilePond to use with drupalSettings.
    $element['#filepond_key'] = $filepond_key;

    // Add key to element wrapper for easy lookup after FilePond replaces input.
    $element['#attributes']['data-filepond-key'] = $filepond_key;

    return $element;
  }

  /**
   * Pre-renders the FilePond element.
   *
   * Builds JS config from root properties and #config, then passes to
   * drupalSettings. Derives acceptedFileTypes from #extensions.
   *
   * @param array $element
   *   The element to pre-render.
   *
   * @return array
   *   The pre-rendered element.
   */
  public static function preRenderFilePond(array $element): array {
    // Skip JS attachment if user doesn't have permission.
    if (!empty($element['#filepond_no_access'])) {
      return $element;
    }

    // Start with #config (UI options).
    $config = $element['#config'];

    // Add server-side properties to JS config (derived from root properties).
    // Convert max_filesize to FilePond's format (e.g., "20MB").
    if (!empty($element['#max_filesize'])) {
      $bytes = Bytes::toNumber($element['#max_filesize']);
      $mb = round($bytes / pow(Bytes::KILOBYTE, 2), 2);
      $config['maxFileSize'] = $mb ? "{$mb}MB" : NULL;
    }

    // Pass maxFiles for JS limit checking.
    $config['maxFiles'] = $element['#max_files'];

    // Derive acceptedFileTypes from #extensions.
    // This cannot be overridden to ensure server and client validation match.
    $extensions = $element['#extensions'] ?? FilePondUploader::DEFAULT_EXTENSIONS;
    $config['acceptedFileTypes'] = array_values(FilePondUploader::extensionsToMimeTypes($extensions));

    // Pass extensions to JS (for display in upload requirements).
    $config['extensions'] = $extensions;

    // Build existing files array for FilePond initialization.
    // Uses File Poster plugin to display thumbnails via CDN/image style URLs.
    $config['files'] = self::buildExistingFilesArray($element);

    // Attach library and settings.
    // Use #filepond_key (set during processing) to key the settings,
    // matching the data-filepond-id attribute on the input element.
    $element['#attached']['library'][] = 'filepond/filepond.element';
    $element['#attached']['drupalSettings']['filepond']['instances'][$element['#filepond_key']] = $config;

    static::setAttributes($element, ['filepond-element']);

    // Add CSS variables for grid layout.
    $style = static::buildGridStyleAttribute(
      (int) ($element['#columns'] ?? 5),
      $element['#max_width'] ?? NULL,
      $element['#attributes']['style'] ?? ''
    );
    if ($style) {
      $element['#attributes']['style'] = $style;
    }

    return $element;
  }

  /**
   * Builds the existing files array for FilePond initialization.
   *
   * Uses FilePond's File Poster plugin to display thumbnails. This allows
   * existing files (including those on S3/CDN) to be displayed without
   * downloading them through Drupal.
   *
   * @param array $element
   *   The form element.
   *
   * @return array
   *   Array of file objects for FilePond's files property.
   *
   * @see https://pqina.nl/filepond/docs/api/instance/properties/#files
   */
  protected static function buildExistingFilesArray(array $element): array {
    $existing_files = [];
    $default_value = $element['#default_value'] ?? [];

    if (empty($default_value)) {
      return [];
    }

    // Normalize to array of file IDs.
    $fids = is_array($default_value) ? $default_value : [$default_value];
    $fids = array_filter($fids);

    if (empty($fids)) {
      return [];
    }

    // Load file entities.
    /** @var \Drupal\file\FileStorageInterface $file_storage */
    $file_storage = \Drupal::entityTypeManager()->getStorage('file');
    $files = $file_storage->loadMultiple($fids);

    // Get image style if set (from #config).
    $image_style_name = $element['#config']['previewImageStyle'] ?? NULL;
    $image_style = $image_style_name ? ImageStyle::load($image_style_name) : NULL;

    /** @var \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator */
    $file_url_generator = \Drupal::service('file_url_generator');

    // Maintain order from $fids array.
    foreach ($fids as $fid) {
      if (!isset($files[$fid])) {
        continue;
      }

      /** @var \Drupal\file\FileInterface $file */
      $file = $files[$fid];
      $uri = $file->getFileUri();

      // Build metadata - only set poster for image files.
      $metadata = [];
      $mime_type = $file->getMimeType();
      if (str_starts_with($mime_type, 'image/')) {
        // Generate poster URL - use image style if available.
        if ($image_style) {
          $metadata['poster'] = $image_style->buildUrl($uri);
        }
        else {
          $metadata['poster'] = $file_url_generator->generateAbsoluteString($uri);
        }
      }

      // Build FilePond file object.
      // Using 'local' type means FilePond won't try to upload it.
      // The source is the file ID which we'll use for the hidden field.
      $existing_files[] = [
        'source' => (string) $file->id(),
        'options' => [
          'type' => 'local',
          'file' => [
            'name' => $file->getFilename(),
            'size' => (int) $file->getSize(),
            'type' => $mime_type,
          ],
          'metadata' => $metadata,
        ],
      ];
    }

    return $existing_files;
  }

  /**
   * {@inheritdoc}
   */
  public static function valueCallback(
    &$element,
    $input,
    FormStateInterface $form_state,
  ): array {
    $return = ['fids' => []];

    if ($input !== FALSE) {
      // Get user input from the hidden field.
      $user_input = NestedArray::getValue(
        $form_state->getUserInput(),
        array_merge($element['#parents'], ['fids'])
      );

      if (!empty($user_input)) {
        // File IDs stored as semicolon-separated values.
        $fids = array_filter(array_map('intval', explode(';', $user_input)));

        // Validate files following core's ManagedFile pattern.
        if (!empty($fids)) {
          /** @var \Drupal\file\FileStorageInterface $file_storage */
          $file_storage = \Drupal::entityTypeManager()->getStorage('file');
          $files = $file_storage->loadMultiple($fids);
          $current_uid = \Drupal::currentUser()->id();

          foreach ($files as $file) {
            // Temporary files must be owned by current user. This prevents
            // users from claiming another user's just-uploaded files.
            // Permanent files are allowed - they're already attached to
            // entities and form access controls who can edit them.
            if ($file->isTemporary() && $file->getOwnerId() != $current_uid) {
              continue;
            }
            $return['fids'][] = (int) $file->id();
          }
        }
      }

      $form_state->setValueForElement($element, $return);
    }

    return $return;
  }

  /**
   * Element validation callback for aspect ratio fields.
   *
   * FilePond's styleItemPanelAspectRatio expects ratio format like "16:9".
   * Use as #element_validate callback on textfields.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function validateAspectRatio(array &$element, FormStateInterface $form_state): void {
    $value = $form_state->getValue($element['#parents']);
    if (empty($value)) {
      return;
    }

    if (!self::isValidAspectRatioFormat($value)) {
      $form_state->setError($element, new TranslatableMarkup('Aspect ratio must be in format "W:H" (e.g., "16:9", "4:3", "1:1").'));
    }
  }

  /**
   * Checks if a string is valid aspect ratio format.
   *
   * @param string $ratio
   *   The ratio string to validate.
   *
   * @return bool
   *   TRUE if valid ratio format like "16:9", "4:3", "1:1".
   */
  public static function isValidAspectRatioFormat(string $ratio): bool {
    if (!str_contains($ratio, ':')) {
      return FALSE;
    }
    $parts = explode(':', $ratio, 2);
    if (count($parts) !== 2) {
      return FALSE;
    }
    $width = trim($parts[0]);
    $height = trim($parts[1]);
    return is_numeric($width) && is_numeric($height) && $width > 0 && $height > 0;
  }

  /**
   * Element validate callback for upload location field.
   *
   * Validates that the value is a valid stream wrapper URI.
   * Use as #element_validate callback on textfields.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function validateUploadLocation(array &$element, FormStateInterface $form_state): void {
    $value = $form_state->getValue($element['#parents']);
    if (empty($value)) {
      return;
    }

    $stream_wrapper_manager = \Drupal::service('stream_wrapper_manager');
    if (!$stream_wrapper_manager->isValidUri($value)) {
      $form_state->setError($element, new TranslatableMarkup('Upload location must be a valid URI (e.g., public://uploads, private://files).'));
    }
  }

  /**
   * Element validate callback for file extensions field.
   *
   * Validates that extensions are space-separated alphanumeric values.
   * Use as #element_validate callback on textfields.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function validateExtensions(array &$element, FormStateInterface $form_state): void {
    $value = $form_state->getValue($element['#parents']);
    if (empty($value)) {
      return;
    }

    $extensions = explode(' ', $value);
    foreach ($extensions as $extension) {
      if (preg_match('%^\w+$%', $extension) !== 1) {
        $form_state->setError($element, new TranslatableMarkup('Invalid extension list format. Use space-separated extensions (e.g., "jpg png gif").'));
        break;
      }
    }
  }

  /**
   * Builds CSS style string for grid layout.
   *
   * Generates CSS custom properties for FilePond grid layout based on
   * columns and max_width settings.
   *
   * @param int $columns
   *   Number of columns (0 to skip width calculation).
   * @param string|null $max_width
   *   Optional max-width value (e.g., '200px' or numeric for pixels).
   * @param string $existing_style
   *   Existing style attribute value to append to.
   *
   * @return string|null
   *   The style attribute value, or NULL if no CSS vars needed.
   */
  public static function buildGridStyleAttribute(int $columns, ?string $max_width = NULL, string $existing_style = ''): ?string {
    $css_vars = [];

    if ($columns > 0) {
      $width_percent = round(100 / $columns, 2);
      $css_vars[] = '--filepond-item-width: ' . $width_percent . '%';
    }

    if (!empty($max_width)) {
      if (is_numeric($max_width)) {
        $max_width .= 'px';
      }
      $css_vars[] = '--filepond-item-max-width: ' . $max_width;
    }

    if (empty($css_vars)) {
      return $existing_style ?: NULL;
    }

    return trim($existing_style . ' ' . implode('; ', $css_vars) . ';');
  }

}
