<?php

declare(strict_types=1);

namespace Drupal\filepond\Event;

use Drupal\Component\EventDispatcher\Event;

/**
 * Event dispatched before a FilePond upload is moved to its final destination.
 *
 * This event fires while the uploaded file is still on the local filesystem,
 * allowing subscribers to modify the file before it's moved to remote storage
 * (e.g., S3). This is the ideal point for operations like:
 * - EXIF orientation correction
 * - Virus scanning
 * - Image optimization/compression
 * - Watermarking
 * - Metadata stripping.
 *
 * After this event completes, the file will be moved to its final destination
 * via core's FileUploadHandler.
 *
 * Note: This event addresses a gap in Drupal core where hook_file_presave()
 * fires AFTER the file is moved, making it expensive for remote storage.
 * A similar event in core's FileUploadHandler would benefit all upload methods.
 *
 * @see https://www.drupal.org/project/drupal/issues/2664632
 * @see https://www.drupal.org/project/drupal/issues/2986669
 */
class FilePondUploadPreMoveEvent extends Event {

  /**
   * Event name.
   */
  public const EVENT_NAME = 'filepond.upload_pre_move';

  /**
   * The local file path (writable).
   */
  protected string $filePath;

  /**
   * The file's MIME type.
   */
  protected string $mimeType;

  /**
   * The original filename from the client.
   */
  protected string $originalFilename;

  /**
   * The destination URI where the file will be moved.
   */
  protected string $destinationUri;

  /**
   * Optional context data from the upload request.
   *
   * @var array<string, mixed>
   */
  protected array $context;

  /**
   * Whether the destination is remote storage (e.g., S3).
   *
   * @var bool
   */
  protected bool $isRemoteDestination;

  /**
   * Constructs a FilePondUploadPreMoveEvent.
   *
   * @param string $file_path
   *   The local filesystem path to the uploaded file. Subscribers can
   *   modify the file at this path directly.
   * @param string $mime_type
   *   The file's MIME type (e.g., 'image/jpeg').
   * @param string $original_filename
   *   The original filename from the client.
   * @param string $destination_uri
   *   The destination URI where the file will be moved.
   * @param array $context
   *   Optional context data (e.g., field name, entity type, bundle).
   * @param bool $is_remote_destination
   *   Whether the destination is remote storage (e.g., S3).
   */
  public function __construct(
    string $file_path,
    string $mime_type,
    string $original_filename,
    string $destination_uri,
    array $context = [],
    bool $is_remote_destination = FALSE,
  ) {
    $this->filePath = $file_path;
    $this->mimeType = $mime_type;
    $this->originalFilename = $original_filename;
    $this->destinationUri = $destination_uri;
    $this->context = $context;
    $this->isRemoteDestination = $is_remote_destination;
  }

  /**
   * Gets the local file path.
   *
   * The file at this path can be modified directly by subscribers.
   *
   * @return string
   *   The local filesystem path.
   */
  public function getFilePath(): string {
    return $this->filePath;
  }

  /**
   * Gets the file's MIME type.
   *
   * @return string
   *   The MIME type (e.g., 'image/jpeg').
   */
  public function getMimeType(): string {
    return $this->mimeType;
  }

  /**
   * Gets the original filename.
   *
   * @return string
   *   The original filename from the client.
   */
  public function getOriginalFilename(): string {
    return $this->originalFilename;
  }

  /**
   * Gets the destination URI.
   *
   * This is where the file will be moved after the event completes.
   *
   * @return string
   *   The destination URI (e.g., 's3://bucket/path/file.jpg').
   */
  public function getDestinationUri(): string {
    return $this->destinationUri;
  }

  /**
   * Sets the destination URI.
   *
   * Allows subscribers to change the filename or path where the file
   * will be stored. Useful for filename obfuscation or reorganization.
   *
   * @param string $uri
   *   The new destination URI.
   *
   * @return $this
   */
  public function setDestinationUri(string $uri): self {
    $this->destinationUri = $uri;
    return $this;
  }

  /**
   * Sets an override filename (alias for setOverrideFilename).
   *
   * @param string $filename
   *   The new filename (with extension).
   *
   * @return $this
   */
  public function setDestinationFilename(string $filename): self {
    return $this->setOverrideFilename($filename);
  }

  /**
   * The overridden filename (if set by a subscriber).
   */
  protected ?string $overrideFilename = NULL;

  /**
   * Sets an overridden filename to use instead of the original.
   *
   * This changes the filename used when storing the file, while keeping
   * the original filename available via getOriginalFilename().
   *
   * @param string $filename
   *   The new filename (with extension).
   *
   * @return $this
   */
  public function setOverrideFilename(string $filename): self {
    $this->overrideFilename = $filename;
    return $this;
  }

  /**
   * Gets the overridden filename, if set.
   *
   * @return string|null
   *   The overridden filename, or NULL if not set.
   */
  public function getOverrideFilename(): ?string {
    return $this->overrideFilename;
  }

  /**
   * Gets the filename to use for storage.
   *
   * Returns the overridden filename if set, otherwise the original filename.
   *
   * @return string
   *   The filename to use.
   */
  public function getFilename(): string {
    return $this->overrideFilename ?? $this->originalFilename;
  }

  /**
   * Gets the upload context.
   *
   * @return array<string, mixed>
   *   The context data.
   */
  public function getContext(): array {
    return $this->context;
  }

  /**
   * Gets a specific context value.
   *
   * @param string $key
   *   The context key.
   * @param mixed $default
   *   Default value if key doesn't exist.
   *
   * @return mixed
   *   The context value.
   */
  public function getContextValue(string $key, mixed $default = NULL): mixed {
    return $this->context[$key] ?? $default;
  }

  /**
   * Checks if the file is an image.
   *
   * @return bool
   *   TRUE if the file is an image based on MIME type.
   */
  public function isImage(): bool {
    return str_starts_with($this->mimeType, 'image/');
  }

  /**
   * Checks if the destination is remote storage.
   *
   * @return bool
   *   TRUE if destination is not a local filesystem (e.g., S3).
   */
  public function isRemoteDestination(): bool {
    return $this->isRemoteDestination;
  }

}
