<?php

declare(strict_types=1);

namespace Drupal\filepond;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\filepond\Element\FilePond as FilePondElement;

/**
 * Provides shared form building for FilePond configuration.
 *
 * This trait provides consistent configuration form elements for FilePond
 * uploaders across different contexts (Entity Browser widget, Views area, etc).
 *
 * Two usage patterns:
 * 1. Use buildFilePondConfigForm() for full form with fieldsets (EB, Views)
 * 2. Use individual get*Fields() methods for flat fields (field widgets)
 */
trait FilePondConfigFormTrait {

  /**
   * The module handler (lazy-loaded).
   *
   * Prefixed to avoid collision with HandlerBase::$moduleHandler in Views.
   */
  protected ?ModuleHandlerInterface $traitModuleHandler = NULL;

  /**
   * Gets the module handler.
   */
  protected function getModuleHandler(): ModuleHandlerInterface {
    if (!$this->traitModuleHandler) {
      $this->traitModuleHandler = \Drupal::moduleHandler();
    }
    return $this->traitModuleHandler;
  }

  /**
   * Resolves max files setting against field cardinality.
   *
   * The widget's max_files setting can further restrict uploads, but
   * field cardinality is always the upper bound.
   *
   * @param int $max_files_setting
   *   The widget's max_files setting (0 = unlimited).
   * @param int $cardinality
   *   The field cardinality (-1 = unlimited, >0 = limit).
   *
   * @return int
   *   The resolved max files value (0 = unlimited).
   */
  protected function resolveMaxFiles(int $max_files_setting, int $cardinality): int {
    if ($cardinality > 0) {
      // Limited cardinality is the upper bound.
      return ($max_files_setting > 0) ? min($max_files_setting, $cardinality) : $cardinality;
    }
    // Unlimited cardinality (-1) - use widget setting.
    return $max_files_setting;
  }

  /**
   * Gets default FilePond configuration values for widgets.
   *
   * Returns empty values so widgets inherit from filepond.settings at runtime.
   * Only explicit overrides are stored in widget config.
   *
   * @return array
   *   Default configuration array with empty values.
   */
  protected function getFilePondDefaults(): array {
    return [
      'media_type' => '',
      'inherit_settings' => TRUE,
      'upload_location' => '',
      'max_filesize' => '',
      'extensions' => '',
      'upload_prompt' => '',
      'max_files' => 0,
      'chunk_size' => NULL,
      'max_parallel_uploads' => NULL,
      'item_panel_aspect_ratio' => '',
      'preview_fit_mode' => '',
      'columns' => NULL,
      'max_width' => '',
      'auto_select' => FALSE,
    ];
  }

  /**
   * Gets the display form fields.
   *
   * @param array $config
   *   Current configuration values.
   *
   * @return array
   *   Flat array of display field definitions.
   */
  protected function getDisplayFields(array $config): array {
    return [
      'item_panel_aspect_ratio' => [
        '#type' => 'textfield',
        '#title' => new TranslatableMarkup('Panel aspect ratio'),
        '#default_value' => $config['item_panel_aspect_ratio'] ?? NULL,
        '#description' => new TranslatableMarkup('Aspect ratio for file panels (e.g., "16:9", "4:3", "1:1").'),
        '#size' => 10,
        '#element_validate' => [[FilePondElement::class, 'validateAspectRatio']],
      ],
      'preview_fit_mode' => [
        '#type' => 'select',
        '#title' => new TranslatableMarkup('Preview fit mode'),
        '#options' => [
          'contain' => new TranslatableMarkup('Contain (letterbox)'),
          'cover' => new TranslatableMarkup('Cover (crop to fill)'),
        ],
        '#default_value' => $config['preview_fit_mode'] ?? NULL,
        '#description' => new TranslatableMarkup('How image previews fit within the panel.'),
      ],
      'columns' => [
        '#type' => 'number',
        '#title' => new TranslatableMarkup('Columns'),
        '#default_value' => $config['columns'] ?? NULL,
        '#description' => new TranslatableMarkup('Number of columns for the upload grid.'),
        '#min' => 1,
        '#max' => 12,
      ],
      'max_width' => [
        '#type' => 'textfield',
        '#title' => new TranslatableMarkup('Max width'),
        '#default_value' => $config['max_width'] ?? NULL,
        '#description' => new TranslatableMarkup('Optional max-width for items (e.g., "200px").'),
        '#size' => 10,
      ],
    ];
  }

  /**
   * Gets the uploader form fields.
   *
   * @param array $config
   *   Current configuration values.
   *
   * @return array
   *   Flat array of uploader field definitions.
   */
  protected function getUploaderFields(array $config): array {
    return [
      'upload_prompt' => [
        '#type' => 'textfield',
        '#title' => new TranslatableMarkup('Upload prompt'),
        '#description' => new TranslatableMarkup('Text shown in the drop area. Use <code>@code</code> for clickable styling.', [
          '@code' => Markup::create('&lt;span class="filepond--label-action"&gt;Click here&lt;/span&gt;'),
        ]),
        '#default_value' => $config['upload_prompt'] ?? NULL,
      ],
      'max_files' => [
        '#type' => 'number',
        '#title' => new TranslatableMarkup('Maximum files'),
        '#min' => 0,
        '#default_value' => $config['max_files'] ?? NULL,
        '#description' => new TranslatableMarkup('The total number of files allowed in each upload batch. Zero means unlimited.'),
      ],
      'chunk_size' => [
        '#type' => 'number',
        '#title' => new TranslatableMarkup('Chunk size'),
        '#min' => 1,
        '#max' => 50,
        '#field_suffix' => new TranslatableMarkup('MB'),
        '#default_value' => $config['chunk_size'] ?? NULL,
        '#description' => new TranslatableMarkup('Size of each upload chunk for large files.'),
      ],
      'max_parallel_uploads' => [
        '#type' => 'number',
        '#title' => new TranslatableMarkup('Max parallel uploads'),
        '#min' => 1,
        '#max' => 10,
        '#default_value' => $config['max_parallel_uploads'] ?? NULL,
        '#description' => new TranslatableMarkup('Number of files to upload simultaneously.'),
      ],
    ];
  }

  /**
   * Gets the constraints form fields.
   *
   * @param array $config
   *   Current configuration values.
   * @param array $options
   *   Options:
   *   - show_extensions: Whether to include extensions field (default TRUE).
   *   - override_states: States array for conditional visibility.
   *
   * @return array
   *   Flat array of constraints field definitions.
   */
  protected function getConstraintsFields(array $config, array $options = []): array {
    $show_extensions = $options['show_extensions'] ?? TRUE;
    $override_states = $options['override_states'] ?? [];

    $fields = [];

    if ($show_extensions) {
      $fields['extensions'] = [
        '#type' => 'textfield',
        '#title' => new TranslatableMarkup('Allowed file extensions'),
        '#description' => new TranslatableMarkup('Space-separated list of allowed extensions.'),
        '#default_value' => $config['extensions'] ?? NULL,
        '#element_validate' => [[FilePondElement::class, 'validateExtensions']],
      ];
    }

    $fields['max_filesize'] = [
      '#type' => 'textfield',
      '#title' => new TranslatableMarkup('Maximum file size'),
      '#description' => new TranslatableMarkup('e.g., "10M", "500K".'),
      '#default_value' => $config['max_filesize'] ?? NULL,
      '#size' => 10,
    ];
    if ($override_states) {
      $fields['max_filesize']['#states'] = $override_states;
    }

    $fields['upload_location'] = [
      '#type' => 'textfield',
      '#title' => new TranslatableMarkup('Upload location'),
      '#default_value' => $config['upload_location'] ?? NULL,
      '#description' => new TranslatableMarkup('Destination folder for uploaded files. Supports tokens.'),
      '#element_validate' => [[FilePondElement::class, 'validateUploadLocation']],
    ];
    if ($override_states) {
      $fields['upload_location']['#states'] = $override_states;
    }

    // Show token browser if token module is enabled.
    if ($this->getModuleHandler()->moduleExists('token')) {
      $fields['token_help'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['current-user', 'date'],
      ];
      if ($override_states) {
        $fields['token_help']['#states'] = $override_states;
      }
    }

    return $fields;
  }

  /**
   * Gets the create (media type) form fields.
   *
   * @param array $config
   *   Current configuration values.
   * @param array $options
   *   Options:
   *   - media_type_options: Array of media type options for select.
   *   - media_type_required: Whether media type is required (default FALSE).
   *
   * @return array
   *   Flat array of create field definitions.
   */
  protected function getCreateFields(array $config, array $options = []): array {
    $media_type_options = $options['media_type_options'] ?? [];

    return [
      'media_type' => [
        '#type' => 'select',
        '#title' => new TranslatableMarkup('Media type'),
        '#required' => $options['media_type_required'] ?? FALSE,
        '#default_value' => $config['media_type'] ?? NULL,
        '#options' => ['' => new TranslatableMarkup('- Select -')] + $media_type_options,
        '#description' => new TranslatableMarkup('The type of media entity to create from uploaded files. File extensions are always inherited from the media type.'),
      ],
      'inherit_settings' => [
        '#type' => 'checkbox',
        '#title' => new TranslatableMarkup('Inherit upload location and file size from media type'),
        '#description' => new TranslatableMarkup('When enabled, these settings are taken from the media type source field. Uncheck to specify custom values below.'),
        '#default_value' => $config['inherit_settings'] ?? NULL,
        '#attributes' => ['class' => ['filepond-inherit-settings']],
      ],
    ];
  }

  /**
   * Builds the FilePond configuration form elements.
   *
   * Uses get*Fields() methods internally, wrapping in fieldsets.
   * For flat fields without fieldsets, call get*Fields() directly.
   *
   * @param array $form
   *   The form array to add elements to.
   * @param array $config
   *   Current configuration values.
   * @param array $options
   *   Options for form building:
   *   - media_type_options: Array of media type options for select.
   *   - media_type_required: Whether media type is required (default FALSE).
   *   - show_auto_select: Whether to show auto-select option (default FALSE).
   *   - show_media_type: Whether to show media type/inherit fields
   *     (default TRUE).
   *   - show_extensions: Whether to show extensions field (default TRUE).
   *     Set to FALSE in widget contexts where extensions come from media type.
   *   - include_sections: Array of section names to include. If empty, all
   *     sections are built. Valid: 'create', 'constraints', 'uploader',
   *     'display'.
   *   - use_details: Use collapsible details elements instead of fieldsets
   *     (default FALSE). Sections start collapsed.
   *
   * @return array
   *   The form array with FilePond elements added.
   */
  protected function buildFilePondConfigForm(array $form, array $config, array $options = []): array {
    $show_media_type = $options['show_media_type'] ?? TRUE;
    $show_extensions = $options['show_extensions'] ?? TRUE;
    $include_sections = $options['include_sections'] ?? [];
    $use_details = $options['use_details'] ?? FALSE;

    // Helper to check if a section should be included.
    $include = fn(string $section) => empty($include_sections) || in_array($section, $include_sections, TRUE);

    // Use details (collapsible) or fieldset based on option.
    $wrapper_type = $use_details ? 'details' : 'fieldset';

    // Media type fieldset - only for widget contexts, not global settings.
    if ($show_media_type && $include('create')) {
      $form['create'] = [
        '#type' => $wrapper_type,
        '#title' => new TranslatableMarkup('Create media'),
        '#description' => new TranslatableMarkup('Controls how media entities are created from the uploaded files.'),
        '#description_display' => 'before',
      ] + $this->getCreateFields($config, $options);
    }

    // States to show constraint fields only when inherit_settings is off.
    $override_states = $show_media_type && $include('create') ? [
      'visible' => [
        '.filepond-inherit-settings' => ['checked' => FALSE],
      ],
    ] : [];

    // File constraints fieldset.
    if ($include('constraints')) {
      $form['constraints'] = [
        '#type' => $wrapper_type,
        '#title' => new TranslatableMarkup('File constraints'),
        '#description' => new TranslatableMarkup('Limits on uploaded files.'),
        '#description_display' => 'before',
      ] + $this->getConstraintsFields($config, [
        'show_extensions' => $show_extensions,
        'override_states' => $override_states,
      ]);

      if ($override_states && !$show_extensions) {
        $form['constraints']['#states'] = $override_states;
      }
    }

    // Uploader settings fieldset.
    if ($include('uploader')) {
      $form['uploader'] = [
        '#type' => $wrapper_type,
        '#title' => new TranslatableMarkup('Uploader'),
        '#description' => new TranslatableMarkup('General upload behavior settings.'),
        '#description_display' => 'before',
      ] + $this->getUploaderFields($config);
    }

    // Display options fieldset.
    if ($include('display')) {
      $form['display'] = [
        '#type' => $wrapper_type,
        '#title' => new TranslatableMarkup('Display options'),
        '#description' => new TranslatableMarkup('Controls how individual file items are displayed in the uploader.'),
        '#description_display' => 'before',
      ] + $this->getDisplayFields($config);
    }

    // Auto-select is primarily useful in Entity Browser/Views contexts.
    $show_auto_select = $options['show_auto_select'] ?? FALSE;
    if ($show_auto_select) {
      $form['auto_select'] = [
        '#type' => 'checkbox',
        '#title' => new TranslatableMarkup('Auto-select uploaded items'),
        '#description' => new TranslatableMarkup('When used in an Entity Browser, automatically select newly uploaded media items.'),
        '#default_value' => $config['auto_select'] ?? NULL,
      ];
    }

    return $form;
  }

  /**
   * Processes FilePond config values after form submission.
   *
   * @param array $values
   *   The submitted values.
   *
   * @return array
   *   The processed values with nested display values flattened.
   */
  protected function processFilePondConfigValues(array $values): array {
    // Append 'M' to max_filesize if it's just a number.
    if (!empty($values['max_filesize']) && is_numeric($values['max_filesize'])) {
      $values['max_filesize'] = $values['max_filesize'] . 'M';
    }

    // Trim trailing slashes from upload_location.
    if (!empty($values['upload_location'])) {
      $values['upload_location'] = rtrim($values['upload_location'], '/');
    }

    // Flatten fieldset values to root level.
    $groups = ['create', 'constraints', 'uploader', 'display'];
    foreach ($groups as $group) {
      if (isset($values[$group]) && is_array($values[$group])) {
        foreach ($values[$group] as $key => $value) {
          $values[$key] = $value;
        }
        unset($values[$group]);
      }
    }

    // Append 'px' to max_width if it's just a number.
    if (!empty($values['max_width']) && is_numeric($values['max_width'])) {
      $values['max_width'] = $values['max_width'] . 'px';
    }

    // Remove empty values so they inherit from filepond.settings at runtime.
    // Keep 0 (unlimited max_files) and FALSE (checkbox values).
    return array_filter($values, fn($v) => $v !== '' && $v !== NULL);
  }

}
