<?php

declare(strict_types=1);

namespace Drupal\filepond;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StreamWrapper\LocalStream;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\file\FileUsage\FileUsageInterface;
use Drupal\file\Upload\FileUploadHandler;
use Drupal\file\Validation\FileValidatorInterface;
use Drupal\filepond\Event\FilePondUploadCompleteEvent;
use Drupal\filepond\Event\FilePondUploadPreMoveEvent;
use Drupal\filepond\Upload\TempUploadedFile;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\Mime\MimeTypeGuesserInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Handles FilePond upload operations.
 *
 * This service provides the core file upload handling for FilePond:
 * - Chunked upload support (process/patch/revert protocol)
 * - File validation (extension, size, MIME type, image content)
 * - Transfer ID management
 * - File entity creation.
 *
 * Media entities are NOT created by this service. Subscribe to the
 * FilePondUploadCompleteEvent to create media entities or perform
 * other post-upload processing.
 *
 * @see \Drupal\filepond\Event\FilePondUploadCompleteEvent
 */
class FilePondUploadHandler {

  /**
   * Default temporary folder URI for uploads.
   */
  public const DEFAULT_TEMP_LOCATION = 'temporary://filepond';

  /**
   * Default allowed MIME types.
   */
  public const DEFAULT_ALLOWED_MIME_TYPES = [
    'image/jpeg',
    'image/jpg',
    'image/pjpeg',
    'image/png',
  ];

  /**
   * Default allowed file extensions.
   */
  public const DEFAULT_ALLOWED_EXTENSIONS = ['jpg', 'jpeg', 'png'];

  /**
   * The file system service.
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The stream wrapper manager.
   */
  protected StreamWrapperManagerInterface $streamWrapperManager;

  /**
   * The time service.
   */
  protected TimeInterface $time;

  /**
   * The current user.
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The event dispatcher.
   */
  protected EventDispatcherInterface $eventDispatcher;

  /**
   * The MIME type guesser.
   */
  protected MimeTypeGuesserInterface $mimeTypeGuesser;

  /**
   * The file validator.
   */
  protected FileValidatorInterface $fileValidator;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Core's file upload handler.
   */
  protected FileUploadHandler $coreUploadHandler;

  /**
   * The file usage service.
   */
  protected FileUsageInterface $fileUsage;

  /**
   * Cache backend for tracking completed transfers.
   */
  protected CacheBackendInterface $cache;

  /**
   * Constructs a FilePondUploadHandler.
   *
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $stream_wrapper_manager
   *   The stream wrapper manager.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Symfony\Component\Mime\MimeTypeGuesserInterface $mime_type_guesser
   *   The MIME type guesser.
   * @param \Drupal\file\Validation\FileValidatorInterface $file_validator
   *   The file validator.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\file\Upload\FileUploadHandler $core_upload_handler
   *   Core's file upload handler.
   * @param \Drupal\file\FileUsage\FileUsageInterface $file_usage
   *   The file usage service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend for tracking completed transfers.
   */
  public function __construct(
    FileSystemInterface $file_system,
    StreamWrapperManagerInterface $stream_wrapper_manager,
    TimeInterface $time,
    AccountProxyInterface $current_user,
    EventDispatcherInterface $event_dispatcher,
    MimeTypeGuesserInterface $mime_type_guesser,
    FileValidatorInterface $file_validator,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    FileUploadHandler $core_upload_handler,
    FileUsageInterface $file_usage,
    CacheBackendInterface $cache,
  ) {
    $this->fileSystem = $file_system;
    $this->streamWrapperManager = $stream_wrapper_manager;
    $this->time = $time;
    $this->currentUser = $current_user;
    $this->eventDispatcher = $event_dispatcher;
    $this->mimeTypeGuesser = $mime_type_guesser;
    $this->fileValidator = $file_validator;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->coreUploadHandler = $core_upload_handler;
    $this->fileUsage = $file_usage;
    $this->cache = $cache;
  }

  /**
   * Initializes a chunked upload transfer.
   *
   * @param int $size
   *   The total file size in bytes.
   * @param string $filename
   *   The original filename.
   * @param array $options
   *   Additional options:
   *   - temp_location: Custom temp directory URI.
   *   - allowed_extensions: Array of allowed extensions.
   *   - max_size: Maximum file size in bytes.
   *   - context: Context data to store with transfer (unused).
   *
   * @return array{success: bool, transfer_id?: string, error?: string, code?:
   *   int} Result array with success status and transfer_id or error.
   */
  public function initializeChunkedUpload(int $size, string $filename, array $options = []): array {
    $tempLocation = $options['temp_location'] ?? self::DEFAULT_TEMP_LOCATION;
    $allowedExtensions = $options['allowed_extensions'] ?? self::DEFAULT_ALLOWED_EXTENSIONS;
    $maxSize = $options['max_size'] ?? 0;

    // Validate file size.
    if ($maxSize > 0 && $size > $maxSize) {
      return ['success' => FALSE, 'error' => 'File too large', 'code' => 413];
    }

    // Validate file extension.
    if (!$this->validateFileExtension($filename, $allowedExtensions)) {
      return [
        'success' => FALSE,
        'error' => 'Invalid file type',
        'code' => 415,
      ];
    }

    // Prepare temp directory.
    $this->prepareDirectory($tempLocation);

    // Generate transfer ID.
    $transferId = $this->generateTransferId();

    // We derive state from the temp file itself:
    // - Offset = current file size
    // - Filename/size come from FilePond headers on each PATCH request.
    return ['success' => TRUE, 'transfer_id' => $transferId];
  }

  /**
   * Processes a chunk of an upload.
   *
   * @param string $transferId
   *   The transfer ID.
   * @param string $chunk
   *   The chunk data.
   * @param int $offset
   *   The upload offset for this chunk.
   * @param array $options
   *   Additional options:
   *   - temp_location: Custom temp directory URI.
   *   - destination: Final destination URI.
   *   - allowed_mime_types: Array of allowed MIME types.
   *   - upload_name: Original filename from Upload-Name header.
   *   - upload_length: Total file size from Upload-Length header.
   *
   * @return array{success: bool, complete?: bool, file?: FileInterface,
   *   offset?: int, error?: string, code?: int,
   *   event_results?: array<string, mixed>}
   *   Result array with success status and either new offset or file entity.
   */
  public function processChunk(string $transferId, string $chunk, int $offset, array $options = []): array {
    $tempLocation = $options['temp_location'] ?? self::DEFAULT_TEMP_LOCATION;

    // Validate transfer ID format.
    if (!$this->validateTransferId($transferId)) {
      return [
        'success' => FALSE,
        'error' => 'Invalid transfer ID',
        'code' => 400,
      ];
    }

    // Check if this transfer is already complete. This handles a race condition
    // observed in headless Chrome (Drupal CI) where FilePond sends duplicate
    // PATCH requests after the upload completes. Without this check:
    // 1. Final chunk completes → 200 + file ID returned → temp file deleted
    // 2. Duplicate PATCH arrives → temp file gone → offset mismatch → 400 error
    // 3. FilePond retries → more 400s → upload marked as PROCESSING_ERROR
    // By caching completed transfers, we return success with the file ID for
    // any late requests, making the endpoint idempotent.
    $cached = $this->cache->get('filepond_transfer:' . $transferId);
    if ($cached) {
      $fileStorage = $this->entityTypeManager->getStorage('file');
      /** @var \Drupal\file\FileInterface|null $file */
      $file = $fileStorage->load($cached->data);
      if ($file) {
        return [
          'success' => TRUE,
          'complete' => TRUE,
          'file' => $file,
        ];
      }
    }

    // FilePond sends Upload-Name and Upload-Length headers with each PATCH.
    $tempPath = $tempLocation . '/' . $transferId;
    $filename = $options['upload_name'] ?? 'unknown.jpg';
    $totalSize = (int) ($options['upload_length'] ?? 0);

    // Get current offset from actual file size (like dropzonejs does).
    $currentOffset = file_exists($tempPath) ? (int) @filesize($tempPath) : 0;

    // Validate that client's claimed offset matches actual file size.
    // This prevents chunk skipping or replay attacks.
    if ($offset !== $currentOffset) {
      return [
        'success' => FALSE,
        'error' => "Offset mismatch: expected $currentOffset, got $offset",
        'code' => 400,
      ];
    }

    // Append chunk to file.
    $mode = $offset === 0 ? 'wb' : 'ab';
    $handle = fopen($tempPath, $mode);
    if (!$handle) {
      return [
        'success' => FALSE,
        'error' => 'Failed to write chunk',
        'code' => 500,
      ];
    }
    fwrite($handle, $chunk);
    fclose($handle);

    // Calculate new offset from actual file size.
    $newOffset = (int) @filesize($tempPath);

    // Check if upload is complete.
    if ($totalSize > 0 && $newOffset >= $totalSize) {
      return $this->finalizeUpload($transferId, $filename, $options);
    }

    return ['success' => TRUE, 'complete' => FALSE, 'offset' => $newOffset];
  }

  /**
   * Handles a non-chunked file upload.
   *
   * @param \Symfony\Component\HttpFoundation\File\UploadedFile $uploadedFile
   *   The uploaded file.
   * @param array $options
   *   Additional options:
   *   - temp_location: Custom temp directory URI.
   *   - destination: Final destination URI.
   *   - allowed_extensions: Array of allowed extensions.
   *   - allowed_mime_types: Array of allowed MIME types.
   *   - max_size: Maximum file size in bytes.
   *   - context: Context data to pass to event.
   *
   * @return array{success: bool, file?: FileInterface, error?: string, code?:
   *   int} Result array with success status and file entity or error.
   */
  public function handleUpload(UploadedFile $uploadedFile, array $options = []): array {
    $tempLocation = $options['temp_location'] ?? self::DEFAULT_TEMP_LOCATION;
    $allowedExtensions = $options['allowed_extensions'] ?? self::DEFAULT_ALLOWED_EXTENSIONS;
    $allowedMimeTypes = $options['allowed_mime_types'] ?? self::DEFAULT_ALLOWED_MIME_TYPES;
    $maxSize = $options['max_size'] ?? 0;

    $filename = $uploadedFile->getClientOriginalName();

    // Validate file extension.
    if (!$this->validateFileExtension($filename, $allowedExtensions)) {
      return [
        'success' => FALSE,
        'error' => 'Invalid file type',
        'code' => 415,
      ];
    }

    // Validate file size.
    if ($maxSize > 0 && $uploadedFile->getSize() > $maxSize) {
      return ['success' => FALSE, 'error' => 'File too large', 'code' => 413];
    }

    // Prepare temp directory.
    $this->prepareDirectory($tempLocation);

    $transferId = $this->generateTransferId();
    $tempPath = $tempLocation . '/' . $transferId;

    // Move uploaded file to temp location.
    $uploadedFile->move($this->fileSystem->realpath($tempLocation), $transferId);
    $realTempPath = $this->fileSystem->realpath($tempPath);

    // Validate MIME type.
    $mimeType = $this->detectMimeType($realTempPath, $filename);

    if (!in_array($mimeType, $allowedMimeTypes, TRUE)) {
      $this->fileSystem->unlink($tempPath);
      return [
        'success' => FALSE,
        'error' => 'Invalid file type: ' . $mimeType,
        'code' => 415,
      ];
    }

    // Validate image content if it's an image.
    if (str_starts_with($mimeType, 'image/')) {
      $imageValidation = $this->validateImageContent($realTempPath, $allowedMimeTypes);
      if ($imageValidation !== TRUE) {
        $this->fileSystem->unlink($tempPath);
        return [
          'success' => FALSE,
          'error' => 'Invalid image: ' . $imageValidation,
          'code' => 415,
        ];
      }
    }

    // Pass detected MIME type to finalizeUpload for the pre-move event.
    $options['detected_mime_type'] = $mimeType;

    return $this->finalizeUpload($transferId, $filename, $options);
  }

  /**
   * Finalizes an upload and creates the file entity.
   *
   * @param string $transferId
   *   The transfer ID.
   * @param string $originalFilename
   *   The original filename.
   * @param array $options
   *   Additional options:
   *   - temp_location: Custom temp directory URI.
   *   - destination: Final destination URI.
   *   - allowed_mime_types: Array of allowed MIME types.
   *   - context: Context data to pass to event.
   *
   * @return array{success: bool, complete?: bool, file?: FileInterface,
   *   error?: string, code?: int} Result array with file entity or error.
   */
  public function finalizeUpload(string $transferId, string $originalFilename, array $options = []): array {
    $tempLocation = $options['temp_location'] ?? self::DEFAULT_TEMP_LOCATION;
    $allowedMimeTypes = $options['allowed_mime_types'] ?? self::DEFAULT_ALLOWED_MIME_TYPES;

    $tempPath = $tempLocation . '/' . $transferId;
    $realTempPath = $this->fileSystem->realpath($tempPath);

    if (!$realTempPath || !file_exists($realTempPath)) {
      return ['success' => FALSE, 'error' => 'File not found', 'code' => 404];
    }

    // Get MIME type - prefer passed value from handleUpload(), fallback to
    // detection.
    $mimeType = $options['detected_mime_type']
      ?? $this->detectMimeType($realTempPath, $originalFilename);

    // Validate image content if uploading images (checks actual file bytes).
    // Only run this check if allowed MIME types include images.
    $hasImageMimeTypes = array_filter(
      $allowedMimeTypes,
      fn($mime) => str_starts_with($mime, 'image/')
    );
    if (!empty($hasImageMimeTypes)) {
      $imageValidation = $this->validateImageContent($realTempPath, $allowedMimeTypes);
      if ($imageValidation !== TRUE) {
        $this->fileSystem->unlink($tempPath);
        return [
          'success' => FALSE,
          'error' => 'Invalid image: ' . $imageValidation,
          'code' => 415,
        ];
      }
    }

    // Determine destination.
    $destination = $options['destination'] ?? $this->getDefaultDestination();
    $this->prepareDirectory($destination);

    // Build validators array from options.
    $validators = [];
    if (!empty($options['allowed_extensions'])) {
      $extensions = implode(' ', $options['allowed_extensions']);
      $validators['FileExtension'] = ['extensions' => $extensions];
    }
    if (!empty($options['max_size'])) {
      $validators['FileSizeLimit'] = ['fileLimit' => $options['max_size']];
    }

    // Dispatch pre-move event - allows subscribers to modify the local file
    // before it's moved to its final destination (e.g., S3).
    // This is the ideal point for EXIF rotation, virus scanning, optimization,
    // and filename obfuscation.
    $context = $options['context'] ?? [];
    $isRemoteDestination = $this->isRemoteUri($destination);
    $preMoveEvent = new FilePondUploadPreMoveEvent(
      $realTempPath,
      $mimeType,
      $originalFilename,
      $destination,
      $context,
      $isRemoteDestination
    );
    $this->eventDispatcher->dispatch($preMoveEvent, FilePondUploadPreMoveEvent::EVENT_NAME);

    // Use overridden filename if set by subscriber, otherwise original.
    $filenameForUpload = $preMoveEvent->getFilename();

    // Create TempUploadedFile from our temp file.
    // We use our own class (not FormUploadedFile) because core's handler
    // checks instanceof FormUploadedFile and uses moveUploadedFile() which
    // fails for files not in $_FILES. TempUploadedFile uses move() instead.
    $uploadedFile = new TempUploadedFile($realTempPath, $filenameForUpload);

    // Capture image dimensions while file is still local (after any rotation).
    // This prevents expensive S3 downloads later when ImageItem::preSave()
    // runs.
    $imageDimensions = NULL;
    if (str_starts_with($mimeType, 'image/')) {
      $imageInfo = @getimagesize($realTempPath);
      if ($imageInfo !== FALSE) {
        $imageDimensions = [
          'width' => $imageInfo[0],
          'height' => $imageInfo[1],
        ];
      }
    }

    // Delegate to core's handler for proper file handling:
    // - Filename sanitization (security event)
    // - File locking (prevents concurrent writes)
    // - File validation (extension, size, filename length)
    // - File entity creation and validation
    // - chmod permissions.
    try {
      $result = $this->coreUploadHandler->handleFileUpload(
        $uploadedFile,
        $validators,
        $destination,
        FileExists::Rename,
        FALSE
      );
    }
    catch (\Exception $e) {
      // Clean up temp file on failure.
      if (file_exists($realTempPath)) {
        $this->fileSystem->unlink($tempPath);
      }
      return [
        'success' => FALSE,
        'error' => $e->getMessage(),
        'code' => 500,
      ];
    }

    // Check for validation violations.
    $violations = $result->getViolations();
    if (count($violations) > 0) {
      // Clean up temp file on validation failure.
      if (file_exists($realTempPath)) {
        $this->fileSystem->unlink($tempPath);
      }
      $errors = [];
      foreach ($violations as $violation) {
        $errors[] = $violation->getMessage();
      }
      return [
        'success' => FALSE,
        'error' => implode(', ', $errors),
        'code' => 422,
      ];
    }

    $file = $result->getFile();

    // Store image dimensions for later use by widgets.
    // This prevents expensive S3 downloads in ImageItem::preSave().
    if ($imageDimensions !== NULL) {
      ImageDimensionHelper::store(
        $file->id(),
        $imageDimensions['width'],
        $imageDimensions['height']
      );
    }

    // Dispatch event for implementing modules.
    $context = $options['context'] ?? [];
    $event = new FilePondUploadCompleteEvent($file, $transferId, $originalFilename, $context);
    $this->eventDispatcher->dispatch($event, FilePondUploadCompleteEvent::EVENT_NAME);

    // Cache the completed transfer so late/duplicate PATCH requests return
    // success. See processChunk() for full explanation of the race condition.
    $this->cache->set(
      'filepond_transfer:' . $transferId,
      $file->id(),
      $this->time->getRequestTime() + 3600
    );

    return [
      'success' => TRUE,
      'complete' => TRUE,
      'file' => $file,
      'event_results' => $event->getResults(),
    ];
  }

  /**
   * Reverts/cancels an upload.
   *
   * @param string $serverId
   *   Either a transfer ID (hex string) or file ID (numeric).
   * @param array $options
   *   Additional options:
   *   - temp_location: Custom temp directory URI.
   *
   * @return array{success: bool, error?: string, code?: int}
   *   Result array with success status or error.
   */
  public function revert(string $serverId, array $options = []): array {
    $tempLocation = $options['temp_location'] ?? self::DEFAULT_TEMP_LOCATION;

    // Check if this is a file ID (numeric) or transfer ID (hex string).
    if (is_numeric($serverId)) {
      $fileStorage = $this->entityTypeManager->getStorage('file');
      /** @var \Drupal\file\FileInterface|null $file */
      $file = $fileStorage->load($serverId);
      if (!$file) {
        return ['success' => FALSE, 'error' => 'File not found', 'code' => 404];
      }
      // Access is verified via signed token in the controller - if we get here,
      // this is a file ID that was issued by our upload endpoint.
      // Only delete temporary files with no usages. Permanent files are left
      // for the file usage system to clean up - this matches core's
      // managed_file behavior.
      // @see file_managed_file_submit()
      if ($file->isTemporary()) {
        $usage = $this->fileUsage->listUsage($file);
        if (empty($usage)) {
          $file->delete();
          return ['success' => TRUE, 'action' => 'deleted'];
        }
        return ['success' => TRUE, 'action' => 'skipped_has_usage'];
      }
      return ['success' => TRUE, 'action' => 'skipped_permanent'];
    }

    // Validate transfer ID format.
    if (!$this->validateTransferId($serverId)) {
      return [
        'success' => FALSE,
        'error' => 'Invalid transfer ID',
        'code' => 400,
      ];
    }

    // Clean up temp file.
    $tempPath = $tempLocation . '/' . $serverId;
    $realTempPath = $this->fileSystem->realpath($tempPath) ?: '';
    if ($realTempPath && file_exists($realTempPath)) {
      $this->fileSystem->unlink($tempPath);
      return ['success' => TRUE, 'action' => 'temp_deleted'];
    }

    return ['success' => TRUE, 'action' => 'temp_not_found'];
  }

  /**
   * Generates a unique transfer ID.
   *
   * @return string
   *   A 32-character hex string.
   */
  public function generateTransferId(): string {
    return bin2hex(random_bytes(16));
  }

  /**
   * Validates a transfer ID format.
   *
   * @param string $transferId
   *   The transfer ID to validate.
   *
   * @return bool
   *   TRUE if valid.
   */
  public function validateTransferId(string $transferId): bool {
    return (bool) preg_match('/^[a-f0-9]{32}$/', $transferId);
  }

  /**
   * Gets the current upload offset for a transfer.
   *
   * Used by HEAD requests to determine resume position for chunked uploads.
   *
   * @param string $transferId
   *   The transfer ID.
   * @param array $options
   *   Additional options:
   *   - temp_location: Custom temp directory URI.
   *
   * @return int
   *   The current offset in bytes (0 if no file exists).
   */
  public function getUploadOffset(string $transferId, array $options = []): int {
    if (!$this->validateTransferId($transferId)) {
      return 0;
    }

    $tempLocation = $options['temp_location'] ?? self::DEFAULT_TEMP_LOCATION;
    $tempPath = $tempLocation . '/' . $transferId;

    return file_exists($tempPath) ? (int) @filesize($tempPath) : 0;
  }

  /**
   * Validates a file extension.
   *
   * @param string $filename
   *   The filename to check.
   * @param array $allowedExtensions
   *   Array of allowed extensions (lowercase).
   *
   * @return bool
   *   TRUE if allowed.
   */
  public function validateFileExtension(string $filename, array $allowedExtensions): bool {
    $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    return in_array($extension, $allowedExtensions, TRUE);
  }

  /**
   * Validates that a file is a valid image.
   *
   * @param string $filePath
   *   The path to the file.
   * @param array $allowedMimeTypes
   *   Array of allowed MIME types.
   *
   * @return bool|string
   *   TRUE if valid, or error message string.
   */
  public function validateImageContent(string $filePath, array $allowedMimeTypes): bool|string {
    if (!file_exists($filePath)) {
      return 'file not found';
    }

    $imageInfo = @getimagesize($filePath);
    if ($imageInfo === FALSE) {
      return 'not a valid image';
    }

    $detectedMime = $imageInfo['mime'] ?? '';
    if (!in_array($detectedMime, $allowedMimeTypes, TRUE)) {
      return 'mime type ' . $detectedMime . ' not allowed';
    }

    return TRUE;
  }

  /**
   * Prepares a directory for writing.
   *
   * @param string $directory
   *   The directory URI.
   */
  public function prepareDirectory(string $directory): void {
    $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
  }

  /**
   * Gets the default destination URI.
   *
   * @return string
   *   The destination URI.
   */
  protected function getDefaultDestination(): string {
    $scheme = $this->configFactory->get('system.file')->get('default_scheme');
    return $scheme . '://filepond-uploads';
  }

  /**
   * Detects the MIME type of a file.
   *
   * Uses multiple fallback strategies:
   * 1. finfo (file content analysis)
   * 2. getimagesize() for images
   * 3. Extension-based guessing for non-images (e.g., PDFs)
   *
   * @param string $filePath
   *   The real path to the file.
   * @param string|null $filename
   *   The original filename (for extension-based fallback).
   *
   * @return string
   *   The detected MIME type.
   */
  protected function detectMimeType(string $filePath, ?string $filename = NULL): string {
    // Primary detection via finfo.
    $finfo = new \finfo(FILEINFO_MIME_TYPE);
    $mimeType = $finfo->file($filePath) ?: 'application/octet-stream';

    // If generic octet-stream, try fallbacks.
    if ($mimeType === 'application/octet-stream') {
      // Try image detection first.
      $imageInfo = @getimagesize($filePath);
      if ($imageInfo !== FALSE && !empty($imageInfo['mime'])) {
        return $imageInfo['mime'];
      }
      // Fallback to extension-based guessing for non-images (e.g., PDFs).
      if (!empty($filename)) {
        $guessedMime = $this->mimeTypeGuesser->guessMimeType($filename);
        if ($guessedMime && $guessedMime !== 'application/octet-stream') {
          return $guessedMime;
        }
      }
    }

    return $mimeType;
  }

  /**
   * Checks if a URI points to remote storage.
   *
   * Uses Drupal's stream wrapper system to determine if the URI
   * is a local filesystem or remote (e.g., S3). This correctly handles
   * cases where public:// is mapped to remote storage via s3fs.
   *
   * @param string $uri
   *   The URI to check (e.g., 'public://images/photo.jpg').
   *
   * @return bool
   *   TRUE if destination is not a local filesystem.
   */
  protected function isRemoteUri(string $uri): bool {
    $scheme = $this->streamWrapperManager->getScheme($uri);
    if (!$scheme) {
      return FALSE;
    }

    $wrapper = $this->streamWrapperManager->getViaScheme($scheme);
    if ($wrapper === FALSE) {
      return FALSE;
    }

    return !$wrapper instanceof LocalStream;
  }

}
